/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import javax.media.opengl.GL;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IAnimatableLayerReference;
import ch.kuramo.javie.api.IAnimatableValue;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IEffectContext;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.services.AudioRenderSupport;
import ch.kuramo.javie.core.services.RenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

public abstract class AbstractEffectContext implements IEffectContext {

	private final RenderContext _context;

	private final VideoRenderSupport _vrSupport;

	private final AudioRenderSupport _arSupport;


	public AbstractEffectContext(RenderContext context, VideoRenderSupport vrSupport, AudioRenderSupport arSupport) {
		super();
		_context = context;
		_vrSupport = vrSupport;
		_arSupport = arSupport;
	}


	public Time getTime() {
		return _context.getTime();
	}

	public void setTime(Time time) {
		_context.setTime(time);
	}

	public Resolution getVideoResolution() {
		return _context.getVideoResolution();
	}

	public ColorMode getColorMode() {
		return _context.getColorMode();
	}

	public AudioMode getAudioMode() {
		return _context.getAudioMode();
	}

	public int getAudioFrameCount() {
		return _context.getAudioFrameCount();
	}

	public void setAudioFrameCount(int count) {
		_context.setAudioFrameCount(count);
	}

	public GL getGL() {
		return _context.getGL();
	}

	public GLU getGLU() {
		return _context.getGLU();
	}

	public IVideoBuffer createVideoBuffer(VideoBounds bounds) {
		return _vrSupport.createVideoBuffer(_context.getColorMode(), bounds);
	}


	public IAudioBuffer createAudioBuffer() {
		return _arSupport.createAudioBuffer();
	}

	public <V> V value(IAnimatableValue<V> avalue) {
		@SuppressWarnings("unchecked")
		AnimatableValue<V> av = (AnimatableValue<V>) avalue;
		return av.value(_context);
	}

	public IVideoBuffer getLayerVideoFrame(IAnimatableLayerReference layerRef) {
		Composition comp = _context.getComposition();
		if (!(comp instanceof LayerComposition)) {
			throw new IllegalStateException("Composition in current context is not LayerComposition");
		}

		final LayerComposition layerComp = (LayerComposition) comp;
		final Layer layer = layerComp.getLayer(value(layerRef));

		if (layer == null) {
			return null;
		}

		return _context.saveAndExecute(new WrappedOperation<IVideoBuffer>() {
			public IVideoBuffer execute() {
				return layerComp.getLayerVideoFrame(layer, false, false);
			}
		});
	}

	public IAudioBuffer getLayerAudioChunk(IAnimatableLayerReference layerRef) {
		Composition comp = _context.getComposition();
		if (!(comp instanceof LayerComposition)) {
			throw new IllegalStateException("Composition in current context is not LayerComposition");
		}

		final LayerComposition layerComp = (LayerComposition) comp;
		final Layer layer = layerComp.getLayer(value(layerRef));

		if (layer == null) {
			return null;
		}

		return _context.saveAndExecute(new WrappedOperation<IAudioBuffer>() {
			public IAudioBuffer execute() {
				return layerComp.getLayerAudioChunk(layer, false, false);
			}
		});
	}

}
