/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.Arrays;
import java.util.Set;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.IVideoBuffer.TextureWrapMode;
import ch.kuramo.javie.api.services.IBlurSupport;
import ch.kuramo.javie.api.services.IConvolutionSupport;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.api.services.IConvolutionSupport.ConvolutionDirection;
import ch.kuramo.javie.core.Util;

import com.google.inject.Inject;

public class BlurSupportImpl implements IBlurSupport {

	private final IVideoRenderSupport support;

	private final IConvolutionSupport convolution;

	@Inject
	public BlurSupportImpl(IVideoRenderSupport support, IConvolutionSupport convolution) {
		this.support = support;
		this.convolution = convolution;
	}

	private static final int[][] DOWNSAMPLE_RATIOS = {
		{ 300, 7, 7 },
		{ 200, 5, 7 },
		{ 150, 5, 5 },
		{ 100, 3, 5 },
		{  50, 3, 3 },
		{  30, 5 },
		{  10, 3 }
	};

	private int[] getDownSampleRatios(double hint) {
		for (int i = 0; i < DOWNSAMPLE_RATIOS.length; ++i) {
			if (hint >= DOWNSAMPLE_RATIOS[i][0]) {
				int[] ratios = new int[DOWNSAMPLE_RATIOS[i].length - 1];
				for (int j = 0; j < ratios.length; ++j) {
					ratios[j] = DOWNSAMPLE_RATIOS[i][j+1];
				}
				return ratios;
			}
		}
		return new int[0];
	}

	private void doDownSample(
			IVideoBuffer input, IVideoBuffer output, int hRatio, int vRatio) {

		int ksize = hRatio * vRatio;
		float[] kernel = new float[ksize];
		float[] offset = new float[ksize*2];

		for (int j = 0; j < vRatio; ++j) {
			for (int i = 0; i < hRatio; ++i) {
				int k = j*hRatio+i;
				kernel[k] = 1f/ksize;
				offset[k*2  ] = i - hRatio/2;
				offset[k*2+1] = j - vRatio/2;
			}
		}

		final VideoBounds bounds = output.getBounds();
		int w = bounds.width*hRatio;
		int h = bounds.height*vRatio;

		final double[][] texCoords = new double[][] { {0, 0}, {w, 0}, {w, h}, {0, h} };

		Runnable operation = new Runnable() {
			public void run() {
				support.ortho2D(bounds);
				support.quad2D(bounds, texCoords);
			}
		};

		convolution.convolve(input, output, kernel, offset, operation, 0);
	}

	private void doUpSample(
			IVideoBuffer input, IVideoBuffer output, int hRatio, int vRatio) {

		final VideoBounds bounds = output.getBounds();
		double w = (double)bounds.width/hRatio;
		double h = (double)bounds.height/vRatio;

		final double[][] texCoords = new double[][] { {0, 0}, {w, 0}, {w, h}, {0, h} };

		Runnable operation = new Runnable() {
			public void run() {
				support.ortho2D(bounds);
				support.quad2D(bounds, texCoords);
			}
		};

		TextureFilter currFilter = input.getTextureFilter();
		try {
			input.setTextureFilter(TextureFilter.LINEAR);
			support.useFramebuffer(operation, 0, output, input);
		} finally {
			input.setTextureFilter(currFilter);
		}
	}

	private VideoBounds expandBounds(VideoBounds bounds, int hExpand, int vExpand) {
		int dx = -hExpand;
		int dw = hExpand*2;

		int dy = -vExpand;
		int dh = vExpand*2;

		return new VideoBounds(
				bounds.x + dx, bounds.y + dy,
				bounds.width + dw, bounds.height + dh);
	}

	private VideoBounds calcBlurredBounds(
			VideoBounds bounds, double radius, BlurDimensions dimensions,
			boolean fast, boolean boxBlur) {

		if (radius <= 0) {
			throw new IllegalArgumentException("radius must be greater than 0");
		}

		boolean horz = (dimensions != BlurDimensions.VERTICAL);
		boolean vert = (dimensions != BlurDimensions.HORIZONTAL);

		int totalRatio = 1;
		if (fast) {
			for (int ratio : getDownSampleRatios(radius)) {
				int hRatio = horz ? ratio : 1;
				int vRatio = vert ? ratio : 1;
				bounds = new VideoBounds(bounds.x/hRatio, bounds.y/vRatio,
						(bounds.width+hRatio-1)/hRatio, (bounds.height+vRatio-1)/vRatio);
				totalRatio *= ratio;
			}
			radius /= totalRatio;

			if (boxBlur && totalRatio != 1) {
				radius -= 1.0/3.0;
			}
		}

		int ceiledRadius = (int) Math.ceil(radius);

		int hExpand = horz ? ceiledRadius : 0;
		int vExpand = vert ? ceiledRadius : 0;
		bounds = expandBounds(bounds, hExpand, vExpand);

		int hRatio = horz ? totalRatio : 1;
		int vRatio = vert ? totalRatio : 1;
		return new VideoBounds(bounds.x*hRatio, bounds.y*vRatio,
				bounds.width*hRatio, bounds.height*vRatio);
	}

	private float[] createBlurKernel(double radius, boolean boxBlur) {
		int ceiledRadius = (int) Math.ceil(radius);
		float[] kernel = new float[ceiledRadius*2+1];

		if (boxBlur) {
			Arrays.fill(kernel, (float)(1/(radius*2+1)));
			if (radius != ceiledRadius) {
				kernel[0] *= 1-(ceiledRadius-radius);
				kernel[kernel.length-1] *= 1-(ceiledRadius-radius);
			}
		} else {
			// 標準偏差の2.5倍の位置がぼかしの端となるようにする（これでだいたいAEと同じになる）
			double sigma = radius / 2.5;
			double sigmaSquare = sigma * sigma;

			float sum = 0;
			for (int i = 1; i <= ceiledRadius; ++i) {
				sum += 2 * (kernel[ceiledRadius-i] = (float) Math.exp(-i * i / (2 * sigmaSquare)));
			}
			kernel[ceiledRadius] = 1 / (++sum);
			for (int i = 1; i <= ceiledRadius; ++i) {
				kernel[ceiledRadius+i] = (kernel[ceiledRadius-i] /= sum);
			}
		}

		return kernel;
	}

	private IVideoBuffer blur(
			IVideoBuffer input, ColorMode colorMode, double radius, BlurDimensions dimensions,
			boolean repeatEdgePixels, boolean fast, boolean boxBlur) {

		if (radius <= 0) {
			throw new IllegalArgumentException("radius must be greater than 0");
		}

		Set<IVideoBuffer> tmpBuffers = Util.newSet();
		try {

			boolean horz = (dimensions != BlurDimensions.VERTICAL);
			boolean vert = (dimensions != BlurDimensions.HORIZONTAL);

			IVideoBuffer buffer = input;
			VideoBounds inputBounds = input.getBounds();

			int totalRatio = 1;
			if (fast) {
				VideoBounds bounds = inputBounds;
				for (int ratio : getDownSampleRatios(radius)) {
					if (repeatEdgePixels) {
						buffer.setTextureWrapMode(TextureWrapMode.CLAMP_TO_EDGE);
					}

					int hRatio = horz ? ratio : 1;
					int vRatio = vert ? ratio : 1;
					bounds = new VideoBounds(bounds.x/hRatio, bounds.y/vRatio,
							(bounds.width+hRatio-1)/hRatio, (bounds.height+vRatio-1)/vRatio);

					IVideoBuffer tmp = support.createVideoBuffer(bounds, colorMode);
					tmpBuffers.add(tmp);

					doDownSample(buffer, tmp, hRatio, vRatio);
					buffer = tmp;

					totalRatio *= ratio;
				}
				radius /= totalRatio;

				if (boxBlur && totalRatio != 1) {
					radius -= 1.0/3.0;  // 理由がわからないが、1/3引いた方がダウンサンプルしない場合の結果に近くなる。
				}
			}

			int ceiledRadius = (int) Math.ceil(radius);
			float[] kernel = createBlurKernel(radius, boxBlur);

			if (horz) {
				VideoBounds bounds = buffer.getBounds();

				if (repeatEdgePixels) {
					buffer.setTextureWrapMode(TextureWrapMode.CLAMP_TO_EDGE);
				} else {
					bounds = expandBounds(bounds, ceiledRadius, 0);
				}

				IVideoBuffer tmp = support.createVideoBuffer(bounds, colorMode);
				tmpBuffers.add(tmp);

				convolution.convolve1D(buffer, tmp, kernel, ConvolutionDirection.HORIZONTAL);
				buffer = tmp;
			}

			if (vert) {
				VideoBounds bounds = buffer.getBounds();

				if (repeatEdgePixels) {
					buffer.setTextureWrapMode(TextureWrapMode.CLAMP_TO_EDGE);
				} else {
					bounds = expandBounds(bounds, 0, ceiledRadius);
				}

				IVideoBuffer tmp = support.createVideoBuffer(bounds, colorMode);
				tmpBuffers.add(tmp);

				convolution.convolve1D(buffer, tmp, kernel, ConvolutionDirection.VERTICAL);
				buffer = tmp;
			}

			if (totalRatio != 1) {
				int hRatio = horz ? totalRatio : 1;
				int vRatio = vert ? totalRatio : 1;
				VideoBounds bounds;

				if (repeatEdgePixels) {
					buffer.setTextureWrapMode(TextureWrapMode.CLAMP_TO_EDGE);
					bounds = inputBounds;
				} else {
					bounds = buffer.getBounds();
					bounds = new VideoBounds(bounds.x*hRatio, bounds.y*vRatio, bounds.width*hRatio, bounds.height*vRatio);
				}

				IVideoBuffer tmp = support.createVideoBuffer(bounds, colorMode);
				tmpBuffers.add(tmp);

				doUpSample(buffer, tmp, hRatio, vRatio);
				buffer = tmp;
			}

			tmpBuffers.remove(buffer);
			return buffer;

		} finally {
			for (IVideoBuffer vb : tmpBuffers) {
				vb.dispose();
			}
		}
	}

	public VideoBounds calcGaussianBlurredBounds(
			VideoBounds inputBounds, double radius, BlurDimensions dimensions, boolean fast) {

		return calcBlurredBounds(inputBounds, radius, dimensions, fast, false);
	}

	public float[] createGaussianBlurKernel(double radius) {
		return createBlurKernel(radius, false);
	}

	public IVideoBuffer gaussianBlur(
			IVideoBuffer input, double radius, BlurDimensions dimensions,
			boolean repeatEdgePixels, boolean fast) {

		return gaussianBlur(input, input.getColorMode(), radius, dimensions, repeatEdgePixels, fast);
	}

	public IVideoBuffer gaussianBlur(
			IVideoBuffer input, ColorMode colorMode, double radius, BlurDimensions dimensions,
			boolean repeatEdgePixels, boolean fast) {

		// repeatEdgePixels が true の場合、
		// blur メソッド内で input の WrapMode が変更されるので、あとで元に戻す。
		TextureWrapMode wrapMode = repeatEdgePixels ? input.getTextureWrapMode() : null;
		try {
			return blur(input, colorMode, radius, dimensions, repeatEdgePixels, fast, false);
		} finally {
			if (wrapMode != null) input.setTextureWrapMode(wrapMode);
		}
	}

	public VideoBounds calcBoxBlurredBounds(
			VideoBounds inputBounds, double radius, BlurDimensions dimensions,
			boolean fast, int boxIterations) {

		if (boxIterations < 1) {
			throw new IllegalArgumentException("boxIterations must be 1 or greater");
		}

		VideoBounds bounds = inputBounds;

		for (int i = 0; i < boxIterations; ++i) {
			bounds = calcBlurredBounds(bounds, radius, dimensions, fast, true);
		}

		return bounds;
	}

	public float[] createBoxBlurKernel(double radius) {
		return createBlurKernel(radius, true);
	}

	public IVideoBuffer boxBlur(
			IVideoBuffer input, double radius, BlurDimensions dimensions,
			boolean repeatEdgePixels, boolean fast, int boxIterations) {

		return boxBlur(input, input.getColorMode(), radius, dimensions, repeatEdgePixels, fast, boxIterations);
	}

	public IVideoBuffer boxBlur(
			IVideoBuffer input, ColorMode colorMode, double radius, BlurDimensions dimensions,
			boolean repeatEdgePixels, boolean fast, int boxIterations) {

		if (boxIterations < 1) {
			throw new IllegalArgumentException("boxIterations must be 1 or greater");
		}

		// repeatEdgePixels が true の場合、
		// blur メソッド内で input の WrapMode が変更されるので、あとで元に戻す。
		TextureWrapMode wrapMode = repeatEdgePixels ? input.getTextureWrapMode() : null;
		try {

			IVideoBuffer buffer = blur(input, colorMode, radius, dimensions, repeatEdgePixels, fast, true);

			for (int i = 1; i < boxIterations; ++i) {
				IVideoBuffer old = buffer;
				try {
					buffer = blur(buffer, colorMode, radius, dimensions, repeatEdgePixels, fast, true);
				} finally {
					old.dispose();
				}
			}

			return buffer;

		} finally {
			if (wrapMode != null) input.setTextureWrapMode(wrapMode);
		}
	}

}
