/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.io.File;
import java.io.IOException;
import java.net.URI;

import net.arnx.jsonic.JSON;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.services.ProjectEncoder;

public class ProjectEncoderImpl implements ProjectEncoder {

	public String encode(Project p, File baseDir) throws IOException {
		return encode(p, baseDir, false);
	}

	public String encode(Project p, File baseDir, boolean prettyPrint) throws IOException {
		for (File root : File.listRoots()) {
			if (baseDir.equals(root)) {
				baseDir = null;
				break;
			}
		}

		EncoderJSON json = (baseDir != null) ? new EncoderJSON(baseDir) : new EncoderJSON();
		json.setPrettyPrint(prettyPrint);
		return json.format(p);
	}

	public String encodeElement(Object element) {
		EncoderJSON json = new EncoderJSON();
		return json.format(element);
	}

	private static class EncoderJSON extends JSON {

		private final URI _baseURI;

		private EncoderJSON(File baseDir) throws IOException {
			super();
			_baseURI = baseDir.getCanonicalFile().toURI();
		}

		private EncoderJSON() {
			super();
			_baseURI = null;
		}

		@Override
		protected Object preformat(Context context, Object value) throws Exception {
			if (value instanceof File) {
				return relativize((File) value);

			} else if (value instanceof Color) {
				return toArray((Color) value);

			} else if (value instanceof Size2i) {
				return toArray((Size2i) value);

			} else if (value instanceof VideoBounds) {
				return toArray((VideoBounds) value);

			} else if (value instanceof Vec2d) {
				return toArray((Vec2d) value);

			} else if (value instanceof Vec3d) {
				return toArray((Vec3d) value);

			} else if (value instanceof Time) {
				return toArray((Time) value);

			} else if (value instanceof Enum<?>) {
				return ((Enum<?>) value).name();
			}

			return super.preformat(context, value);
		}

		private String relativize(File file) {
			URI uri;
			try {
				uri = file.getCanonicalFile().toURI();
			} catch (IOException e) {
				uri = file.getAbsoluteFile().toURI();
			}

			if (_baseURI == null) {
				return uri.toASCIIString();
			}

			if (uri.getPath().startsWith(_baseURI.getPath())) {
				return _baseURI.relativize(uri).toASCIIString();
			} else {
				return uri.toASCIIString();
			}
		}

		private double[] toArray(Color c) {
			return new double[] { c.r, c.g, c.b, c.a };
		}

		private int[] toArray(Size2i d) {
			return new int[] { d.width, d.height };
		}

		private Object[] toArray(VideoBounds bounds) {
			return new Object[] { bounds.x, bounds.y, bounds.width, bounds.height };
		}

		private double[] toArray(Vec2d vec2d) {
			return new double[] { vec2d.x, vec2d.y };
		}

		private double[] toArray(Vec3d vec3d) {
			return new double[] { vec3d.x, vec3d.y, vec3d.z };
		}

		private long[] toArray(Time t) {
			return new long[] { t.timeValue, t.timeScale };
		}

	}

}
