/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.output;

import java.util.Arrays;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.misc.AtiIntelLock;
import ch.kuramo.javie.core.services.AudioRenderSupport;
import ch.kuramo.javie.core.services.GLGlobal;
import ch.kuramo.javie.core.services.RenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

public abstract class PullSourceOutput extends Output {

	private static final Logger _logger = LoggerFactory.getLogger(PullSourceOutput.class);


	protected static class VideoRequest {
		protected final Time time;
		protected VideoRequest(Time time) {
			this.time = time;
		}
	}

	protected static class AudioRequest {
		protected final Time time;
		protected final int frameCount;
		protected AudioRequest(Time time, int frameCount) {
			this.time = time;
			this.frameCount = frameCount;
		}
	}

	private static final VideoRequest END_VIDEO = new VideoRequest(null);

	private static final AudioRequest END_AUDIO = new AudioRequest(null, 0);


	private ProgressMonitor _progressMonitor;

	private final BlockingQueue<VideoRequest> _videoRequestQueue = new ArrayBlockingQueue<VideoRequest>(1);

	private final BlockingQueue<Object> _videoResponseQueue = new ArrayBlockingQueue<Object>(1);

	private final BlockingQueue<AudioRequest> _audioRequestQueue = new ArrayBlockingQueue<AudioRequest>(1);

	private final BlockingQueue<Object> _audioResponseQueue = new ArrayBlockingQueue<Object>(1);

	@Inject
	private RenderContext _context;

	@Inject
	private VideoRenderSupport _vrSupport;

	@Inject
	private AudioRenderSupport _arSupport;

	@Inject
	private GLGlobal _glGlobal;


	public void doOutput(ProgressMonitor progMon) {
		if (getComposition() == null) throw new IllegalStateException("composition is not set");
		if (getFile() == null) throw new IllegalStateException("file is not set");

		boolean videoEnabled = isVideoEnabled();
		boolean audioEnabled = isAudioEnabled();

		if (!videoEnabled && !audioEnabled) {
			throw new IllegalStateException("neither video nor audio is enabled.");
		}

		_progressMonitor = (progMon != null) ? progMon : ProgressMonitor.NULL;

		if (videoEnabled) {
			Thread videoThread = new Thread(new Runnable() { public void run() { videoLoop(); } });
			videoThread.start();
		}

		if (audioEnabled) {
			Thread audioThread = new Thread(new Runnable() { public void run() { audioLoop(); } });
			audioThread.start();
		}

		doOutput();

		if (videoEnabled) {
			try {
				_videoRequestQueue.put(END_VIDEO);
			} catch (InterruptedException e) {
				_logger.warn("unexpected interruption", e);
			}
		}

		if (audioEnabled) {
			try {
				_audioRequestQueue.put(END_AUDIO);
			} catch (InterruptedException e) {
				_logger.warn("unexpected interruption", e);
			}
		}
	}

	protected abstract void doOutput();


	protected void videoLoop() {
		AtiIntelLock atiIntelLock = AtiIntelLock.get(_glGlobal);

		Composition comp = getComposition();
		AudioMode audioMode = getAudioMode();

		_context.activate();
		try {
			comp.prepareExpression(_context.createInitialExpressionScope(comp));

			while (true) {
				IVideoBuffer vb = null;
				try {
					VideoRequest request = _videoRequestQueue.take();
					Time time = request.time;

					if (time == null) break;

					_context.reset();
					_context.setVideoResolution(Resolution.FULL);
					_context.setVideoFrameDuration(comp.getFrameDuration());
					_context.setAudioMode(audioMode);
					_context.setAudioAnimationRate(audioMode.sampleRate/100);
					_context.setTime(time);

					atiIntelLock.lock();
					try {

						vb = comp.renderVideoFrame();

						switch (getVideoChannels()) {
							case RGB:
								_vrSupport.premultiply(vb, Color.BLACK, true);
								break;
							case ALPHA:
								_vrSupport.alphaToGrayscale(vb);
								break;
							case RGBA_STRAIGHT:
								// nothing to do
								break;
							case RGBA_PREMULTIPLIED:
								_vrSupport.premultiply(vb, getColorMatte(), false);
								break;
						}

						// glFinishしておけば finishVideoRequest は AtiIntelLock の外でいいはず。
						//gl.glFinish();

						finishVideoRequest(request, vb);
						_videoResponseQueue.put(this);		// 入れる値は何でもよい。

					} finally {
						atiIntelLock.unlock();
					}

				} catch (InterruptedException e) {
					_logger.warn("unexpected interruption", e);
				} finally {
					if (vb != null) vb.dispose();
				}
			}
		} finally {
			_context.deactivate();
		}
	}

	protected void postVideoRequest(VideoRequest request) throws InterruptedException {
		_videoRequestQueue.put(request);
		_videoResponseQueue.take();
	}

	protected abstract void finishVideoRequest(VideoRequest request, IVideoBuffer vb);


	protected void audioLoop() {
		Composition comp = getComposition();
		AudioMode audioMode = getAudioMode();

		long totalFrames = comp.getDuration().toFrameNumber(audioMode.sampleDuration);

		_context.activate();
		try {
			comp.prepareExpression(_context.createInitialExpressionScope(comp));

			while (true) {
				IAudioBuffer ab = null;
				try {
					AudioRequest request = _audioRequestQueue.take();
					Time time = request.time;

					if (time == null) break;

					_context.reset();
					_context.setAudioMode(audioMode);
					_context.setAudioAnimationRate(audioMode.sampleRate/100);
					_context.setAudioFrameCount((int) Math.min(request.frameCount, totalFrames-time.timeValue));
					_context.setVideoResolution(Resolution.FULL);
					_context.setVideoFrameDuration(comp.getFrameDuration());
					_context.setTime(time);

					ab = comp.renderAudioChunk();
					if (ab == null) {
						ab = createAudioBufferClear();
					}

					finishAudioRequest(request, ab);
					_audioResponseQueue.put(this);		// 入れる値は何でもよい。

				} catch (InterruptedException e) {
					_logger.warn("unexpected interruption", e);
				} finally {
					if (ab != null) ab.dispose();
				}
			}
		} finally {
			_context.deactivate();
		}
	}

	protected void postAudioRequest(AudioRequest request) throws InterruptedException {
		_audioRequestQueue.put(request);
		_audioResponseQueue.take();
	}

	protected abstract void finishAudioRequest(AudioRequest request, IAudioBuffer ab);


	private IAudioBuffer createAudioBufferClear() {
		// このメソッドは AudioRenderSupport にあってもいいかもしれない。

		IAudioBuffer ab = _arSupport.createAudioBuffer();

		switch (ab.getAudioMode().dataType) {
			case SHORT:
				Arrays.fill((short[])ab.getData(), 0, ab.getDataLength(), (short)0);
				break;
			case INT:
				Arrays.fill((int[])ab.getData(), 0, ab.getDataLength(), 0);
				break;
			case FLOAT:
				Arrays.fill((float[])ab.getData(), 0, ab.getDataLength(), 0);
				break;
			default:
				throw new UnsupportedOperationException(
						"unsupported AudioMode.DataType: " + ab.getAudioMode().dataType);
		}

		return ab;
	}


	protected void beginTask(int totalWork) {
		_progressMonitor.beginTask(totalWork);
	}

	protected void currentWork(int currentWork) {
		_progressMonitor.currentWork(currentWork);
	}

	protected void done() {
		_progressMonitor.done();
	}

	protected boolean isCanceled() {
		return _progressMonitor.isCanceled();
	}

}
