/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.output;

import java.util.Arrays;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.misc.AtiIntelLock;
import ch.kuramo.javie.core.services.AudioRenderSupport;
import ch.kuramo.javie.core.services.GLGlobal;
import ch.kuramo.javie.core.services.RenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

public abstract class PushSourceOutput extends Output {

	@Inject
	private RenderContext _context;

	@Inject
	private VideoRenderSupport _vrSupport;

	@Inject
	private AudioRenderSupport _arSupport;

	@Inject
	private GLGlobal _glGlobal;


	public void doOutput(ProgressMonitor progMon) {
		boolean videoEnabled = isVideoEnabled();
		boolean audioEnabled = isAudioEnabled();

		if (!videoEnabled && !audioEnabled) {
			throw new IllegalStateException("neither video nor audio is enabled.");
		}

		initialize();

		final ProgressMonitor monitor = (progMon != null) ? progMon : ProgressMonitor.NULL;

		if (videoEnabled) {
			Thread audioThread = null;

			if (audioEnabled) {
				final ProgressMonitor audioMon = new ProgressMonitor() {
					public void beginTask(int totalWork) { }
					public void currentWork(int currentWork) { }
					public void done() { }
					public boolean isCanceled() { return monitor.isCanceled(); }
				};
				audioThread = new Thread() { public void run() { audioLoop(audioMon); } };
				audioThread.start();
			}

			try {
				videoLoop(monitor);
			} finally {
				if (audioThread != null) {
					// TODO writeAudioでブロックされているときは永久に待ち続けるかもしれない。
					try { audioThread.join(); } catch (InterruptedException e) { }
				}
				finish();
			}

		} else /*if (audioEnabled)*/ {
			try {
				audioLoop(monitor);
			} finally {
				finish();
			}
		}
	}

	protected abstract void initialize();

	protected abstract void finish();


	protected void videoLoop(ProgressMonitor monitor) {
		AtiIntelLock atiIntelLock = AtiIntelLock.get(_glGlobal);

		Composition comp = getComposition();
		AudioMode audioMode = getAudioMode();

		Time frameDuration = comp.getFrameDuration();
		long numFrames = comp.getDuration().toFrameNumber(frameDuration);

		monitor.beginTask((int)numFrames);

		_context.activate();
		try {
			comp.prepareExpression(_context.createInitialExpressionScope(comp));

			for (long i = 0; i < numFrames && !monitor.isCanceled(); ++i) {
				Time time = Time.fromFrameNumber(i, frameDuration);
				_context.reset();
				_context.setVideoResolution(Resolution.FULL);
				_context.setVideoFrameDuration(comp.getFrameDuration());
				_context.setAudioMode(audioMode);
				_context.setAudioAnimationRate(audioMode.sampleRate/100);
				_context.setTime(time);

				IVideoBuffer vb = null;
				try {
					atiIntelLock.lock();
					try {

						vb = comp.renderVideoFrame();

						switch (getVideoChannels()) {
							case RGB:
								_vrSupport.premultiply(vb, Color.BLACK, true);
								break;
							case ALPHA:
								_vrSupport.alphaToGrayscale(vb);
								break;
							case RGBA_STRAIGHT:
								// nothing to do
								break;
							case RGBA_PREMULTIPLIED:
								_vrSupport.premultiply(vb, getColorMatte(), false);
								break;
						}

						// glFinishしておけば writeVideo は AtiIntelLock の外でいいはず。
						//gl.glFinish();

						writeVideo(i, time, vb);

					} finally {
						atiIntelLock.unlock();
					}
				} finally {
					if (vb != null) vb.dispose();
				}

				monitor.currentWork((int)i);
			}
		} finally {
			_context.deactivate();
		}

		monitor.done();
	}

	protected abstract void writeVideo(long frameNumber, Time time, IVideoBuffer vb);


	protected void audioLoop(ProgressMonitor monitor) {
		Composition comp = getComposition();
		AudioMode audioMode = getAudioMode();

		long totalFrames = comp.getDuration().toFrameNumber(audioMode.sampleDuration);

		// １回あたりの処理量を１秒分のフレーム数（＝サンプルレートの値と同じ数）とする。
		int chunkFrames = audioMode.sampleRate;

		monitor.beginTask((int)totalFrames);

		_context.activate();
		try {
			comp.prepareExpression(_context.createInitialExpressionScope(comp));

			for (long i = 0; i < totalFrames && !monitor.isCanceled(); ) {
				Time time = Time.fromFrameNumber(i, audioMode.sampleDuration);
				_context.reset();
				_context.setAudioMode(audioMode);
				_context.setAudioAnimationRate(audioMode.sampleRate/100);
				_context.setAudioFrameCount((int) Math.min(chunkFrames, totalFrames-i));
				_context.setVideoResolution(Resolution.FULL);
				_context.setVideoFrameDuration(comp.getFrameDuration());
				_context.setTime(time);

				IAudioBuffer ab = comp.renderAudioChunk();
				try {
					if (ab == null) {
						ab = createAudioBufferClear();
					}
					writeAudio(i, time, ab);
					i+= ab.getFrameCount();
				} finally {
					ab.dispose();
				}

				monitor.currentWork((int)i);
			}
		} finally {
			_context.deactivate();
		}

		monitor.done();
	}

	protected abstract void writeAudio(long frameNumber, Time time, IAudioBuffer ab);


	private IAudioBuffer createAudioBufferClear() {
		// このメソッドは AudioRenderSupport にあってもいいかもしれない。
		// PullSourceOutputにも同じメソッドがあるし。

		IAudioBuffer ab = _arSupport.createAudioBuffer();

		switch (ab.getAudioMode().dataType) {
			case SHORT:
				Arrays.fill((short[])ab.getData(), 0, ab.getDataLength(), (short)0);
				break;
			case INT:
				Arrays.fill((int[])ab.getData(), 0, ab.getDataLength(), 0);
				break;
			case FLOAT:
				Arrays.fill((float[])ab.getData(), 0, ab.getDataLength(), 0);
				break;
			default:
				throw new UnsupportedOperationException(
						"unsupported AudioMode.DataType: " + ab.getAudioMode().dataType);
		}

		return ab;
	}

}
