/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#import "QTMovieOutputJNI.h"
#import "QTMovieOutput.h"


JNIEXPORT jint JNICALL JNI_OnLoad(JavaVM* vm, void* reserved)
{
	return JNI_VERSION_1_4;
}

NSString* jstringToNSString(JNIEnv* env, jstring jstr)
{
	const jchar* chars = (*env)->GetStringChars(env, jstr, NULL);
	NSString* nsstr = [NSString stringWithCharacters:(UniChar*)chars
											  length:(*env)->GetStringLength(env, jstr)];
	(*env)->ReleaseStringChars(env, jstr, chars);
	return nsstr;
}

jstring jstringFromNSString(JNIEnv* env, NSString* nsstr)
{
	CFStringRef cfStrRef = (CFStringRef)nsstr;
	CFIndex length = CFStringGetLength(cfStrRef);
	const UniChar* chars = CFStringGetCharactersPtr(cfStrRef);
	if (chars != NULL) {
		return (*env)->NewString(env, chars, length);
	} else {
		UniChar* buffer = malloc(length * sizeof(UniChar));
		CFStringGetCharacters(cfStrRef, CFRangeMake(0, length), buffer);
		jstring jstr = (*env)->NewString(env, buffer, length);
		free(buffer);
		return jstr;
	}
}

jobject newInteger(JNIEnv* env, jint intValue)
{
	jclass cls = (*env)->FindClass(env, "Ljava/lang/Integer;");
	jmethodID cns = (*env)->GetMethodID(env, cls, "<init>", "(I)V");
	return (*env)->NewObject(env, cls, cns, intValue);
}

jobject newDouble(JNIEnv* env, jdouble doubleValue)
{
	jclass cls = (*env)->FindClass(env, "Ljava/lang/Double;");
	jmethodID cns = (*env)->GetMethodID(env, cls, "<init>", "(D)V");
	return (*env)->NewObject(env, cls, cns, doubleValue);
}

jint jbyteArrayToQTAtomContainer(JNIEnv* env, jbyteArray ba, QTAtomContainer* ac)
{
	jint error = noErr;
	char* p = NULL;
	
	p = (*env)->GetPrimitiveArrayCritical(env, ba, NULL);
	if (p == NULL) {
		NSLog(@"GetPrimitiveArrayCritical: failed");
		error = memFullErr;
		goto bail;
	}
	
	error = PtrToHand(p, (Handle*)ac, (*env)->GetArrayLength(env, ba));
	if (error != noErr) {
		NSLog(@"PtrToHand: error %d", error);
		goto bail;
	}
	
bail:
	if (p != NULL) (*env)->ReleasePrimitiveArrayCritical(env, ba, p, 0);
	return error;
}

jint jbyteArrayFromQTAtomContainer(JNIEnv* env, QTAtomContainer ac, jbyteArray* ba)
{
	size_t size = GetHandleSize(ac);
	jbyteArray newArray = (*env)->NewByteArray(env, size);
	if (newArray == NULL) {
		NSLog(@"NewByteArray: failed");
		return memFullErr;
	}
	
	// OSXではハンドルのロックは(SetHandleSizeを使わないなら)必要ないらしいのでロックしない。
	//SInt8 hState = HGetState(ac);
	//HLock(ac);
	
	(*env)->SetByteArrayRegion(env, newArray, 0, size, (jbyte*)*ac);
	
	//HSetState(ac, hState);
	
	*ba = newArray;
	return noErr;
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_output_MacOSXQTMovieOutput_doVideoCompressorSettings
(JNIEnv* env, jclass class,
 jbyteArray defaultSettings, jdouble defaultFrameRate, jboolean showDialog, jobjectArray newSettings)
{
	NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
	jint error = noErr;
	
	QTAtomContainer defaultAtomContainer = NULL;
	QTAtomContainer newAtomContainer = NULL;
	
	if (defaultSettings != NULL) {
		error = jbyteArrayToQTAtomContainer(env, defaultSettings, &defaultAtomContainer);
		if (error != noErr) goto bail;
	}
	
	short newDepth;
	double newFrameRate;
	NSString* newSettingsAsText;
	error = [QTMovieOutput doVideoCompressorSettings:defaultAtomContainer :defaultFrameRate :showDialog
													:&newAtomContainer :&newDepth :&newFrameRate :&newSettingsAsText];
	if (error != noErr) goto bail;
	
	jbyteArray byteArray;
	error = jbyteArrayFromQTAtomContainer(env, newAtomContainer, &byteArray);
	if (error != noErr) goto bail;
	
	(*env)->SetObjectArrayElement(env, newSettings, 0, byteArray);
	(*env)->SetObjectArrayElement(env, newSettings, 1, newInteger(env, newDepth));
	(*env)->SetObjectArrayElement(env, newSettings, 2, newDouble(env, newFrameRate));
	(*env)->SetObjectArrayElement(env, newSettings, 3, jstringFromNSString(env, newSettingsAsText));
	
bail:
	if (defaultAtomContainer != NULL) QTDisposeAtomContainer(defaultAtomContainer);
	if (newAtomContainer != NULL) QTDisposeAtomContainer(newAtomContainer);
	[pool release];
	return error;
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_output_MacOSXQTMovieOutput_doAudioCompressorSettings
(JNIEnv* env, jclass class,
 jbyteArray defaultSettings, jdouble defaultSampleRate, jboolean showDialog, jobjectArray newSettings)
{
	NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
	jint error = noErr;
	
	QTAtomContainer defaultAtomContainer = NULL;
	QTAtomContainer newAtomContainer = NULL;
	
	if (defaultSettings != NULL) {
		error = jbyteArrayToQTAtomContainer(env, defaultSettings, &defaultAtomContainer);
		if (error != noErr) goto bail;
	}
	
	double newSampleRate;
	NSString* newSettingsAsText;
	error = [QTMovieOutput doAudioCompressorSettings:defaultAtomContainer :defaultSampleRate :showDialog
													:&newAtomContainer :&newSampleRate :&newSettingsAsText];
	if (error != noErr) goto bail;
	
	jbyteArray byteArray;
	error = jbyteArrayFromQTAtomContainer(env, newAtomContainer, &byteArray);
	if (error != noErr) goto bail;
	
	(*env)->SetObjectArrayElement(env, newSettings, 0, byteArray);
	(*env)->SetObjectArrayElement(env, newSettings, 1, newDouble(env, newSampleRate));
	(*env)->SetObjectArrayElement(env, newSettings, 2, jstringFromNSString(env, newSettingsAsText));
	
bail:
	if (defaultAtomContainer != NULL) QTDisposeAtomContainer(defaultAtomContainer);
	if (newAtomContainer != NULL) QTDisposeAtomContainer(newAtomContainer);
	[pool release];
	return error;
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_output_MacOSXQTMovieOutput_doOutput
(JNIEnv* env, jobject javaObj,
 jbyteArray videoSettings, jbyteArray audioSettings, jstring filename,
 jint vWidth, jint vHeight, jint vTimeScale, jlong vFrameDuration, jlong vDuration,
 jint aChannels, jint aSampleRate, jint aSampleSize, jboolean aFloat, jlong aDuration)
{
	if (vDuration <= 0 && aDuration <= 0) return paramErr;
	
	NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
	jint error = noErr;
	
	QTAtomContainer videoAtomContainer = NULL;
	QTAtomContainer audioAtomContainer = NULL;
	
	QTMovieOutput* qtOut = [[[QTMovieOutput alloc] init] autorelease];
	error = [qtOut openMovieExportComponent];
	if (error != noErr) goto bail;
	
	if (videoSettings != NULL) {
		error = jbyteArrayToQTAtomContainer(env, videoSettings, &videoAtomContainer);
		if (error != noErr) goto bail;
	}
	
	if (audioSettings != NULL) {
		error = jbyteArrayToQTAtomContainer(env, audioSettings, &audioAtomContainer);
		if (error != noErr) goto bail;
	}
	
	if (videoAtomContainer != NULL || audioAtomContainer != NULL) {
		error = [qtOut setCompressorSettings:videoAtomContainer :audioAtomContainer];
		if (error != noErr) goto bail;
	}
	
	if (vDuration > 0) {
		error = [qtOut addVideoSource:vWidth:vHeight:vTimeScale:vFrameDuration:vDuration];
		if (error != noErr) goto bail;
	}
	
	if (aDuration > 0) {
		error = [qtOut addAudioSource:aChannels:aSampleRate:aSampleSize:aFloat:aDuration];
		if (error != noErr) goto bail;
	}
	
	error = [qtOut doOutput:env:javaObj:(CFStringRef)jstringToNSString(env, filename)];
	if (error != noErr) goto bail;
	
bail:
	if (videoAtomContainer != NULL) QTDisposeAtomContainer(videoAtomContainer);
	if (audioAtomContainer != NULL) QTDisposeAtomContainer(audioAtomContainer);
	[pool release];
	return error;
}
