/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects;

import java.nio.FloatBuffer;

import javax.media.opengl.GL2;

import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.services.IVideoEffectContext;

public class Texture1D {

	private static final float[] FLOAT0000 = new float[] { 0, 0, 0, 0 };

	public static int fromArray(IArray<float[]> array, IVideoEffectContext context) {
		return fromArray(array.getArray(), 0, array.getLength(),
				GL2.GL_ALPHA, GL2.GL_LINEAR, GL2.GL_CLAMP_TO_EDGE, context);
	}

	public static int fromArray(float[] array, IVideoEffectContext context) {
		return fromArray(array, 0, array.length,
				GL2.GL_ALPHA, GL2.GL_LINEAR, GL2.GL_CLAMP_TO_EDGE, context);
	}

	public static int fromArray(
			IArray<float[]> array,
			int format, int filter, int wrapMode,
			IVideoEffectContext context) {

		return fromArray(array.getArray(), 0, array.getLength(),
				format, filter, wrapMode, context);
	}

	public static int fromArray(
			float[] array, int offset, int length,
			int format, int filter, int wrapMode,
			IVideoEffectContext context) {

		int internalFormat;
		int width;

		switch (format) {
			case GL2.GL_ALPHA:
				switch (context.getColorMode()) {
					case RGBA8:			internalFormat = GL2.GL_ALPHA8; break;
					case RGBA16:		internalFormat = GL2.GL_ALPHA16; break;
					case RGBA16_FLOAT:	internalFormat = GL2.GL_ALPHA16F; break;
					case RGBA32_FLOAT:	internalFormat = GL2.GL_ALPHA32F; break;
					default: throw new IllegalStateException("unknown ColorMode: " + context.getColorMode());
				}
				width = length;
				break;

			case GL2.GL_RGBA:
				switch (context.getColorMode()) {
					case RGBA8:			internalFormat = GL2.GL_RGBA8; break;
					case RGBA16:		internalFormat = GL2.GL_RGBA16; break;
					case RGBA16_FLOAT:	internalFormat = GL2.GL_RGBA16F; break;
					case RGBA32_FLOAT:	internalFormat = GL2.GL_RGBA32F; break;
					default: throw new IllegalStateException("unknown ColorMode: " + context.getColorMode());
				}
				width = length / 4;
				if (width * 4 != length) {
					throw new IllegalArgumentException("length must be multiple of 4");
				}
				break;

			default:
				throw new IllegalArgumentException("format must be GL_FLOAT or GL_RGBA");
		}

		GL2 gl = context.getGL().getGL2();

		int[] texture = new int[1];
		int[] current = new int[1];
		gl.glGetIntegerv(GL2.GL_TEXTURE_BINDING_1D, current, 0);
		try {
			gl.glGenTextures(1, texture, 0);
			gl.glBindTexture(GL2.GL_TEXTURE_1D, texture[0]);

			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_MIN_FILTER, filter);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_MAG_FILTER, filter);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_WRAP_S, wrapMode);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_WRAP_T, GL2.GL_CLAMP_TO_EDGE);

			if (wrapMode == GL2.GL_CLAMP_TO_BORDER) {
				gl.glTexParameterfv(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_BORDER_COLOR, FLOAT0000, 0);
			}

			gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, internalFormat,
					width, 0, format, GL2.GL_FLOAT, FloatBuffer.wrap(array, offset, length));

			int result = texture[0];
			texture[0] = 0;
			return result;

		} finally {
			gl.glBindTexture(GL2.GL_TEXTURE_1D, current[0]);
			if (texture[0] != 0) gl.glDeleteTextures(1, texture, 0);
		}
	}

	private Texture1D() { }

}
