/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.blurSharpen;

import java.util.HashSet;
import java.util.Set;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.IVideoBuffer.TextureWrapMode;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IBlurSupport;
import ch.kuramo.javie.api.services.IConvolutionSupport;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.api.services.IBlurSupport.BlurDimensions;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.DirectionalBlur", category=Categories.BLUR_AND_SHARPEN)
public class DirectionalBlur {

	public enum BlurType { BOX, GAUSSIAN }


	@Property
	private IAnimatableDouble direction;

	@Property(value="0", min="0", max="500")
	private IAnimatableDouble blurLength; 

	@Property("BOX")
	private IAnimatableEnum<BlurType> blurType;

	@Property
	private IAnimatableBoolean repeatEdgePixels;

	@Property("true")
	private IAnimatableBoolean fast;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IBlurSupport blurSupport;

	private final IConvolutionSupport convolutionSupport;

	@Inject
	public DirectionalBlur(
			IVideoEffectContext context, IVideoRenderSupport support,
			IBlurSupport blurSupport, IConvolutionSupport convolutionSupport) {

		this.context = context;
		this.support = support;
		this.blurSupport = blurSupport;
		this.convolutionSupport = convolutionSupport;
	}

	public VideoBounds getVideoBounds() {
		VideoBounds bounds = context.getPreviousBounds();
		if (bounds.isEmpty()) {
			return bounds;
		}

		double blurLength = context.value(this.blurLength);

		blurLength = context.getVideoResolution().scale(blurLength);
		if (blurLength == 0) {
			return bounds;
		}

		if (context.value(this.repeatEdgePixels)) {
			return bounds;
		}

		double direction = context.value(this.direction);

		return calcResultBounds(bounds, direction, blurLength);
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();
		if (source.getBounds().isEmpty()) {
			return source;
		}

		double blurLength = context.value(this.blurLength);
		boolean fast = (blurLength > 50) || context.value(this.fast);

		blurLength = context.getVideoResolution().scale(blurLength);
		if (blurLength == 0) {
			return source;
		}

		double direction = context.value(this.direction);
		BlurType blurType = context.value(this.blurType);
		boolean repeatEdgePixels = context.value(this.repeatEdgePixels);

		try {
			if (fast) {
				return doFastDirectionalBlur(source, direction, blurLength, blurType, repeatEdgePixels);
			} else {
				return doDirectionalBlur(source, direction, blurLength, blurType, repeatEdgePixels);
			}
		} finally {
			source.dispose();
		}
	}

	private IVideoBuffer doDirectionalBlur(
			IVideoBuffer source, double direction, double blurLength, BlurType blurType, boolean repeatEdgePixels) {

		float[] kernel;
		switch (blurType) {
			case BOX:
				kernel = blurSupport.createBoxBlurKernel(blurLength);
				break;

			case GAUSSIAN:
				kernel = blurSupport.createGaussianBlurKernel(blurLength);
				break;

			default:
				throw new RuntimeException("unknown BlurType: " + blurType);
		}

		double rotRadians = Math.toRadians(direction);
		double vx =  Math.sin(rotRadians);
		double vy = -Math.cos(rotRadians);

		float[] offset = new float[kernel.length*2];
		for (int i = 1, n = kernel.length/2; i <= n; ++i) {
			offset[(n-i)*2  ] = (float)(vx*i);
			offset[(n-i)*2+1] = (float)(vy*i);
			offset[(n+i)*2  ] = (float)(-vx*i);
			offset[(n+i)*2+1] = (float)(-vy*i);
		}

		IVideoBuffer buffer = null;
		try {
			VideoBounds bounds = source.getBounds();
			VideoBounds resultBounds = repeatEdgePixels ? bounds : calcResultBounds(bounds, direction, blurLength);
			buffer = context.createVideoBuffer(resultBounds);

			source.setTextureFilter(TextureFilter.LINEAR);
			source.setTextureWrapMode(repeatEdgePixels ? TextureWrapMode.CLAMP_TO_EDGE : TextureWrapMode.CLAMP_TO_BORDER);

			convolutionSupport.convolve(source, buffer, kernel, offset);

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

	private IVideoBuffer doFastDirectionalBlur(
			IVideoBuffer source, double direction, double blurLength, BlurType blurType, boolean repeatEdgePixels) {

		Set<IVideoBuffer> tmpBuffers = new HashSet<IVideoBuffer>();
		try {
			VideoBounds bounds = source.getBounds();
			Vec2d rotCenter = new Vec2d(bounds.x+bounds.width*0.5, bounds.y+bounds.height*0.5);
			double rotRadians = Math.toRadians(direction);


			// ブラーの方向がちょうど垂直方向になるように回転する。
			final VideoBounds rotatedBounds = calcRotatedBounds(bounds, rotCenter, -rotRadians);
			final double[][] rotTexCoords = calcRotationTexCoords(bounds, rotatedBounds, rotCenter, -rotRadians);

			Runnable rotOp = new Runnable() {
				public void run() {
					support.ortho2D(rotatedBounds);
					support.quad2D(rotatedBounds, rotTexCoords);
				}
			};

			source.setTextureFilter(TextureFilter.LINEAR);
			source.setTextureWrapMode(repeatEdgePixels ? TextureWrapMode.CLAMP_TO_EDGE : TextureWrapMode.CLAMP_TO_BORDER);

			IVideoBuffer buffer = context.createVideoBuffer(rotatedBounds);
			tmpBuffers.add(buffer);

			support.useFramebuffer(rotOp, 0, buffer, source);


			// 垂直方向にブラーをかける。
			switch (blurType) {
				case BOX:
					buffer = blurSupport.boxBlur(
							buffer, blurLength, BlurDimensions.VERTICAL, repeatEdgePixels, true, 1);
					tmpBuffers.add(buffer);
					break;

				case GAUSSIAN:
					buffer = blurSupport.gaussianBlur(
							buffer, blurLength, BlurDimensions.VERTICAL, repeatEdgePixels, true);
					tmpBuffers.add(buffer);
					break;

				default:
					throw new RuntimeException("unknown BlurType: " + blurType);
			}


			// 反対方向に回転するしてできあがり。
			final VideoBounds resultBounds = repeatEdgePixels ? bounds : calcResultBounds(bounds, direction, blurLength);
			final double[][] revRotTexCoords = calcRotationTexCoords(buffer.getBounds(), resultBounds, rotCenter, rotRadians);

			Runnable revRotOp = new Runnable() {
				public void run() {
					support.ortho2D(resultBounds);
					support.quad2D(resultBounds, revRotTexCoords);
				}
			};

			buffer.setTextureFilter(TextureFilter.LINEAR);

			IVideoBuffer result = context.createVideoBuffer(resultBounds);
			tmpBuffers.add(result);

			support.useFramebuffer(revRotOp, 0, result, buffer);

			tmpBuffers.remove(result);
			return result;

		} finally {
			for (IVideoBuffer vb : tmpBuffers) {
				vb.dispose();
			}
		}
	}

	private VideoBounds calcResultBounds(VideoBounds bounds, double direction, double blurLength) {
		double rotRadians = Math.toRadians(direction);
		double dx = Math.abs(Math.sin(rotRadians)) * blurLength;
		double dy = Math.abs(Math.cos(rotRadians)) * blurLength;

		return new VideoBounds(
				bounds.x - dx,
				bounds.y - dy,
				bounds.width + (int)Math.ceil(dx*2),
				bounds.height + (int)Math.ceil(dy*2));
	}

	private VideoBounds calcRotatedBounds(VideoBounds bounds, Vec2d rotCenter, double rotRadians) {
		double[][] pt = new double[][] {
				{ bounds.x             , bounds.y               },
				{ bounds.x+bounds.width, bounds.y               },
				{ bounds.x+bounds.width, bounds.y+bounds.height },
				{ bounds.x             , bounds.y+bounds.height },
		};

		double cos = Math.cos(rotRadians);
		double sin = Math.sin(rotRadians);

		double left   = Double.POSITIVE_INFINITY;
		double top    = Double.POSITIVE_INFINITY;
		double right  = Double.NEGATIVE_INFINITY;
		double bottom = Double.NEGATIVE_INFINITY;

		for (int i = 0; i < 4; ++i) {
			rotate(pt[i], rotCenter, cos, sin);
			left   = Math.min(pt[i][0], left);
			top    = Math.min(pt[i][1], top);
			right  = Math.max(pt[i][0], right);
			bottom = Math.max(pt[i][1], bottom);
		}

		return new VideoBounds(
				left,
				top,
				(int)Math.ceil(right-left),
				(int)Math.ceil(bottom-top));
	}

	private double[][] calcRotationTexCoords(VideoBounds bounds, VideoBounds rotatedBounds, Vec2d rotCenter, double rotRadians) {
		double x = rotatedBounds.x - bounds.x;
		double y = rotatedBounds.y - bounds.y;
		double w = rotatedBounds.width;
		double h = rotatedBounds.height;

		double[][] pt = new double[][] {
				{ x  , y   },
				{ x+w, y   },
				{ x+w, y+h },
				{ x  , y+h },
		};

		rotCenter = new Vec2d(rotCenter.x-bounds.x, rotCenter.y-bounds.y);

		double cos = Math.cos(-rotRadians);
		double sin = Math.sin(-rotRadians);

		for (int i = 0; i < 4; ++i) {
			rotate(pt[i], rotCenter, cos, sin);
		}

		return pt;
	}

	private void rotate(double[] pt, Vec2d rotCenter, double cos, double sin) {
		double x = pt[0], y = pt[1];
		double x0 = rotCenter.x, y0 = rotCenter.y;
		pt[0] = (x-x0)*cos - (y-y0)*sin + x0;
		pt[1] = (x-x0)*sin + (y-y0)*cos + y0;
	}

}
