/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.blurSharpen;

import java.nio.FloatBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableInteger;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.IVideoBuffer.TextureWrapMode;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.ZoomBlur", category=Categories.BLUR_AND_SHARPEN)
public class ZoomBlur {

	public enum Direction { ZOOM_IN, ZOOM_OUT, ZOOM_BOTH } 


	@Property(value="0", min="0", max="1000")
	private IAnimatableDouble amount;

	@Property
	private IAnimatableVec2d center;

	@Property("ZOOM_IN")
	private IAnimatableEnum<Direction> direction;

	@Property(value="20", min="0")
	private IAnimatableInteger maxSamples;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram zoomInProgram;

	private final IShaderProgram zoomOutProgram;

	private final IShaderProgram zoomBothProgram;

	@Inject
	public ZoomBlur(IVideoEffectContext context, IVideoRenderSupport support, IShaderRegistry shaders) {
		this.context = context;
		this.support = support;
		zoomInProgram = shaders.getProgram(ZoomBlur.class, "ZOOM_IN");
		zoomOutProgram = shaders.getProgram(ZoomBlur.class, "ZOOM_OUT");
		zoomBothProgram = shaders.getProgram(ZoomBlur.class, "ZOOM_BOTH");
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();
		VideoBounds bounds = source.getBounds();
		if (bounds.isEmpty()) {
			return source;
		}

		Resolution resolution = context.getVideoResolution();

		double amount = resolution.scale(context.value(this.amount));
		int maxSamples = context.value(this.maxSamples);
		if (amount == 0 || maxSamples == 0) {
			return source;
		}

		Vec2d center = resolution.scale(context.value(this.center));
		Direction direction = context.value(this.direction);

		try {
			IShaderProgram program;
			double amountScale;
			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();

			switch (direction) {
				case ZOOM_IN:
					program = zoomInProgram;
					amountScale = resolution.scale(200.0);
					uniforms.add(new GLUniformData("amountIn", (float)Math.min(amount, amountScale)));
					break;

				case ZOOM_OUT:
					program = zoomOutProgram;
					amountScale = resolution.scale(200.0);
					uniforms.add(new GLUniformData("amountOut", (float)amount));
					break;

				case ZOOM_BOTH:
					program = zoomBothProgram;
					amountScale = resolution.scale(400.0);
					uniforms.add(new GLUniformData("amountIn", (float)Math.min(amount, amountScale)));
					uniforms.add(new GLUniformData("amountOut", (float)amount));
					break;

				default:
					throw new RuntimeException("unknown zoom direction: " + direction);
			}

			uniforms.add(new GLUniformData("source", 0));
			uniforms.add(new GLUniformData("center", 2, toFloatBuffer(center.x-bounds.x, center.y-bounds.y)));
			uniforms.add(new GLUniformData("amountScale", (float)amountScale));
			uniforms.add(new GLUniformData("maxSamples", (float)maxSamples));

			source.setTextureFilter(TextureFilter.LINEAR);
			source.setTextureWrapMode(TextureWrapMode.CLAMP_TO_EDGE);

			return support.useShaderProgram(program, uniforms, null, source);

		} finally {
			source.dispose();
		}
	}

	private FloatBuffer toFloatBuffer(double... values) {
		float[] array = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			array[i] = (float)values[i];
		}
		return FloatBuffer.wrap(array);
	}

	private static String[] createShaderSource(Direction direction) {
		boolean in = (direction != Direction.ZOOM_OUT);
		boolean out = (direction != Direction.ZOOM_IN);

		List<String> list = new ArrayList<String>(Arrays.asList(new String[] {
					"uniform sampler2DRect source;",
					"uniform float amountScale;",
			in ?	"uniform float amountIn;" : "",
			out ?	"uniform float amountOut;" : "",
					"uniform vec2 center;",
					"uniform float maxSamples;",
					"",
					"void main(void)",
					"{",
					"	vec2 coord = gl_TexCoord[0].st;",
					"	vec2 v = center - coord;",
					"	float d = length(v);",
					"",
					"	vec4 color = texture2DRect(source, coord);",
					"",
					"	if (d > 0.0) {",
					"		int n = 0;"
		}));

		if (in) {
			list.addAll(Arrays.asList(new String[] {
					"		float a1 = amountIn*d/amountScale;",
					"		int   n1 = int(min(a1+1.0, maxSamples));",
					"		vec2  v1 = v/d*a1/float(n1);",
					"		for (int i = 1; i <= n1; ++i) {",
					"			color += texture2DRect(source, coord + v1*float(i));",
					"		}",
					"		n += n1;"
			}));
		}

		if (out) {
			list.addAll(Arrays.asList(new String[] {
					"		float a2 = amountOut*d/amountScale;",
					"		int   n2 = int(min(a2+1.0, maxSamples));",
					"		vec2  v2 = v/d*a2/float(n2);",
					"		for (int i = 1; i <= n2; ++i) {",
					"			color += texture2DRect(source, coord - v2*float(i));",
					"		}",
					"		n += n2;"
			}));
		}

		list.addAll(Arrays.asList(new String[] {
					"",
					"		color /= float(n+1);",
					"	}",
					"",
					"	gl_FragColor = color;",
					"}"
		}));

		return list.toArray(new String[list.size()]);
	}

	@ShaderSource
	public static final String[] ZOOM_IN = createShaderSource(Direction.ZOOM_IN);

	@ShaderSource
	public static final String[] ZOOM_OUT = createShaderSource(Direction.ZOOM_OUT);

	@ShaderSource
	public static final String[] ZOOM_BOTH = createShaderSource(Direction.ZOOM_BOTH);

}
