/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.generate;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GL2;
import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.BlendMode;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IBlendSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.effects.Texture1D;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.Grid", category=Categories.GENERATE)
public class Grid {

	@Property
	private IAnimatableVec2d anchor;

	@Property(value="100", min="0", max="4000")
	private IAnimatableVec2d gridSize;

	@Property(value="5", min="0", max="4000")
	private IAnimatableVec2d border;

	@Property(value="0", min="0", max="400")
	private IAnimatableVec2d feather;

	@Property
	private IAnimatableBoolean invert;

	@Property("1,1,1")
	private IAnimatableColor color;

	@Property(value="100", min="0", max="100")
	private IAnimatableDouble opacity;

	@Property
	private IAnimatableEnum<BlendMode> blendMode;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IBlendSupport blendSupport;

	private final IArrayPools arrayPools;

	private final IShaderProgram gridSamplerProgram;

	private final IShaderProgram invertSamplerProgram;

	@Inject
	public Grid(
			IVideoEffectContext context, IVideoRenderSupport support, IBlendSupport blendSupport,
			IArrayPools arrayPools, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		this.blendSupport = blendSupport;
		this.arrayPools = arrayPools;
		gridSamplerProgram = shaders.getProgram(Grid.class, "GRID_SAMPLER");
		invertSamplerProgram = shaders.getProgram(Grid.class, "INVERT_SAMPLER");
	}

	private class GridParams {
		private final Vec2d anchor;
		private final Vec2d gridSize;
		private final Vec2d border;

		private final boolean invert;
		private final Color color;
		private final double opacity;

		private final int[] texture;
		private final int[] texSize = new int[2];

		private GridParams() {
			Resolution resolution = context.getVideoResolution();

			anchor = resolution.scale(context.value(Grid.this.anchor));
			gridSize = resolution.scale(context.value(Grid.this.gridSize));
			border = resolution.scale(context.value(Grid.this.border));

			invert = context.value(Grid.this.invert);
			opacity = context.value(Grid.this.opacity) / 100;

			Color color = context.value(Grid.this.color).clamp();
			double a = color.a;
			this.color = new Color(color.r*a, color.g*a, color.b*a, a);

			Vec2d feather = resolution.scale(context.value(Grid.this.feather));
			texture = createGridTextures(feather, texSize);
		}
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer original;
		final VideoBounds bounds;

		BlendMode blendMode = context.value(this.blendMode);

		if (blendMode == BlendMode.NONE) {
			bounds = context.getPreviousBounds();
			original = context.createVideoBuffer(bounds);
		} else {
			original = context.doPreviousEffect();
			bounds = original.getBounds();
		}

		if (bounds.isEmpty()) {
			return original;
		}


		GridParams params = null;
		IVideoBuffer grid = null;
		try {
			final GridParams p = params = new GridParams();

			IShaderProgram program = p.invert ? invertSamplerProgram : gridSamplerProgram;

			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texX", 0));
			uniforms.add(new GLUniformData("texY", 1));

			uniforms.add(new GLUniformData("texSize", 4, FloatBuffer.wrap(new float[] {
					p.texSize[0], p.texSize[1], p.texSize[0], p.texSize[1] })));

			uniforms.add(new GLUniformData("texOffset", 4, FloatBuffer.wrap(new float[] {
					(float)((p.gridSize.x-p.border.x-p.texSize[0])*0.5),
					(float)((p.gridSize.y-p.border.y-p.texSize[1])*0.5),
					(float)((p.gridSize.x+p.border.x-p.texSize[0])*0.5),
					(float)((p.gridSize.y+p.border.y-p.texSize[1])*0.5) })));

			float gridSizeX = (float)Math.max(p.gridSize.x, 1e-10);
			float gridSizeY = (float)Math.max(p.gridSize.y, 1e-10);
			uniforms.add(new GLUniformData("gridSize", 4, FloatBuffer.wrap(new float[] {
					gridSizeX, gridSizeY, gridSizeX, gridSizeY })));

			uniforms.add(new GLUniformData("anchor", 2, FloatBuffer.wrap(new float[] {
					(float)(p.anchor.x-bounds.x), (float)(p.anchor.y-bounds.y) })));

			uniforms.add(new GLUniformData("color", 4, FloatBuffer.wrap(new float[] {
					(float)p.color.r, (float)p.color.g, (float)p.color.b, (float)p.color.a })));

			Runnable operation = new Runnable() {
				public void run() {
					GL2 gl = context.getGL().getGL2();
					gl.glActiveTexture(GL2.GL_TEXTURE0);
					gl.glBindTexture(GL2.GL_TEXTURE_1D, p.texture[0]);
					gl.glActiveTexture(GL2.GL_TEXTURE1);
					gl.glBindTexture(GL2.GL_TEXTURE_1D, p.texture[1]);

					support.ortho2D(bounds);
					support.quad2D(bounds);
				}
			};

			grid = context.createVideoBuffer(bounds);

			support.useShaderProgram(program, uniforms, operation, GL2.GL_TEXTURE_BIT, grid);


			if (blendMode == BlendMode.NONE && p.opacity == 1) {
				IVideoBuffer result = grid;
				grid = null;
				return result;

			} else {
				return blendSupport.blend(grid, original, null, blendMode, p.opacity, context);
			}

		} finally {
			if (original != null) original.dispose();
			if (grid != null) grid.dispose();
			if (params != null) deleteTextures(params.texture);
		}
	}

	private int[] createGridTextures(Vec2d feather, int[] texSize) {
		int tex0 = 0, tex1 = 0;
		IArray<float[]> data0 = null, data1 = null;
		try {
			data0 = createGridData(feather.x);
			tex0 = Texture1D.fromArray(data0, context);
			texSize[0] = data0.getLength();

			data1 = createGridData(feather.y);
			tex1 = Texture1D.fromArray(data1, context);
			texSize[1] = data1.getLength();

			int[] result = new int[] { tex0, tex1 };
			tex0 = tex1 = 0;
			return result;

		} finally {
			if (data0 != null) data0.release();
			if (data1 != null) data1.release();
			if (tex0 != 0) deleteTextures(tex0);
			if (tex1 != 0) deleteTextures(tex1);
		}
	}

	private IArray<float[]> createGridData(double feather) {
		int halfLen = (int)Math.ceil(feather) + 1;

		IArray<float[]> data = arrayPools.getFloatArray(halfLen*2);
		float[] array = data.getArray();
		int arrayLen = data.getLength();

		for (int i = 0; i < arrayLen; ++i) {
			double t = i + 0.5;
			double d = (t <= halfLen-feather) ? 0
					 : (t >= halfLen+feather) ? 1
					 : 0.5*(1-Math.cos(2*Math.PI*(t-(halfLen-feather))/(4*feather)));
			array[i] = (float)d;
		}

		return data;
	}

	private void deleteTextures(int... texture) {
		GL2 gl = context.getGL().getGL2();
		gl.glDeleteTextures(texture.length, texture, 0);
	}

	private static final String[] createSamplerProgram(boolean invert) {
		return new String[] {
				"uniform sampler1D texX;",
				"uniform sampler1D texY;",
				"uniform vec4 texSize;",
				"uniform vec4 texOffset;",
				"uniform vec4 gridSize;",
				"uniform vec2 anchor;",
				"uniform vec4 color;",		// ATIでgl_Colorを使うには頂点シェーダで明示的にglFrontColorを設定する必要がある。
				"",							// ここではそれをせず、uniform変数で色を渡している。
				"void main(void)",
				"{",
				"	vec2 gridCoord = gl_FragCoord.st - anchor;",
				"	gridCoord = gridCoord / gridSize.st + 0.5;",
				"	gridCoord -= floor(gridCoord);",
				"	gridCoord *= gridSize.st;",
				"",
				"	vec4 texCoord = (vec4(gridCoord, gridCoord) - texOffset) / texSize;",
				"",
				"	vec2 a1 = vec2(texture1D(texX, texCoord.x).a, texture1D(texY, texCoord.y).a);",
				"	vec2 a2 = vec2(texture1D(texX, texCoord.z).a, texture1D(texY, texCoord.w).a);",
				"	vec2 a = 1.0 - a1*(1.0-a2);",
				"",
	   invert ? "	gl_FragColor = color * a.x*a.y;"
			  : "	gl_FragColor = color * (1.0-a.x*a.y);",
				"}"
		};
	}

	@ShaderSource
	public static final String[] GRID_SAMPLER = createSamplerProgram(false);

	@ShaderSource
	public static final String[] INVERT_SAMPLER = createSamplerProgram(true);

}
