/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.generate;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GL2;
import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.BlendMode;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IBlendSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.RadialGradient2", category=Categories.GENERATE)
public class RadialGradient2 extends GradientBase {

	@Property
	private IAnimatableVec2d center;

	@Property(value="0", min="0", max="4000")
	private IAnimatableDouble radiusA;

	@Property(value="100", min="0", max="4000")
	private IAnimatableDouble radiusB;

	@Property("1,1,1")
	private IAnimatableColor colorA;

	@Property("0,0,0")
	private IAnimatableColor colorB;

	@Property("RGB")
	private IAnimatableEnum<ColorSpace> colorSpace;

	@Property("CLAMP_TO_EDGE")
	private IAnimatableEnum<RepeatMode> repeatMode;

	@Property(value="100", min="0", max="100")
	private IAnimatableDouble opacity;

	@Property
	private IAnimatableEnum<BlendMode> blendMode;


	private final IVideoRenderSupport support;

	private final IShaderProgram program;

	@Inject
	public RadialGradient2(
			IVideoEffectContext context, IBlendSupport blendSupport, IArrayPools arrayPools,
			IVideoRenderSupport support, IShaderRegistry shaders) {

		super(context, blendSupport, arrayPools);
		this.support = support;
		program = shaders.getProgram(RadialGradient2.class, "RADIAL_SAMPLER");
	}

	public IVideoBuffer doVideoEffect() {
		return doGradientEffect(context.value(blendMode), context.value(opacity)/100);
	}

	@Override
	protected IVideoBuffer createGradient(final VideoBounds bounds) {
		Resolution resolution = context.getVideoResolution();

		Vec2d center = resolution.scale(context.value(this.center));
		double radiusA = resolution.scale(context.value(this.radiusA));
		double radiusB = resolution.scale(context.value(this.radiusB));
		Color colorA = context.value(this.colorA);
		Color colorB = context.value(this.colorB);
		ColorSpace colorSpace = context.value(this.colorSpace);
		RepeatMode repeatMode = context.value(this.repeatMode);

		double innerRadius;
		double radiusDiff;
		if (radiusA <= radiusB) {
			innerRadius = radiusA;
			radiusDiff = radiusB - radiusA;
		} else {
			innerRadius = radiusB;
			radiusDiff = radiusA - radiusB;

			Color tmp = colorA;
			colorA = colorB;
			colorB = tmp;

			switch (colorSpace) {
				case HSL: colorSpace = ColorSpace.HSL_REVERSE_HUE; break;
				case HSL_REVERSE_HUE: colorSpace = ColorSpace.HSL; break;
			}
		}

		int texSize = Math.min(Math.max((int)Math.ceil(radiusDiff), 2), 4000);
		final int texture;
		switch (colorSpace) {
			case RGB:
				texture = createRGBGradientTexture(colorA, colorB, texSize, repeatMode);
				break;
			case HSL:
				texture = createHSLGradientTexture(colorA, colorB, false, texSize, repeatMode);
				break;
			case HSL_REVERSE_HUE:
				texture = createHSLGradientTexture(colorA, colorB, true, texSize, repeatMode);
				break;
			default:
				// never reached here
				throw new Error();
		}

		IVideoBuffer gradient = null;
		try {
			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture", 0));
			uniforms.add(new GLUniformData("center", 2, FloatBuffer.wrap(
					new float[] { (float)(center.x-bounds.x), (float)(center.y-bounds.y) })));
			uniforms.add(new GLUniformData("innerRadius", (float)innerRadius));
			uniforms.add(new GLUniformData("radiusDiff", (float)radiusDiff));

			Runnable operation = new Runnable() {
				public void run() {
					GL2 gl = context.getGL().getGL2();
					gl.glActiveTexture(GL2.GL_TEXTURE0);
					gl.glBindTexture(GL2.GL_TEXTURE_1D, texture);

					support.ortho2D(bounds);
					support.quad2D(bounds);
				}
			};

			gradient = context.createVideoBuffer(bounds);
			gradient.clear();

			support.useShaderProgram(program, uniforms, operation, GL2.GL_TEXTURE_BIT, gradient);

			IVideoBuffer result = gradient;
			gradient = null;
			return result;

		} finally {
			if (gradient != null) gradient.dispose();
			deleteTextures(texture);
		}
	}

	@ShaderSource
	public static final String[] RADIAL_SAMPLER = {
		"uniform sampler1D texture;",
		"uniform vec2 center;",
		"uniform float innerRadius;",
		"uniform float radiusDiff;",
		"",
		"void main(void)",
		"{",
		"	float d = distance(center, gl_FragCoord.st);",
		"	gl_FragColor = texture1D(texture, (d-innerRadius)/radiusDiff);",
		"}"
	};

}
