/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.keying;

import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

public abstract class BlueGreenScreenKeyBase {

	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram keyingProgram;

	private final IShaderProgram maskOnlyProgram;

	private final IShaderProgram smoothingLowProgram;

	private final IShaderProgram smoothingHighProgram;


	protected BlueGreenScreenKeyBase(
			IVideoEffectContext context, IVideoRenderSupport support, IShaderRegistry shaders, boolean blue) {

		this.context = context;
		this.support = support;

		if (blue) {
			keyingProgram = shaders.getProgram(KeyingShaders.class, "BLUE_SCREEN_KEY");
			maskOnlyProgram = shaders.getProgram(KeyingShaders.class, "BLUE_SCREEN_KEY_MASK_ONLY");
		} else {
			keyingProgram = shaders.getProgram(KeyingShaders.class, "GREEN_SCREEN_KEY");
			maskOnlyProgram = shaders.getProgram(KeyingShaders.class, "GREEN_SCREEN_KEY_MASK_ONLY");
		}

		smoothingLowProgram = shaders.getProgram(KeyingShaders.class, "SMOOTHING_LOW");
		smoothingHighProgram = shaders.getProgram(KeyingShaders.class, "SMOOTHING_HIGH");
	}

	protected IVideoBuffer doKeying(
			IAnimatableDouble threshold,
			IAnimatableDouble cutoff,
			IAnimatableBoolean maskOnly,
			IAnimatableEnum<Smoothing> smoothing) {

		IVideoBuffer input = context.doPreviousEffect();

		VideoBounds bounds = input.getBounds();
		if (bounds.isEmpty()) {
			return input;
		}

		double t = context.value(threshold) / 100;
		double c = context.value(cutoff) / 100;

		double t_minus_c = t - c;
		if (t_minus_c == 0) {
			if (t == 0) {
				t_minus_c = Double.POSITIVE_INFINITY;
			} else if (t == 1) {
				t = Double.POSITIVE_INFINITY;
			}
		}

		Smoothing sm = context.value(smoothing);
		boolean mask = context.value(maskOnly);

		IShaderProgram program1 = mask ? maskOnlyProgram : keyingProgram;
		IShaderProgram program2 = (sm == Smoothing.LOW) ? smoothingLowProgram
								: (sm == Smoothing.HIGH) ? smoothingHighProgram : null;

		IVideoBuffer buffer;
		try {
			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture", 0));
			uniforms.add(new GLUniformData("threshold", (float)t));
			uniforms.add(new GLUniformData("t_minus_c", (float)t_minus_c));

			buffer = support.useShaderProgram(program1, uniforms, null, input);

		} finally {
			input.dispose();
		}

		if (program2 == null) {
			return buffer;
		}

		try {
			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture", 0));

			return support.useShaderProgram(program2, uniforms, null, buffer);

		} finally {
			buffer.dispose();
		}
	}

}
