/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.transition;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableLayerReference;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.GradientWipe", category=Categories.TRANSITION)
public class GradientWipe {

	@ShaderSource
	public static final String[] GRADIENT_WIPE_NORMAL = createShaderSource(false, false);

	@ShaderSource
	public static final String[] GRADIENT_WIPE_NORMAL_TILE = createShaderSource(false, true);

	@ShaderSource
	public static final String[] GRADIENT_WIPE_INVERT = createShaderSource(true, false);

	@ShaderSource
	public static final String[] GRADIENT_WIPE_INVERT_TILE = createShaderSource(true, true);


	private static String[] createShaderSource(boolean invert, boolean tile) {
		return new String[] {
				"uniform sampler2DRect srcTex;",
				"uniform sampler2DRect grdTex;",
				"uniform float min;",
				"uniform float max;",

		 tile ? "uniform vec2 size;"
			  : "",

				"",
				"const vec3 yvec = vec3(0.299, 0.587, 0.114);",
				"",
				"void main(void)",
				"{",
				"	vec4 src = texture2DRect(srcTex, gl_TexCoord[0].st);",

				"	vec2 grdCoord = gl_TexCoord[1].st;",
		 tile ? "	vec4 grd = texture2DRect(grdTex, grdCoord - floor(grdCoord/size)*size);"
			  : "	vec4 grd = texture2DRect(grdTex, grdCoord);",
	   invert ? "	float y = 1.0 - dot(grd.rgb, yvec);"
			  : "	float y = dot(grd.rgb, yvec);",

				"	float a = clamp((y-min)/(max-min), 0.0, 1.0);",
				"	gl_FragColor = src*a;",
				"}"
		};
	}


	public enum GradientPlacement { TILE, CENTER, STRETCH }


	@Property(min="0", max="100")
	private IAnimatableDouble transitionCompletion;

	@Property(min="0", max="100")
	private IAnimatableDouble transitionSoftness;

	@Property
	private IAnimatableLayerReference gradientLayer;

	@Property("STRETCH")
	private IAnimatableEnum<GradientPlacement> gradientPlacement;

	@Property
	private IAnimatableBoolean invertGradient;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram normalProgram;

	private final IShaderProgram normalTileProgram;

	private final IShaderProgram invertProgram;

	private final IShaderProgram invertTileProgram;

	@Inject
	public GradientWipe(IVideoEffectContext context, IVideoRenderSupport support, IShaderRegistry shaders) {
		this.context = context;
		this.support = support;
		this.normalProgram = shaders.getProgram(GradientWipe.class, "GRADIENT_WIPE_NORMAL");
		this.normalTileProgram = shaders.getProgram(GradientWipe.class, "GRADIENT_WIPE_NORMAL_TILE");
		this.invertProgram = shaders.getProgram(GradientWipe.class, "GRADIENT_WIPE_INVERT");
		this.invertTileProgram = shaders.getProgram(GradientWipe.class, "GRADIENT_WIPE_INVERT_TILE");
	}

	public IVideoBuffer doVideoEffect() {
		double completion = context.value(transitionCompletion)/100;
		if (completion == 1) {
			IVideoBuffer buffer = context.createVideoBuffer(context.getPreviousBounds());
			buffer.clear();
			return buffer;
		}

		IVideoBuffer input = context.doPreviousEffect();

		if (completion == 0) {
			return input;
		}

		final VideoBounds bounds = input.getBounds();
		if (bounds.isEmpty()) {
			return input;
		}


		Set<IVideoBuffer> tmpBuffers = new HashSet<IVideoBuffer>();
		tmpBuffers.add(input);

		try {
			double softness = context.value(transitionSoftness)/100;
			GradientPlacement placement = context.value(gradientPlacement);
			boolean tile = (placement == GradientPlacement.TILE);
			boolean invert = context.value(invertGradient);
	
			double max = (1 + softness) * completion;
			double min = max - softness;

			IVideoBuffer gradient = context.getLayerVideoFrame(gradientLayer);
			if (gradient == null) {
				gradient = context.createVideoBuffer(new VideoBounds(0, 0));
				gradient.clear();
			}
			tmpBuffers.add(gradient);

			VideoBounds gradBounds = gradient.getBounds();


			final double[][][] texCoords = new double[2][][];

			int w = bounds.width;
			int h = bounds.height;
			texCoords[0] = new double[][] { {0, 0}, {w, 0}, {w, h}, {0, h} };

			switch (placement) {
				case CENTER: {
					float s = (gradBounds.width - bounds.width) / 2f;
					float t = (gradBounds.height - bounds.height) / 2f;
					texCoords[1] = new double[][] { {s, t}, {s+w, t}, {s+w, t+h}, {s, t+h} };
					break;
				}

				case STRETCH: {
					int gw = gradBounds.width;
					int gh = gradBounds.height;
					texCoords[1] = new double[][] { {0, 0}, {gw, 0}, {gw, gh}, {0, gh} };
					break;
				}

				default:
					texCoords[1] = new double[][] { {0, 0}, {w, 0}, {w, h}, {0, h} };
					break;
			}

			IShaderProgram program = tile ? (invert ? invertTileProgram : normalTileProgram)
										  : (invert ? invertProgram : normalProgram);

			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("srcTex", 0));
			uniforms.add(new GLUniformData("grdTex", 1));
			uniforms.add(new GLUniformData("min", (float)min));
			uniforms.add(new GLUniformData("max", (float)max));
			if (tile) {
				uniforms.add(new GLUniformData("size", 2, FloatBuffer.wrap(
						new float[] { gradBounds.width, gradBounds.height })));
			}

			Runnable operation = new Runnable() {
				public void run() {
					support.ortho2D(bounds);
					support.quad2D(bounds, texCoords);
				}
			};

			if (placement == GradientPlacement.STRETCH) {
				// 終わったら削除してしまうので、元に戻す必要はない。
				gradient.setTextureFilter(TextureFilter.LINEAR);
			}

			return support.useShaderProgram(program, uniforms, operation, 0, null, input, gradient);

		} finally {
			for (IVideoBuffer vb : tmpBuffers) {
				vb.dispose();
			}
		}
	}

}
