/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.examples.effect;

import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

/*
 * ビデオエフェクトのサンプル。
 * 全ピクセルを一様に反転するだけの単純なエフェクトです。
 */

/* 
 * エフェクトクラスには Effect アノテーションを付けます。
 * 
 * id には、他のエフェクトと重複しない一意の値を指定してください。
 * このエフェクトが属するカテゴリを category に指定してください。このサンプルでは “色調補正” カテゴリを指定しています。
 */
@Effect(id="ch.kuramo.javie.examples.effect.InvertColor",
		category=Categories.COLOR_CORRECTION)
public class InvertColor {

	/*
	 * このエフェクトのシェーダです。ShaderSource アノテーションを付けます。
	 * 
	 * このサンプルではシェーダのソースコードを直接記述しています。
	 * シェーダのソースコードをリソースファイルに記述する場合は次のようにします。
	 * 
	 *   @ShaderSource
	 *   public static final String INVERT_COLOR = "invert_color.frag";
	 * 
	 */
	@ShaderSource
	public static final String[] INVERT_COLOR = {
		"uniform sampler2DRect source;",
		"uniform float blend;",
		"",
		"void main(void)",
		"{",
		"	vec4 prmult = texture2DRect(source, gl_TexCoord[0].st);",
		"	if (prmult.a != 0.0) {",
		"",
		"",		// ビデオデータは乗算済みアルファで保持されているので、必要に応じて除算してください。
		"		vec3 unmult = prmult.rgb/prmult.a;",
		"",
		"",		// 反転
		"		unmult = 1.0 - unmult;",
		"",
		"",		// 乗算済みに戻します。
		"		vec4 prmult2 = vec4(unmult, 1.0) * prmult.a;",
		"",
		"",		// 元の画像とブレンド
		"		gl_FragColor = prmult2*(1.0-blend) + prmult*blend;",
		"",
		"	} else {",
		"		gl_FragColor = vec4(0.0);",
		"	}",
		"}"
	};


	/*
	 * プロパティには Property アノテーションを付けます。
	 * value, min, max にはそれぞれ、初期値、最小値、最大値を指定します。
	 */
	@Property(value="0", min="0", max="100")
	private IAnimatableDouble blendWithOriginal;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram program;

	/*
	 * コンストラクタに Inject アノテーションを付けると、サービスのインスタンスを受け取ることができます。
	 * 各サービスの詳細は javadoc を参照してください。
	 */
	@Inject
	public InvertColor(IVideoEffectContext context, IVideoRenderSupport support, IShaderRegistry shaders) {
		this.context = context;
		this.support = support;

		// シェーダのソースコードに対応する IShaderProgram オブジェクトを取得します。
		program = shaders.getProgram(InvertColor.class, "INVERT_COLOR");
	}

	/*
	 * レイヤーのサイズを変更するビデオエフェクトには、getVideoBounds という名前で
	 * 戻り値が VideoBounds の public なメソッドを実装する必要があります。
	 * このサンプルはレイヤーサイズを変更しないので不要です。
	 * 
	 *   public VideoBounds getVideoBounds() {
	 *   	VideoBounds bounds = context.getPreviousBounds();
	 *   
	 *   	// レイヤーのサイズを上下左右10ピクセルずつ拡張する。
	 *   	return new VideoBounds(bounds.x-10, bounds.y-10, bounds.width+20, bounds.height+20);
	 *   }
	 * 
	 * getVideoBounds メソッドを実装する場合、その戻り値は doVideoEffect メソッドが返す
	 * IVideoBuffer の getBounds メソッドの値と一致している必要があります。
	 */

	/*
	 * ビデオエフェクトクラスには、doVideoEffect という名前で
	 * 戻り値が IVideoBuffer の public なメソッドを実装する必要があります。
	 */
	public IVideoBuffer doVideoEffect() {
		// ひとつ前のエフェクトまで実行し、結果のビデオデータを受け取ります。
		IVideoBuffer source = context.doPreviousEffect();

		if (source.getBounds().isEmpty()) {
			return source;
		}

		try {
			// プロパティの現在時刻における値を取得します。
			double blend = context.value(blendWithOriginal) / 100;


			// サイズに関連するプロパティの場合はレンダリング解像度に従ってスケールする必要があります。
			// (blendWithOriginal はサイズとは無関係のプロパティなので、実際にはスケールしてはいけない。
			// 以下は Resolution の例示のためのものです。)
			//
			//   Resolution resolution = context.getVideoResolution();
			//   blend = resolution.scale(blend);


			// シェーダの Uniform 変数を準備します。
			// ここで使うシェーダには "source", "blend" という名前の Uniform 変数があるので、それらの値を準備します。
			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();

			// テクスチャ番号 0 のテクスチャをシェーダで使用するために、Uniform 変数 "source" を 0 に設定します。
			// (IVideoRenderSupport.useShaderProgram の最後の引数は可変長引数で、テクスチャ番号 0 から順番にバインドされます)
			uniforms.add(new GLUniformData("source", 0));

			// Uniform 変数 "blend" を設定します。
			uniforms.add(new GLUniformData("blend", (float)blend));

			// シェーダを使用します。
			// 戻り値の IVideoBuffer オブジェクトをこのエフェクトの処理結果として返します。
			return support.useShaderProgram(program, uniforms, null, source);

		} finally {
			// 処理に失敗した場合も source が必ず破棄されるよう、try/finally で囲って dispose() します。
			source.dispose();
		}
	}

	/*
	 * オーディオフェクトクラスには、doAudioEffect という名前で
	 * 戻り値が IAudioBuffer の public なメソッドを実装する必要があります。
	 * 
	 *   public IAudioBuffer doAudioEffect() {
	 *   	...
	 *   }
	 */

}
