/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.api;

/**
 * <p><code>IVideoBuffer</code> はビデオ用バッファです。
 * {@link ch.kuramo.javie.api.services.IVideoEffectContext#doPreviousEffect()} から返される他、
 * {@link ch.kuramo.javie.api.services.IVideoRenderSupport#createVideoBuffer(VideoBounds)}
 * 等で作成することができます。</p>
 * 
 * <p><code>IVideoBuffer</code> オブジェクトは、
 * フッテージ、レイヤー、コンポジション等におけるビデオデータの1フレーム分を
 * OpenGLテクスチャまたはJavaVM上の配列いずれかの状態で保持します。
 * OpenGLを用いて処理を行うエフェクトはテクスチャの状態でビデオデータを扱います。
 * CPU側で処理を行う必要がある場合は {@link #getArray()}
 * メソッドを使って配列としてデータを取り出します。
 * 一度 <code>getArray()</code> が呼ばれると、その <code>IVideoBuffer</code>
 * オブジェクトが保持するビデオデータは {@link #getTexture()}
 * が呼ばれるまで配列の状態で保持され続けます。
 */
public interface IVideoBuffer {

	/**
	 * このビデオ用バッファを破棄します。
	 */
	void dispose();

	/**
	 * このビデオ用バッファがテクスチャの状態でデータを保持している場合は true を返します。
	 * 
	 * @return	このビデオ用バッファがテクスチャの状態でデータを保持している場合は true、
	 * 			そうでない場合は false
	 */
	boolean isTexture();

	/**
	 * このビデオ用バッファが配列の状態でデータを保持している場合は true を返します。
	 * 
	 * @return	このビデオ用バッファが配列の状態でデータを保持している場合は true、
	 * 			そうでない場合は false
	 */
	boolean isArray();

	/**
	 * このビデオ用バッファが保持するテクスチャを返します。
	 * このビデオ用バッファが配列の状態でデータを保持している場合、
	 * 配列からテクスチャへのデータ転送が発生します。
	 * 
	 * @return このビデオ用バッファが保持するデータのテクスチャ
	 */
	int getTexture();

	/**
	 * このビデオ用バッファが保持する配列を返します。
	 * このビデオ用バッファがテクスチャの状態でデータを保持している場合、
	 * テクスチャから配列へのデータ転送が発生します。
	 * 
	 * @return このビデオ用バッファが保持するデータの配列
	 */
	Object getArray();

	/**
	 * このビデオ用バッファの色深度を返します。
	 * 
	 * @return このビデオ用バッファの色深度を表す <code>ColorMode</code> オブジェクト
	 */
	ColorMode getColorMode();

	/**
	 * このビデオ用バッファの位置およびサイズを返します。
	 * 
	 * @return このビデオ用バッファの位置およびサイズを表す <code>VideoBounds</code> オブジェクト
	 */
	VideoBounds getBounds();

	/**
	 * このビデオ用バッファが保持するテクスチャの補間タイプを返します。
	 * このビデオ用バッファが配列の状態でデータを保持している場合、
	 * 配列からテクスチャへのデータ転送が発生します。
	 * 
	 * @return このビデオ用バッファが保持するテクスチャの補間タイプ
	 */
	TextureFilter getTextureFilter();

	/**
	 * このビデオ用バッファが保持するテクスチャの補間タイプを、引数 <code>filter</code> で指定したタイプに設定します。
	 * このビデオ用バッファが配列の状態でデータを保持している場合、
	 * 配列からテクスチャへのデータ転送が発生します。
	 * 
	 * @param filter このビデオ用バッファが保持するテクスチャに設定する補間モード
	 */
	void setTextureFilter(TextureFilter filter);

	/**
	 * このビデオ用バッファが保持するテクスチャのラップモードを返します。
	 * このビデオ用バッファが配列の状態でデータを保持している場合、
	 * 配列からテクスチャへのデータ転送が発生します。
	 * 
	 * @return このビデオ用バッファが保持するテクスチャのラップモード
	 */
	TextureWrapMode getTextureWrapMode();

	/**
	 * このビデオ用バッファが保持するテクスチャのラップモードを、引数 <code>wrapMode</code> で指定したモードに設定します。
	 * このビデオ用バッファが配列の状態でデータを保持している場合、
	 * 配列からテクスチャへのデータ転送が発生します。
	 * 
	 * @param wrapMode このビデオ用バッファが保持するテクスチャに設定するラップモード
	 */
	void setTextureWrapMode(TextureWrapMode wrapMode);

	/**
	 * このビデオ用バッファを、無色透明 (r,g,b,a=0,0,0,0) でクリアします。
	 */
	void clear();

	/**
	 * <b>注意：このメソッドの実装は不完全です。</b>
	 * このビデオ用バッファを、引数 <code>color</code> で指定した色でクリアします。
	 * 
	 * 現在の実装では、このビデオ用バッファが配列の状態でデータを保持していて
	 * <code>color</code> が無色透明 (r,g,b,a=0,0,0,0) 以外の場合、
	 * <code>UnsupportedOperationException</code> をスローします。 
	 */
	void clear(Color color);

	/**
	 * 引数 <code>srcArray</code> のデータをこのビデオ用バッファが保持するテクスチャへ転送し、上書きします。
	 * このビデオ用バッファが配列の状態でデータを保持している場合、配列を破棄しテクスチャを作成した後、データを転送します。
	 * <code>srcArray</code> の実際の型は、<code>byte[]</code>, <code>short[]</code>
	 * はたは <code>float[]</code> のいずれかで、引数 <code>srcColorMode</code>
	 * が表す型と一致している必要があります。
	 * 
	 * @param srcArray このビデオ用バッファが保持するテクスチャへ転送するデータの配列
	 * @param srcColorMode <code>srcArray</code> のデータの色深度
	 */
	void copyToTexture(Object srcArray, ColorMode srcColorMode);

	/**
	 * テクスチャの補間フィルタを表す列挙型
	 */
	public enum TextureFilter {

		/**
		 * ニアレスト (補間なし)
		 */
		NEAREST,

		/**
		 * リニア補間
		 */
		LINEAR,

		/**
		 * ミップマップ。異方性フィルタが使用可能な場合は異方性フィルタも使用します。
		 * @since 0.5.0
		 */
		MIPMAP
	}

	/**
	 * テクスチャのラップモードを表す列挙型
	 */
	public enum TextureWrapMode {

		/**
		 * テクスチャ座標が範囲外の場合、境界色が使用されます。
		 * {@link IVideoBuffer} が保持するテクスチャの境界色は通常(OpenGLのAPIを直接使用して変更しない限り)、
		 * 無色透明 (r,g,b,a=0,0,0,0) です。
		 */
		CLAMP_TO_BORDER,

		/**
		 * テクスチャ座標が範囲外の場合、テクスチャの端のテクセルが使用されます。
		 */
		CLAMP_TO_EDGE,

		/**
		 * テクスチャが繰り返されます。
		 * @since 0.5.0
		 */
		REPEAT

	}

}
