/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.api.services;

import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IObjectArray;

/**
 * <p>配列を再利用するための配列プールサービスです。
 * 大きな配列を使用する場合、<code>new</code>演算子を用いて配列を確保すると処理速度への影響が大きいため、
 * できる限りこのサービスを利用してください。</p>
 * 
 * <p>このサービスの各メソッドが返す値は、生の配列ではなく {@link IArray} オブジェクトです。
 * 生の配列を得るには {@link IArray#getArray()} を使用します。
 * 配列の要素はクリアされていないため、クリアする必要がある場合は {@link IArray#clear()}
 * を使用してください。ただし、{@link #getObjectArray(int)} が返すオブジェクト配列は<code>null</code>でクリアされています。
 * 配列プールの実装の性質上、このサービスの各メソッドの引数で指定した長さよりも生の配列は長くなることがあります。
 * したがって「生の配列.length」で取得した長さは使用せずに、{@link IArray#getLength()} による長さを使用してください。
 * 配列を使い終わったら、必ず {@link IArray#release()} を呼び出して配列プールに返却してください。</p>
 * 
 * 使用例：
 * <blockquote><pre>
 *  &#64;Effect
 *  public class BigArrayEffect {
 *  
 *  	private final IArrayPools arrayPools;
 *  
 *  	&#64;Inject
 *  	public BigArrayEffect(IArrayPools arrayPools) {
 *  		this.arrayPools = arrayPools;
 *  	}
 *  
 *  	public IVideoBuffer doVideoEffect() {
 *  
 *  		<b>// 配列を取得する
 *  		IArray&lt;byte[]&gt; array = arrayPools.getByteArray(1234000);
 *  
 *  		// 配列の要素をクリアする
 *  		array.clear();
 *  
 *  		// 生の配列を取得する
 *  		byte[] bytes = array.getArray();
 *  
 *  		// 配列の長さを取得する（これは誤り：1234000より長い）
 *  		int len = bytes.length;
 *  
 *  		// 配列の長さを取得する（こちらが正しい：1234000と一致）
 *  		int len = array.getLength();</b>
 *  
 *  
 *  		// 配列を用いた処理
 *  		for (int i = 0; i < len; ++i) {
 *  			...
 *  			bytes[i] = ...
 *  		}
 *   
 *  
 *  		<b>// 使い終わったらプールへ返却する
 *  		array.release();</b>
 *  
 *  	}
 *  }
 * </pre></blockquote>
 * 
 * @see IArray
 */
public interface IArrayPools {

	/**
	 * 配列プールから<code>byte</code>配列を取得します。
	 * 
	 * @param	length 配列の長さ
	 * @return	配列をラップした {@link IArray} オブジェクト
	 */
	IArray<byte[]> getByteArray(int length);

	/**
	 * 配列プールから<code>short</code>配列を取得します。
	 * 
	 * @param	length 配列の長さ
	 * @return	配列をラップした {@link IArray} オブジェクト
	 */
	IArray<short[]> getShortArray(int length);

	/**
	 * 配列プールから<code>int</code>配列を取得します。
	 * 
	 * @param	length 配列の長さ
	 * @return	配列をラップした {@link IArray} オブジェクト
	 */
	IArray<int[]> getIntArray(int length);

	/**
	 * 配列プールから<code>float</code>配列を取得します。
	 * 
	 * @param	length 配列の長さ
	 * @return	配列をラップした {@link IArray} オブジェクト
	 */
	IArray<float[]> getFloatArray(int length);

	/**
	 * 配列プールから<code>double</code>配列を取得します。
	 * 
	 * @param	length 配列の長さ
	 * @return	配列をラップした {@link IArray} オブジェクト
	 */
	IArray<double[]> getDoubleArray(int length);

	/**
	 * 配列プールからオブジェクト配列を取得します。
	 * 
	 * @param	length 配列の長さ
	 * @return	配列をラップした {@link IObjectArray} オブジェクト
	 */
	<T> IObjectArray<T> getObjectArray(int length);

}
