/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.api.services;

import ch.kuramo.javie.api.BlendMode;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;

/**
 * <p>
 * 描画モードを用いた合成処理を支援するサービスです。
 * このサービスはシングルトンではないため、インジェクトするごとに新しいインスタンスが生成されます。
 * </p>
 * <p>
 * {@link #replace(BlendMode, IShaderProgram)} メソッドや {@link #setProgramsClass(Class)} メソッドによって
 * 描画モードの実装を変更することができます。例えば、フラクタルノイズエフェクトは色相と彩度の実装を独自のものに置き換えています。
 * </p>
 */
public interface IBlendSupport {

	/**
	 * ソースバッファがストレートアルファである場合の、描画モードの実装を取得します。
	 * 
	 * @return 描画モードの実装を持つクラスオブジェクト
	 */
	Class<?> getUnmultSrcClass();

	/**
	 * 描画モードの実装を <code>programsClass</code> のクラスが持つものに全て置き換えます。
	 * <code>programsClass</code> のクラス内で、<code>BlendMode.DANCING_DISSOLVE</code>
	 * を除くすべての描画モードに対してシェーダプログラムが定義されている必要があります。
	 * 
	 * @param programsClass 描画モードの実装をもつクラスオブジェクト
	 */
	void setProgramsClass(Class<?> programsClass);

	/**
	 * <code>blendMode</code> で指定した描画モードの実装を、
	 * <code>program</code> のシェーダプログラムに置き換えます。
	 * 
	 * @param blendMode 実装を置き換える描画モード
	 * @param program 描画モードのシェーダプログラム
	 */
	void replace(BlendMode blendMode, IShaderProgram program);

	/**
	 * <p>
	 * <code>blendMode</code> で指定した描画モードを用いて <code>dstIn</code> の上に
	 * <code>src</code> を合成し、<code>dstOut</code> に結果を出力します。
	 * <code>opacity</code> で <code>src</code> の不透明度を指定することができます。
	 * </p>
	 * <p>
	 * <code>dstOut</code> に <code>null</code> を指定した場合、<code>dstIn</code>
	 * と同じ位置とサイズで出力バッファが作成され、戻り値に返されます。
	 * </p>
	 * <p>
	 * <code>blendMode</code> が <code>BlendMode.DISSOLVE</code> または
	 * <code>BlendMode.DANCING_DISSOLVE</code> の場合、実行中のエフェクトのコンテキストを
	 * 引数 <code>context</code> に与える必要があります。これは、コンテキストから乱数のシードを決定するためです。
	 * シードを明示的に指定する場合は {@link #dissolve(IVideoBuffer, IVideoBuffer, IVideoBuffer, double, double)}
	 * メソッドを使用してください。
	 * </p>
	 * 
	 * @param src 上になる画像
	 * @param dstIn 下になる画像
	 * @param dstOut 合成の結果を出力するバッファ
	 * @param blendMode 合成に用いる描画モード
	 * @param opacity <code>src</code> の不透明度
	 * @param context 実行中のエフェクトのコンテキスト
	 * @return <code>dstOut</code> が <code>null</code> の場合は新たに作成されたバッファ、そうでない場合は <code>dstOut</code>
	 */
	IVideoBuffer blend(
			IVideoBuffer src, IVideoBuffer dstIn, IVideoBuffer dstOut,
			BlendMode blendMode, double opacity, IVideoEffectContext context);

	IVideoBuffer blend(
			IVideoBuffer src, IVideoBuffer dstIn, IVideoBuffer dstOut,
			BlendMode blendMode, double opacity);

	IVideoBuffer dissolve(
			IVideoBuffer src, IVideoBuffer dstIn, IVideoBuffer dstOut,
			double opacity, double seed);

}
