/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.api.services;

import javax.media.opengl.GL;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IAnimatableLayerReference;
import ch.kuramo.javie.api.IAnimatableValue;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;

/**
 * <p><code>IEffectContext</code> サービスは、エフェクトの実行に必要な情報
 * (エフェクトコンテキスト) とそれに関連したいくつかの処理を提供します。
 * エフェクトコンテキストには、現在時刻、レンダリング解像度 ({@link Resolution})、色深度
 * ({@link ColorMode})、{@link AudioMode}、オーディオのサンプルフレーム数が含まれます。
 */
public interface IEffectContext {

	/**
	 * エフェクトコンテキストの現在時刻を返します。
	 * 
	 * @return 現在時刻
	 */
	Time getTime();

	/**
	 * エフェクトコンテキストの現在時刻を変更します。
	 * 現在時刻とは異なる時刻で {@link IVideoEffectContext#doPreviousEffect()} ,
	 * {@link IAudioEffectContext#doPreviousEffect()} ,
	 * {@link #getLayerVideoFrame(IAnimatableLayerReference)} ,
	 * {@link #getLayerAudioChunk(IAnimatableLayerReference)} を実行する必要がある場合、
	 * これらのメソッドよりも前に <code>setTime(Time)</code> を呼び出して現在時刻を変更します。
	 * また、現在時刻の変更は {@link #value(IAnimatableValue)} や
	 * {@link IAudioEffectContext#values(IAnimatableValue)} などの結果にも影響します。
	 * 
	 * @param time 変更後の時刻
	 */
	void setTime(Time time);

	/**
	 * エフェクトコンテキストのレンダリング解像度を返します。
	 * ビデオエフェクトはこのレンダリング解像度に従って適切な処理を行う必要があります。
	 * 
	 * @return レンダリング解像度
	 */
	Resolution getVideoResolution();

	/**
	 * エフェクトコンテキストの色深度を返します。
	 * 
	 * @return 色深度
	 */
	ColorMode getColorMode();

	/**
	 * エフェクトコンテキストの <code>AudioMode</code> を返します。
	 * 
	 * @return <code>AudioMode</code> オブジェクト
	 */
	AudioMode getAudioMode();

	/**
	 * エフェクトコンテキストのサンプルフレーム数を返します。
	 * オーディオエフェクトはこのサンプルフレーム数のオーディオデータを一度に返す必要があります。
	 * 
	 * @return サンプルフレーム数
	 */
	int getAudioFrameCount();

	/**
	 * <p>エフェクトコンテキストのサンプルフレーム数を変更します。
	 * {@link IAudioEffectContext#doPreviousEffect()} や {@link #getLayerAudioChunk(IAnimatableLayerReference)}
	 * を呼び出す前に <code>setAudioFrameCount(int)</code> を呼び出してサンプルフレーム数を変更することができます。</p>
	 * <p>サンプルフレーム数を変更した場合でも、オーディオエフェクトは元のサンプルフレーム数のオーディオデータを返す必要があることに注意してください。</p>
	 * 
	 * @param count 変更後のサンプルフレーム数
	 */
	void setAudioFrameCount(int count);

	/**
	 * エフェクトコンテキストに関連付けられているOpenGL (JOGL) の <code>GL</code> オブジェクトを返します。
	 * OpenGLを使用するエフェクトはこの <code>GL</code> オブジェクトを使用してください。
	 * 
	 * @return エフェクトコンテキストに関連付けられている <code>GL</code> オブジェクト
	 */
	GL getGL();

	/**
	 * エフェクトコンテキストに関連付けられているOpenGL (JOGL) の <code>GLU</code> オブジェクトを返します。
	 * OpenGLを使用するエフェクトはこの <code>GLU</code> オブジェクトを使用してください。
	 *  
	 * @return エフェクトコンテキストに関連付けられている <code>GLU</code> オブジェクト
	 */
	GLU getGLU();

	/**
	 * 引数 <code>bounds</code> の位置とサイズ、エフェクトコンテキストの色深度でビデオ用バッファを作成します。
	 * これは {@link IVideoRenderSupport#createVideoBuffer(VideoBounds)} と同じです。
	 * 
	 * @param bounds 作成するビデオ用バッファの位置とサイズ
	 * @return 作成されたビデオ用バッファ
	 */
	IVideoBuffer createVideoBuffer(VideoBounds bounds);

	/**
	 * エフェクトコンテキストの <code>AudioMode</code> とサンプルフレーム数でオーディオ用バッファを作成します。
	 * 
	 * @return 作成されたオーディオ用バッファ
	 */
	IAudioBuffer createAudioBuffer();

	/**
	 * 引数 avalue の現在時刻における値を返します。
	 * 
	 * @param avalue {@link ch.kuramo.javie.api.IAnimatableDouble} などのアニメーション可能プロパティ
	 * @return 引数 avalue の現在時刻における値
	 */
	<V> V value(IAnimatableValue<V> avalue);

	/**
	 * 引数 <code>layerRef</code> により参照されるレイヤーの、現在時刻におけるビデオデータを取得します。
	 * <code>layerRef</code> がレイヤーを参照していない場合は null を返します。
	 * 
	 * @param layerRef ビデオデータを取得するレイヤー
	 * @return	取得したビデオデータの <code>IVideoBuffer</code> オブジェクト。
	 * 			引数 <code>layerRef</code> がレイヤーを参照していない場合は null
	 */
	IVideoBuffer getLayerVideoFrame(IAnimatableLayerReference layerRef);

	/**
	 * 引数 <code>layerRef</code> により参照されるレイヤーの、現在時刻におけるオーディオデータを取得します。
	 * <code>layerRef</code> がレイヤーを参照していない場合は null を返します。
	 * 
	 * @param layerRef オーディオデータを取得するレイヤー
	 * @return	取得したオーディオデータの <code>IAudioBuffer</code> オブジェクト。
	 * 			引数 <code>layerRef</code> がレイヤーを参照していない場合は null
	 */
	IAudioBuffer getLayerAudioChunk(IAnimatableLayerReference layerRef);

	/**
	 * 実行中のエフェクトの名前を返します。
	 * これはエフェクトの種類の名前ではなく、エフェクトのインスタンスに付けられた名前です。
	 * エフェクトのインスタンスに付けられた名前はユーザーが変更できます。
	 * 
	 * @return 実行中のエフェクトの名前
	 */
	String getEffectName();

}
