/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import ch.kuramo.javie.core.Util;

public class EnumLabels {

	private static final Map<Class<? extends Enum<?>>, List<String>> labelsMap = Util.newMap();

	public static <E extends Enum<E>> String getLabel(E value) {
		@SuppressWarnings("unchecked")
		Class<E> enumType = (Class<E>) value.getClass();

		return getLabels(enumType).get(value.ordinal());
	}

	public static synchronized <E extends Enum<E>> List<String> getLabels(Class<E> enumType) {
		List<String> list = labelsMap.get(enumType);
		if (list != null) {
			return list;
		}

		E[] values = enumType.getEnumConstants();
		String[] labels = new String[values.length];
		for (int i = 0; i < values.length; ++i) {
			labels[i] = '!' + values[i].name() + '!';
		}

		String baseName = enumType.getPackage().getName() + ".messages";
		Locale locale = Locale.getDefault();
		ResourceBundle bundle = null;
		try {
			// enumTypeのクラスローダから取得する。
			bundle = ResourceBundle.getBundle(baseName, locale, enumType.getClassLoader());
		} catch (MissingResourceException e1) {
			try {
				// 現在のクラスローダから取得する。
				bundle = ResourceBundle.getBundle(baseName, locale);
			} catch (MissingResourceException e2) {
				// ignore
			}
		}

		if (bundle != null) {
			for (int i = 0; i < values.length; ++i) {
				StringBuilder sb = new StringBuilder();
				for (Class<?> c = enumType; c != null; c = c.getEnclosingClass()) {
					sb.insert(0, c.getSimpleName());
					try {
						labels[i] = bundle.getString(String.format(
								"%s.%s", sb.toString(), values[i].name()));
					} catch (MissingResourceException e) {
						// ignore
					}
					sb.insert(0, '$');
				}
			}
		}

		list = Collections.unmodifiableList(Arrays.asList(labels));
		labelsMap.put(enumType, list);

		return list;
	}

}
