/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaItemLayer;
import ch.kuramo.javie.core.Project;

public abstract class LayerSwitchOperation extends ProjectOperation {

	private final String _compId;

	private final String _layerId;


	public LayerSwitchOperation(ProjectManager projectManager, Layer layer, String label) {
		super(projectManager, label);

		LayerComposition comp = projectManager.checkLayer(layer);
		_compId = comp.getId();
		_layerId = layer.getId();
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		setSwitchValue(false, getLayer(project), pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		setSwitchValue(true, getLayer(project), pm);
		return Status.OK_STATUS;
	}

	protected abstract void setSwitchValue(boolean undo, Layer layer, ProjectManager pm);

	protected <L extends Layer> L getLayer(Project project) {
		LayerComposition comp = project.getComposition(_compId);
		@SuppressWarnings("unchecked")
		L layer = (L) comp.getLayer(_layerId);
		return layer;
	}


	public static class Video extends LayerSwitchOperation {

		private final boolean _oldValue;

		public Video(ProjectManager projectManager, Layer layer) {
			super(projectManager, layer, "ビデオスイッチの変更");
			_oldValue = LayerNature.isVideoEnabled(layer);
		}

		@Override
		protected void setSwitchValue(boolean undo, Layer layer, ProjectManager pm) {
			LayerNature.setVideoEnabled(layer, undo ? _oldValue : !_oldValue);

			if (pm != null) {
				pm.fireLayerPropertyChange(layer, "videoEnabled");
			}
		}
	}

	public static class Audio extends LayerSwitchOperation {

		private final boolean _oldValue;

		public Audio(ProjectManager projectManager, Layer layer) {
			super(projectManager, layer, "オーディオスイッチの変更");
			_oldValue = LayerNature.isAudioEnabled(layer);
		}

		@Override
		protected void setSwitchValue(boolean undo, Layer layer, ProjectManager pm) {
			LayerNature.setAudioEnabled(layer, undo ? _oldValue : !_oldValue);

			if (pm != null) {
				pm.fireLayerPropertyChange(layer, "audioEnabled");
			}
		}
	}

	public static class Shy extends LayerSwitchOperation {

		private final boolean _oldValue;

		public Shy(ProjectManager projectManager, Layer layer) {
			super(projectManager, layer, "シャイスイッチの変更");
			_oldValue = layer.isShy();
		}

		@Override
		protected void setSwitchValue(boolean undo, Layer layer, ProjectManager pm) {
			layer.setShy(undo ? _oldValue : !_oldValue);

			if (pm != null) {
				pm.fireLayerPropertyChange(layer, "shy");
			}
		}
	}

	public static class Effects extends LayerSwitchOperation {

		private final boolean _oldValue;

		public Effects(ProjectManager projectManager, EffectableLayer layer) {
			super(projectManager, layer, "エフェクトスイッチの変更");
			_oldValue = layer.isEffectsEnabled();
		}

		@Override
		protected void setSwitchValue(boolean undo, Layer layer, ProjectManager pm) {
			((EffectableLayer) layer).setEffectsEnabled(undo ? _oldValue : !_oldValue);

			if (pm != null) {
				pm.fireLayerPropertyChange(layer, "effectsEnabled");
			}
		}
	}

	public static class CTCR extends LayerSwitchOperation {

		private final boolean _oldValue;

		public CTCR(ProjectManager projectManager, Layer layer) {
			super(projectManager, layer, label(layer));
			_oldValue = LayerNature.isCTCR(layer);
		}

		private static String label(Layer layer) {
			if (layer instanceof MediaItemLayer && ((MediaItemLayer) layer).isPrecompositionLayer()) {
				return "コラップストランスフォームスイッチの変更";
			} else {
				return "連続ラスタライズスイッチの変更";
			}
		}

		@Override
		protected void setSwitchValue(boolean undo, Layer layer, ProjectManager pm) {
			LayerNature.setCTCR(layer, undo ? _oldValue : !_oldValue);

			if (pm != null) {
				pm.fireLayerPropertyChange(layer, "CTCR");
			}
		}
	}

	public static class Quality extends LayerSwitchOperation {

		private final ch.kuramo.javie.core.Quality _newValue;
		private final ch.kuramo.javie.core.Quality _oldValue;

		public Quality(ProjectManager projectManager, Layer layer, ch.kuramo.javie.core.Quality newValue) {
			super(projectManager, layer, "画質スイッチの変更");
			_newValue = newValue;
			_oldValue = LayerNature.getQuality(layer);
			_noEffect = (_newValue == _oldValue);
		}

		@Override
		protected void setSwitchValue(boolean undo, Layer layer, ProjectManager pm) {
			LayerNature.setQuality(layer, undo ? _oldValue : _newValue);

			if (pm != null) {
				pm.fireLayerPropertyChange(layer, "quality");
			}
		}
	}

	public static class FrameBlend extends LayerSwitchOperation {
		
		private final ch.kuramo.javie.core.FrameBlend _newValue;
		private final ch.kuramo.javie.core.FrameBlend _oldValue;

		public FrameBlend(ProjectManager projectManager, Layer layer, ch.kuramo.javie.core.FrameBlend newValue) {
			super(projectManager, layer, "フレームブレンドスイッチの変更");
			_newValue = newValue;
			_oldValue = LayerNature.getFrameBlend(layer);
			_noEffect = (_newValue == _oldValue);
		}

		@Override
		protected void setSwitchValue(boolean undo, Layer layer, ProjectManager pm) {
			LayerNature.setFrameBlend(layer, undo ? _oldValue : _newValue);

			if (pm != null) {
				pm.fireLayerPropertyChange(layer, "frameBlend");
			}
		}
	}

	public static class MotionBlur extends LayerSwitchOperation {

		private final ch.kuramo.javie.core.MotionBlur _newValue;
		private final ch.kuramo.javie.core.MotionBlur _oldValue;

		public MotionBlur(ProjectManager projectManager, Layer layer, ch.kuramo.javie.core.MotionBlur newValue) {
			super(projectManager, layer, "モーションブラースイッチの変更");
			_newValue = newValue;
			_oldValue = LayerNature.getMotionBlur(layer);
			_noEffect = (_newValue == _oldValue);
		}

		@Override
		protected void setSwitchValue(boolean undo, Layer layer, ProjectManager pm) {
			LayerNature.setMotionBlur(layer, undo ? _oldValue : _newValue);

			if (pm != null) {
				pm.fireLayerPropertyChange(layer, "motionBlur");
			}
		}
	}

}
