/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.core.AnimatableLayerReference;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaItem;
import ch.kuramo.javie.core.MediaItemLayer;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.PropertyDescriptor;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class PrecomposeOperation extends ProjectOperationCollection {

	@Inject
	private ProjectEncoder encoder;


	public PrecomposeOperation(
			ProjectManager pm, LayerComposition comp,
			List<Layer> layers, boolean leaveAttrs, String newCompName) {

		super(pm, "プリコンポーズ");
		InjectorHolder.getInjector().injectMembers(this);

		pm.checkComposition(comp);

		if (layers.isEmpty()) {
			throw new IllegalArgumentException();
		}

		for (Layer l : layers) {
			if (pm.checkLayer(l) != comp) {
				throw new IllegalArgumentException();
			}
		}

		if (newCompName.length() == 0) {
			throw new IllegalArgumentException();
		}

		if (leaveAttrs) {
			if (layers.size() != 1 || !(layers.get(0) instanceof MediaItemLayer)) {
				throw new IllegalArgumentException();
			}
			initForLeavingAttributes(pm, comp, (MediaItemLayer)layers.get(0), newCompName);
		} else {
			initForMovingAttributes(pm, comp, layers, newCompName);
		}
	}

	private void initForLeavingAttributes(
			final ProjectManager pm, LayerComposition comp, MediaItemLayer layer, String newCompName) {

		// TODO レイヤーのフレームブレンドが有効になってる場合、どうする？
		// TODO コンポジションのフレームブレンドとモーションブラーも同じ状態にする。

		MediaInput input = layer.getMediaInput();

		VideoBounds bounds = input.getVideoFrameBounds();
		Size2i size;
		if (bounds != null) {
			size = new Size2i(bounds.width, bounds.height);
		} else {
			// ここに来ることはあるか？
			size = comp.getSize();
		}

		Time duration = input.getDuration();
		if (duration == null) {
			//duration = comp.getDuration(); // ←AEではこっちみたいだけど↓の方がいいんじゃないかしら？
			duration = layer.getOutPoint().subtract(layer.getStartTime());
		}

		final String compId = comp.getId();
		final String layerId = layer.getId();

		final NewLayerCompositionOperation newCompOp = new NewLayerCompositionOperation(
				pm, null, newCompName, comp.getColorMode(), size, comp.getFrameDuration(), duration,
				comp.isFrameDurationPreserved(), comp.getMotionBlurShutterAngle(),
				comp.getMotionBlurShutterPhase(), comp.getMotionBlurSamples());

		ProjectOperationProxy newLayerOp = new ProjectOperationProxy(pm, "NewLayerFromItemOperation", true) {
			ProjectOperation createOperation() {
				CompositionItem newCompItem = pm.getProject().getItem(newCompOp.getCompositionItemId());
				LayerComposition newComp = (LayerComposition) newCompItem.getComposition();

				LayerComposition comp = pm.getProject().getComposition(compId);
				MediaItemLayer layer = (MediaItemLayer) comp.getLayer(layerId);

				return new NewLayerFromItemOperation(pm, newComp, 0, Arrays.<Item>asList(layer.getItem()));
			}
		};

		ProjectOperationProxy replaceItemOp = new ProjectOperationProxy(pm, "ReplaceLayerItemOperation", true) {
			ProjectOperation createOperation() {
				CompositionItem newCompItem = pm.getProject().getItem(newCompOp.getCompositionItemId());

				LayerComposition comp = pm.getProject().getComposition(compId);
				MediaItemLayer layer = (MediaItemLayer) comp.getLayer(layerId);

				return new ReplaceLayerItemOperation<MediaItem>(pm, layer, newCompItem);
			}
		};

		add(newCompOp);
		add(newLayerOp);
		add(replaceItemOp);
	}

	private void initForMovingAttributes(
			final ProjectManager pm, LayerComposition comp, List<Layer> layers, String newCompName) {

		final String compId = comp.getId();
		final int newLayerIndex = comp.getLayers().indexOf(layers.get(layers.size()-1)) + 1 - layers.size();

		List<String> data = Util.newList();
		final Map<String, String> idMap = Util.newMap();

		for (Layer l : layers) {
			data.add(encoder.encodeElement(l));
			idMap.put(l.getId(), Util.randomId());
		}

		final NewLayerCompositionOperation newCompOp = new NewLayerCompositionOperation(
				pm, null, newCompName, comp.getColorMode(),
				comp.getSize(), comp.getFrameDuration(), comp.getDuration(),
				comp.isFrameDurationPreserved(), comp.getMotionBlurShutterAngle(),
				comp.getMotionBlurShutterPhase(), comp.getMotionBlurSamples());

		NewLayersForPrecomposeOperation newLayersOp = new NewLayersForPrecomposeOperation(
				pm, newCompOp.getLayerCompositionId(), data, idMap);

		ProjectOperationProxy rmOrigLayersOp = new ProjectOperationProxy(pm, "RemoveLayersOperation", true) {
			ProjectOperation createOperation() {
				LayerComposition comp = pm.getProject().getComposition(compId);
				List<Layer> layers = Util.newList();
				for (String id : idMap.keySet()) {
					layers.add(comp.getLayer(id));
				}
				return new RemoveLayersOperation(pm, layers);
			}
		};

		ProjectOperationProxy newPrecompLayerOp = new ProjectOperationProxy(pm, "NewLayerFromItemOperation", true) {
			ProjectOperation createOperation() {
				CompositionItem newCompItem = pm.getProject().getItem(newCompOp.getCompositionItemId());
				LayerComposition comp = pm.getProject().getComposition(compId);
				return new NewLayerFromItemOperation(pm, comp, newLayerIndex, Arrays.<Item>asList(newCompItem));
			}
		};

		add(newCompOp);
		add(newLayersOp);
		add(rmOrigLayersOp);
		add(newPrecompLayerOp);
	}

}

abstract class ProjectOperationProxy extends ProjectOperation {

	private ProjectOperation operation;

	public ProjectOperationProxy(ProjectManager projectManager, String label) {
		super(projectManager, label);
	}

	public ProjectOperationProxy(ProjectManager projectManager, String label, String relation) {
		super(projectManager, label, relation);
	}

	public ProjectOperationProxy(ProjectManager projectManager, String label, boolean syncShadow) {
		super(projectManager, label, syncShadow);
	}

	public ProjectOperationProxy(ProjectManager projectManager, String label, String relation, boolean syncShadow) {
		super(projectManager, label, relation, syncShadow);
	}

	abstract ProjectOperation createOperation();

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		if (operation == null) {
			operation = createOperation();
		}

		return operation.executeOrRedo(monitor, info, project, pm);
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		return operation.undo(monitor, info, project, pm);
	}

}

class NewLayersForPrecomposeOperation extends ProjectOperation {

	private final String newCompId;

	private final List<String> data;

	private final Map<String, String> idMap;

	@Inject
	private ProjectDecoder decoder;


	public NewLayersForPrecomposeOperation(
			ProjectManager pm, String newCompId, List<String> data, Map<String, String> idMap) {

		super(pm, "NewLayersForPrecomposeOperation", true);
		InjectorHolder.getInjector().injectMembers(this);

		this.newCompId = newCompId;
		this.data = data;
		this.idMap = idMap;
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition newComp = project.getComposition(newCompId);

		List<Layer> newLayers = Util.newList();
		try {
			for (String d : data) {
				Layer layer = decoder.decodeElement(d, Layer.class);
				setNewId(layer);
				newLayers.add(layer);
			}
			for (Layer layer : newLayers) {
				layer.afterDecode(project, newComp);
			}
		} catch (ProjectDecodeException e) {
			throw new ExecutionException("error decoding layer data", e);
		}

		newComp.getLayers().addAll(newLayers);

		if (pm != null) {
			pm.fireLayersAdd(newComp, newLayers);
		}

		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition newComp = project.getComposition(newCompId);

		List<Layer> removedLayers = Util.newList();
		for (String id : idMap.values()) {
			removedLayers.add(newComp.getLayer(id));
		}

		newComp.getLayers().removeAll(removedLayers);

		if (pm != null) {
			pm.fireLayersRemove(newComp, removedLayers);
		}

		return Status.OK_STATUS;
	}

	private void setNewId(Layer layer) {
		String newId = idMap.get(layer.getId());
		PropertyUtil.setProperty(layer, "id", newId);

		layer.setParentId(idMap.get(layer.getParentId()));

		if (layer instanceof EffectableLayer) {
			for (Effect e : ((EffectableLayer) layer).getEffects()) {
				for (PropertyDescriptor pd : e.getEffectDescriptor().getPropertyDescriptors()) {
					if (AnimatableLayerReference.class.isAssignableFrom(pd.getPropertyClass())) {
						AnimatableLayerReference alr = (AnimatableLayerReference) pd.get(e);
						if (alr.hasKeyframe()) {
							for (Keyframe<String> kf : alr.getKeyframeMap().values()) {
								String refId = idMap.get(kf.value);
								alr.putKeyframe(kf.time, refId != null ? refId : "", kf.interpolation);
							}
						} else {
							String refId = idMap.get(alr.getStaticValue());
							alr.clearKeyframes(refId != null ? refId : "");
						}
					}
				}
			}
		}
	}

}
