/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class RemoveLayersOperation extends ProjectOperation {

	private static class Entry {
		private String data;
		private String layerId;
		private int layerIndex;

		private Entry(String data, String layerId, int layerIndex) {
			super();
			this.data = data;
			this.layerId = layerId;
			this.layerIndex = layerIndex;
		}
	}
	

	private final String _compId;

	private final List<Entry> _entries = Util.newList();

	private final Map<String, String> _parents = Util.newMap();

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;


	public RemoveLayersOperation(ProjectManager pm, Collection<Layer> layers) {
		super(pm, "レイヤーの消去");
		InjectorHolder.getInjector().injectMembers(this);

		LayerComposition comp = pm.checkLayer(layers.iterator().next());
		_compId = comp.getId();

		Set<Layer> set = Util.newSet();

		for (Layer layer : layers) {
			int layerIndex = comp.getLayers().indexOf(layer);
			if (layerIndex == -1) {
				throw new IllegalArgumentException();
			}
			String data = _encoder.encodeElement(layer);
			_entries.add(new Entry(data, layer.getId(), layerIndex));

			set.add(layer);
		}

		Collections.sort(_entries, new Comparator<Entry>() {
			public int compare(Entry o1, Entry o2) {
				return o1.layerIndex - o2.layerIndex;
			}
		});

		// 親レイヤーが削除対象になっているレイヤーを記録しておく。
		for (Layer layer : comp.getLayers()) {
			Layer parent = comp.getParentLayer(layer);
			if (parent != null && set.contains(parent)) {
				_parents.put(layer.getId(), parent.getId());
			}
		}
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);

		for (Map.Entry<String, String> e : _parents.entrySet()) {
			Layer layer = comp.getLayer(e.getKey());
			comp.setParentLayer(layer, null);

			if (pm != null) {
				pm.fireLayerPropertyChange(layer, "parentId");
			}
		}

		List<Layer> layers = Util.newList();

		for (Entry e : _entries) {
			layers.add(comp.getLayer(e.layerId));
		}

		comp.getLayers().removeAll(layers);

		if (pm != null) {
			pm.fireLayersRemove(comp, layers);
		}

		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);

		List<Layer> layers = null;
		if (pm != null) {
			layers = Util.newList();
		}

		for (Entry e : _entries) {
			Layer layer;
			try {
				layer = _decoder.decodeElement(e.data, Layer.class);
				// 親レイヤーも一緒に削除されていた場合、afterDecodeが失敗するので一旦 null にしておく。
				if (_parents.containsKey(layer.getId())) {
					comp.setParentLayer(layer, null);
				}
				layer.afterDecode(project, comp);
			} catch (ProjectDecodeException e1) {
				throw new ExecutionException("error decoding layer data", e1);
			}

			comp.getLayers().add(e.layerIndex, layer);

			if (pm != null) {
				layers.add(layer);
			}
		}

		for (Map.Entry<String, String> e : _parents.entrySet()) {
			Layer layer = comp.getLayer(e.getKey());
			comp.setParentLayer(layer, comp.getLayer(e.getValue()));

			if (pm != null) {
				pm.fireLayerPropertyChange(layer, "parentId");
			}
		}

		if (pm != null) {
			pm.fireLayersAdd(comp, layers);
		}

		return Status.OK_STATUS;
	}

}
