/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.TreeItem;

import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.app.project.ModifyLayerPropertyOperation;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.Layer;

public abstract class LayerPropertyElement<T> extends Element {

	private static final int[] UNDERLINE_DASH = new int[] { 2, 2 };

	private static final boolean COCOA = SWT.getPlatform().equals("cocoa");

	private static final boolean WIN32 = SWT.getPlatform().equals("win32");


	protected final Layer layer;

	protected final String property;

	protected final String label;

	protected final String operationLabel;

	protected Rectangle valueArea;


	public LayerPropertyElement(
			Element parent, Layer layer, String property, String label) {

		this(parent, layer, property, label, label+"を変更");
	}

	public LayerPropertyElement(
			Element parent, Layer layer, String property,
			String label, String operationLabel) {

		super(parent);
		this.layer = layer;
		this.property = property;
		this.label = label;
		this.operationLabel = operationLabel;
	}

	public Layer getLayer() {
		return layer;
	}

	public String getProperty() {
		return property;
	}

	@Override
	public String getColumnText(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				return label;

			case LayerCompositionView.VALUE_COL:
				// ここでnullや空文字を返すと、なぜかCellEditorがすぐにフォーカスアウトしてしまう場合がある。
				return " ";

			default:
				return null;
		}
	}

	@Override
	public void paintColumn(Event event) {
		if (event.index == LayerCompositionView.VALUE_COL) {
			String valueStr = getValueString();

			GC gc = event.gc;
			int x = event.x + 5;
			int y = event.y;
			int height = ((TreeItem) event.item).getBounds(event.index).height;
			boolean focused = ((event.detail & SWT.SELECTED) != 0 && ((Control) event.widget).isFocusControl());


			gc.setForeground(gc.getDevice().getSystemColor(
					focused ? SWT.COLOR_WHITE : SWT.COLOR_DARK_BLUE));

			Point extent = gc.textExtent(valueStr, SWT.DRAW_TRANSPARENT);

			y += (height - extent.y) / 2;
			gc.drawString(valueStr, x, y, true);

			valueArea = new Rectangle(x, y, extent.x, extent.y);

			y += extent.y;
			gc.setLineDash(UNDERLINE_DASH);
			if (COCOA) {
				gc.drawLine(x, y - 1, x + extent.x, y - 1);
			} else if (WIN32) {
				gc.drawLine(x, y, x + extent.x - 2, y);
			} else {
				gc.drawLine(x, y, x + extent.x, y);
			}

		} else {
			super.paintColumn(event);
		}
	}

	@Override
	public void updateCursor(MouseEvent event, int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.VALUE_COL:
				Cursor cursor = null;
				if (valueArea != null && valueArea.contains(event.x, event.y)) {
					cursor = event.display.getSystemCursor(SWT.CURSOR_HAND);
				}
				viewer.getTree().setCursor(cursor);
				break;

			default:
				super.updateCursor(event, columnIndex);
				break;
		}
	}

	@Override
	public void mouseDown(MouseEvent event, int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.VALUE_COL:
				if (event.button == 1 && valueArea != null && valueArea.contains(event.x, event.y)) {
					mouseDownOnValueArea();
				}
				break;

			default:
				super.mouseDown(event, columnIndex);
				break;
		}
	}

	protected abstract String getValueString();

	protected abstract void mouseDownOnValueArea();


	protected T getValue() {
		return PropertyUtil.getProperty(layer, property);
	}

	protected void setValue(T newValue) {
		ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);
		pm.postOperation(new ModifyLayerPropertyOperation<T>(pm, layer, property, newValue, operationLabel));
	}

}
