/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.wizards;

import java.util.Arrays;

import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.PaletteData;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.ColorDialog;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Text;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.app.widgets.GridBuilder;

public class SolidColorItemWizardPage extends WizardPage {

	// TODO この最大値は暫定。グラフィックカード毎に異なるはず。
	private static final int MAX_SIZE = 4096;

	private SolidColorItemSettings _settings;

	private Color _color;

	private Text _nameText;

	private Text _widthText;

	private Text _heightText;

	private Button _freezeRatio;

	private ModifyListener _modifyListener;

	private Image _colorImage;

	private int _widthRatio;

	private int _heightRatio;


	public SolidColorItemWizardPage(SolidColorItemSettings settings) {
		super("SolidColorItemWizardPage", "平面", null);
		setDescription("平面を作成します。");
		_settings = settings;
		_color = settings.color;
	}

	public void createControl(Composite parent) {
		GridBuilder gb = new GridBuilder(parent, 6, true);

							gb.hSpan(1).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "名前:");
		_nameText =			gb.hSpan(5).hAlign(SWT.FILL).hGrab().text(SWT.BORDER, _settings.name);

							gb.hSpan(6).size(10, 10).composite(SWT.NULL);

							gb.hSpan(1).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "幅:");
		_widthText =		gb.hSpan(1).hAlign(SWT.FILL).hGrab().text(SWT.BORDER, String.valueOf(_settings.width));
		_freezeRatio =		gb.span(4, 2).hAlign(SWT.FILL).hGrab().button(SWT.CHECK, "縦横比を固定");

							gb.hSpan(1).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "高さ:");
		_heightText =		gb.hSpan(1).hAlign(SWT.FILL).hGrab().tabAfter(_widthText)
								.text(SWT.BORDER, String.valueOf(_settings.height));

							gb.hSpan(6).size(10, 10).composite(SWT.NULL);

							gb.hSpan(1).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "カラー:");
		Button colorBtn =	gb.hSpan(1).button(SWT.PUSH | SWT.FLAT, "");
							gb.hSpan(4).size(10, 10).composite(SWT.NULL);


		_modifyListener = new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				doValidate((Control) e.getSource());
			}
		};
		_nameText.addModifyListener(_modifyListener);
		_widthText.addModifyListener(_modifyListener);
		_heightText.addModifyListener(_modifyListener);

		_freezeRatio.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				doValidate(_freezeRatio);
			}
		});

		colorBtn.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				ColorDialog dialog = new ColorDialog(getShell());
				dialog.setRGB(getRGB());
				RGB rgb = dialog.open();
				if (rgb != null) {
					setRGB(rgb);
					updateColorButtonImage((Button) e.getSource(), rgb);
				}
			}
		});
		updateColorButtonImage(colorBtn, getRGB());


		Composite grid = gb.getComposite();
		grid.setTabList(gb.getTabList());

		doValidate(null);
		setControl(grid);
	}

	public void dispose() {
		if (_colorImage != null) {
			_colorImage.dispose();
			_colorImage = null;
		}
		super.dispose();
	}

	private RGB getRGB() {
		return new RGB((int)(_color.r*255), (int)(_color.g*255), (int)(_color.b*255));
	}

	private void setRGB(RGB rgb) {
		_color = new Color(rgb.red/255.0, rgb.green/255.0, rgb.blue/255.0, 1.0);
	}

	private void updateColorButtonImage(Button button, RGB rgb) {
		PaletteData palette = new PaletteData(new RGB[] { new RGB(0, 0, 0), rgb });
		byte[] data = new byte[192];
		Arrays.fill(data, (byte) 255);
		ImageData imageData = new ImageData(64, 24, 1, palette, 1, data);

		Image image = new Image(button.getDisplay(), imageData);
		button.setImage(image);
		if (_colorImage != null) {
			_colorImage.dispose();
		}
		_colorImage = image;
	}

	private void doValidate(Control control) {
		setPageComplete(false);

		String wStr = _widthText.getText().trim();
		int width;
		try {
			width = Integer.parseInt(wStr);
		} catch (NumberFormatException e) {
			width = 0;
		}

		String hStr = _heightText.getText().trim();
		int height;
		try {
			height = Integer.parseInt(hStr);
		} catch (NumberFormatException e) {
			height = 0;
		}

		boolean frozen = _freezeRatio.getSelection();
		if (frozen) {
			if (control == _widthText && width > 0) {
				height = Math.max(1, width * _heightRatio / _widthRatio);
				_heightText.removeModifyListener(_modifyListener);
				_heightText.setText(String.valueOf(height));
				_heightText.addModifyListener(_modifyListener);
			} else if (control == _heightText && height > 0) {
				width = Math.max(1, height * _widthRatio / _heightRatio);
				_widthText.removeModifyListener(_modifyListener);
				_widthText.setText(String.valueOf(width));
				_widthText.addModifyListener(_modifyListener);
			}
		} else if ((control == null || control == _widthText || control == _heightText) && width > 0 && height > 0) {
			int gcd = gcd(width, height);
			_widthRatio = width / gcd;
			_heightRatio = height / gcd;
			_freezeRatio.setText(String.format("縦横比を%d:%dに固定", _widthRatio, _heightRatio));
		}

		if (_nameText.getText().trim().length() == 0) {
			setErrorMessage("名前が入力されていません。");
			return;
		}
		if (wStr.length() == 0) {
			setErrorMessage("幅が入力されていません。");
			return;
		}
		if (width < 1 || width > MAX_SIZE) {
			setErrorMessage(String.format("幅には1から%dまでの値を入力してください。", MAX_SIZE));
			return;
		}
		if (hStr.length() == 0) {
			setErrorMessage("高さが入力されていません。");
			return;
		}
		if (height < 1 || height > MAX_SIZE) {
			setErrorMessage(String.format("高さには1から%dまでの値を入力してください。", MAX_SIZE));
			return;
		}

		setErrorMessage(null);
		setPageComplete(true);
	}

	private int gcd(int x, int y) {
		while (y != 0) {
			int t = x % y;
			x = y;
			y = t;
		}
		return x;
	}

	public SolidColorItemSettings getResult() {
		if (!isPageComplete()) {
			return null;
		}

		SolidColorItemSettings result = new SolidColorItemSettings();
		result.name = _nameText.getText().trim();
		result.width = Integer.parseInt(_widthText.getText().trim());
		result.height = Integer.parseInt(_heightText.getText().trim());
		result.color = _color;
		return result;
	}

}

class SolidColorItemSettings implements Cloneable {

	String name = "平面";

	int width = 640;

	int height = 480;

	Color color = new Color(1, 0, 0);


	SolidColorItemSettings() {
	}

	SolidColorItemSettings(String name, int width, int height, Color color) {
		this.name = name;
		this.width = width;
		this.height = height;
		this.color = color;
	}

	public SolidColorItemSettings clone() {
		try {
			return (SolidColorItemSettings) super.clone();
		} catch (CloneNotSupportedException e) {
			throw new Error(e);
		}
	}

}
