/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

import net.arnx.jsonic.JSONHint;
import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.BlendMode;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IObjectArray;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IAccumulationSupport;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.core.AbstractAnimatableEnum;
import ch.kuramo.javie.core.AbstractTransformableLayer;
import ch.kuramo.javie.core.AnimatableDouble;
import ch.kuramo.javie.core.AnimatableVec2d;
import ch.kuramo.javie.core.CastsShadows;
import ch.kuramo.javie.core.CoreContext;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.ExpressionScope;
import ch.kuramo.javie.core.FrameBlend;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.MotionBlur;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.Quality;
import ch.kuramo.javie.core.TrackMatte;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.VectorMediaInput;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.VideoLayerRenderer.AbstractCRLayerRenderer;
import ch.kuramo.javie.core.VideoLayerRenderer.AbstractNormalLayerRenderer;
import ch.kuramo.javie.core.VideoLayerRenderer.CRLayerRenderer;
import ch.kuramo.javie.core.VideoLayerRenderer.GeneralLayerRenderer;
import ch.kuramo.javie.core.VideoLayerRenderer.NormalLayerRenderer;
import ch.kuramo.javie.core.services.AudioEffectPipeline;
import ch.kuramo.javie.core.services.AudioRenderSupport;
import ch.kuramo.javie.core.services.RenderContext;
import ch.kuramo.javie.core.services.VideoEffectPipeline;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

public abstract class AbstractMediaLayer extends AbstractTransformableLayer implements MediaLayer {

	private boolean _effectsEnabled = true;

	private List<Effect> _effects = Util.newList();

	private Quality _quality = Quality.NORMAL;

	private MotionBlur _motionBlur = MotionBlur.NONE;

	private TrackMatte _trackMatte = TrackMatte.NONE;

	private AnimatableDouble _opacity = new AnimatableDouble(100d, 0d, 100d);

	private BlendMode _blendMode = BlendMode.NORMAL;

	private AnimatableVec2d _audioLevels = new AnimatableVec2d(Vec2d.ZERO, new Vec2d(-48, -48), new Vec2d(48, 48));

	private AnimatableDouble _timeRemap = new AnimatableDouble(0.0);

	private CastsShadows _castsShadows = CastsShadows.OFF;

	private AnimatableDouble _lightTransmission = new AnimatableDouble(0d, 0d, 100d);

	private boolean _acceptsShadows = true;

	private boolean _acceptsLights = true;

	private AnimatableDouble _ambient = new AnimatableDouble(100d, 0d, 100d);

	private AnimatableDouble _diffuse = new AnimatableDouble(50d, 0d, 100d);

	private AnimatableDouble _specular = new AnimatableDouble(50d, 0d, 100d);

	private AnimatableDouble _shininess = new AnimatableDouble(5d, 0d, 100d);

	private AnimatableDouble _metal = new AnimatableDouble(100d, 0d, 100d);

	@Inject
	private RenderContext _context;

	@Inject
	private IVideoRenderSupport _support;

	@Inject
	private VideoRenderSupport _oldSupport;

	@Inject
	private VideoEffectPipeline _vePipeline;

	@Inject
	private IAccumulationSupport _accumSuport;

	@Inject
	private AudioRenderSupport _arSupport;

	@Inject
	private AudioEffectPipeline _aePipeline;

	@Inject
	private IArrayPools _arrayPools;


	@JSONHint(ignore=true)
	public boolean isVideoNature() {
		return getMediaInput().isVideoAvailable();
	}

	@JSONHint(ignore=true)
	public boolean isAudioNature() {
		return getMediaInput().isAudioAvailable();
	}

	public boolean isEffectsEnabled() {
		return _effectsEnabled;
	}

	public void setEffectsEnabled(boolean effectsEnabled) {
		_effectsEnabled = effectsEnabled;
	}

	public List<Effect> getEffects() {
		return _effects;
	}

	public void setEffects(List<Effect> effects) {
		_effects = effects;
	}

	public Quality getQuality() {
		return _quality;
	}

	public void setQuality(Quality quality) {
		_quality = quality;
	}

	public MotionBlur getMotionBlur() {
		return _motionBlur;
	}

	public void setMotionBlur(MotionBlur motionBlur) {
		_motionBlur = motionBlur;
	}

	public TrackMatte getTrackMatte() {
		return _trackMatte;
	}

	public void setTrackMatte(TrackMatte trackMatte) {
		_trackMatte = trackMatte;
	}

	public AnimatableDouble getOpacity() {
		return _opacity;
	}

	public void setOpacity(AnimatableDouble opacity) {
		opacity.copyConfigurationFrom(_opacity);
		_opacity = opacity;
	}

	public BlendMode getBlendMode() {
		return _blendMode;
	}

	public void setBlendMode(BlendMode blendMode) {
		_blendMode = blendMode;
	}

	public AnimatableVec2d getAudioLevels() {
		return _audioLevels;
	}

	public void setAudioLevels(AnimatableVec2d audioLevels) {
		audioLevels.copyConfigurationFrom(_audioLevels);
		_audioLevels = audioLevels;
	}

	public AnimatableDouble getTimeRemap() {
		return _timeRemap;
	}

	public void setTimeRemap(AnimatableDouble timeRemap) {
		timeRemap.copyConfigurationFrom(_timeRemap);
		_timeRemap = timeRemap;
	}

	public CastsShadows getCastsShadows() {
		return _castsShadows;
	}

	public void setCastsShadows(CastsShadows castsShadows) {
		_castsShadows = castsShadows;
	}

	public AnimatableDouble getLightTransmission() {
		return _lightTransmission;
	}

	public void setLightTransmission(AnimatableDouble lightTransmission) {
		lightTransmission.copyConfigurationFrom(_lightTransmission);
		_lightTransmission = lightTransmission;
	}

	public boolean isAcceptsShadows() {
		return _acceptsShadows;
	}

	public void setAcceptsShadows(boolean acceptsShadows) {
		_acceptsShadows = acceptsShadows;
	}

	public boolean isAcceptsLights() {
		return _acceptsLights;
	}

	public void setAcceptsLights(boolean acceptsLights) {
		_acceptsLights = acceptsLights;
	}

	public AnimatableDouble getAmbient() {
		return _ambient;
	}

	public void setAmbient(AnimatableDouble ambient) {
		ambient.copyConfigurationFrom(_ambient);
		_ambient = ambient;
	}

	public AnimatableDouble getDiffuse() {
		return _diffuse;
	}

	public void setDiffuse(AnimatableDouble diffuse) {
		diffuse.copyConfigurationFrom(_diffuse);
		_diffuse = diffuse;
	}

	public AnimatableDouble getSpecular() {
		return _specular;
	}

	public void setSpecular(AnimatableDouble specular) {
		specular.copyConfigurationFrom(_specular);
		_specular = specular;
	}

	public AnimatableDouble getShininess() {
		return _shininess;
	}

	public void setShininess(AnimatableDouble shininess) {
		shininess.copyConfigurationFrom(_shininess);
		_shininess = shininess;
	}

	public AnimatableDouble getMetal() {
		return _metal;
	}

	public void setMetal(AnimatableDouble metal) {
		metal.copyConfigurationFrom(_metal);
		_metal = metal;
	}

	public boolean isVectorLayer() {
		return (getMediaInput() instanceof VectorMediaInput);
	}

	@Override
	public void prepareExpression(ExpressionScope scope) {
		super.prepareExpression(scope);

		scope.assignTo(_opacity);
		scope.assignTo(_audioLevels);
		scope.assignTo(_timeRemap);

		scope.assignTo(_lightTransmission);
		scope.assignTo(_ambient);
		scope.assignTo(_diffuse);
		scope.assignTo(_specular);
		scope.assignTo(_shininess);
		scope.assignTo(_metal);

		for (Effect effect : _effects) {
			effect.prepareExpression(scope.clone());
		}
	}

	public GeneralLayerRenderer createVideoRenderer() {
		// エフェクトからの参照時にこのチェックがあるとまずいので。同じチェックはコンポ側でもしている。
		//if (!LayerNature.isVideoEnabled(this)) {
		//	return;
		//}

		MediaInput input = getMediaInput();
		if (input == null || !input.isVideoAvailable()) {
			return null;
		}

		if (isVectorLayer() && LayerNature.isCTCR(this)) {
			return createCRLayerRenderer();
		} else {
			return createNormalLayerRenderer();
		}
	}

	private NormalLayerRenderer createNormalLayerRenderer() {
		return new AbstractNormalLayerRenderer() {

			public MediaLayer getLayer() {
				return AbstractMediaLayer.this;
			}

			public double getOpacity() {
				return _opacity.value(_context) / 100;
			}

			public void multModelViewMatrix(double[] mvMatrix) {
				_oldSupport.setMatrix(mvMatrix, null);
				LayerMatrixUtil.multModelViewMatrix(AbstractMediaLayer.this, _context, _oldSupport);
				_oldSupport.getMatrix(mvMatrix, null);
			}

			public VideoBounds calcBounds(boolean withEffects) {
				return _vePipeline.getVideoBounds(getEffects(withEffects), inputBoundsOperation);
			}

			public IVideoBuffer render(boolean withEffects, boolean frameBlendEnabled) {
				return _vePipeline.doVideoEffects(getEffects(withEffects), inputBoundsOperation,
													createInputBufferOperation(frameBlendEnabled));
			}

			private List<Effect> getEffects(boolean withEffects) {
				return (withEffects && _effectsEnabled) ? _effects : Collections.<Effect>emptyList();
			}

			private final WrappedOperation<VideoBounds> inputBoundsOperation = new WrappedOperation<VideoBounds>() {
				public VideoBounds execute() {
					return _context.getVideoResolution().scale(getMediaInput().getVideoFrameBounds());
				}
			};

			private WrappedOperation<IVideoBuffer> createInputBufferOperation(final boolean frameBlendEnabled) {
				return new WrappedOperation<IVideoBuffer>() {
					public IVideoBuffer execute() {
						if (frameBlendEnabled) {
							FrameBlend frameBlend = LayerNature.getFrameBlend(AbstractMediaLayer.this);
							if (frameBlend != FrameBlend.NONE) {
								return blendFrames(frameBlend);
							}
						}
						MediaInput input = getMediaInput();
						return input.getVideoFrame(calcVideoMediaTime(input));
					}
				};
			}
		};
	}

	private CRLayerRenderer createCRLayerRenderer() {
		return new AbstractCRLayerRenderer() {

			public MediaLayer getLayer() {
				return AbstractMediaLayer.this;
			}

			public double getOpacity() {
				return _opacity.value(_context) / 100;
			}

			public void multModelViewMatrix(double[] mvMatrix) {
				_oldSupport.setMatrix(mvMatrix, null);
				LayerMatrixUtil.multModelViewMatrix(AbstractMediaLayer.this, _context, _oldSupport);
				_oldSupport.getMatrix(mvMatrix, null);
			}

			public VideoBounds calcBounds(boolean withEffects, VideoBounds viewport) {
				return _vePipeline.getVideoBounds(getEffects(withEffects), createInputBoundsOperation(viewport));
			}

			public IVideoBuffer render(boolean withEffects, VideoBounds viewport, double[] mvMatrix, double[] prjMatrix) {
				return _vePipeline.doVideoEffects(getEffects(withEffects),
						createInputBoundsOperation(viewport),
						createInputBufferOperation(viewport, mvMatrix, prjMatrix));
			}

			public IVideoBuffer render() {
				return _context.saveAndExecute(new WrappedOperation<IVideoBuffer>() {
					public IVideoBuffer execute() {
						MediaInput input = getMediaInput();
						return input.getVideoFrame(calcVideoMediaTime(input));
					}
				});
			}

			private List<Effect> getEffects(boolean withEffects) {
				return (withEffects && _effectsEnabled) ? _effects : Collections.<Effect>emptyList();
			}

			private WrappedOperation<VideoBounds> createInputBoundsOperation(final VideoBounds viewport) {
				return new WrappedOperation<VideoBounds>() {
					public VideoBounds execute() {
						return viewport;
					}
				};
			}

			private WrappedOperation<IVideoBuffer> createInputBufferOperation(
					final VideoBounds bounds, final double[] mvMatrix, final double[] prjMatrix) {

				return new WrappedOperation<IVideoBuffer>() {
					public IVideoBuffer execute() {
						IVideoBuffer rasterizedBuffer = null;
						try {
							rasterizedBuffer = _support.createVideoBuffer(bounds, _context.getColorMode());
							rasterizedBuffer.clear();

							VectorMediaInput input = (VectorMediaInput) getMediaInput();
							input.rasterize(rasterizedBuffer, mvMatrix, prjMatrix, calcVideoMediaTime(input));

							IVideoBuffer result = rasterizedBuffer;
							rasterizedBuffer = null;
							return result;

						} finally {
							if (rasterizedBuffer != null) rasterizedBuffer.dispose();
						}
					}
				};
			}
		};
	}

	protected Time calcVideoMediaTime(MediaInput input) {
		Time time;

		if (LayerNature.isTimeRemapEnabled(this)) {
			double t0 = _timeRemap.value(_context);

			Time ctxTime = _context.getTime();
			double t1 = _timeRemap.valueAtTime(new Time(ctxTime.timeValue + 1, ctxTime.timeScale), _context);

			Time mediaFrameDuration = input.getVideoFrameDuration();
			int mediaTimeScale = mediaFrameDuration.timeScale;

			time = new Time(Math.round(t0 * mediaTimeScale), mediaTimeScale);

			// 変化量が負の時のときは１フレーム分ずらす。
			if (t0 > t1) {
				time = time.subtract(mediaFrameDuration);
			}

		} else {
			time = _context.getTime().subtract(getStartTime());

			double rate = getRate();
			Time mediaFrameDuration = input.getVideoFrameDuration();

			if (mediaFrameDuration != null) {
				// メディアのタイムスケールにしておかないと他の箇所の計算で誤差が発生することがある。
				int mediaTimeScale = mediaFrameDuration.timeScale;
				time = new Time(Math.round(time.toSecond() * rate * mediaTimeScale), mediaTimeScale);
			} else {
				time = new Time(Math.round(time.timeValue * rate), time.timeScale);
			}

			if (rate < 0.0) {
				Time mediaDuration = input.getDuration();
				if (mediaDuration != null) {
					time = time.add(mediaDuration);

					// 逆再生時は１フレーム分ずらす。
					time = time.subtract(mediaFrameDuration);

				} else {
					mediaDuration = getOutPoint().subtract(getStartTime());
					time = time.add(mediaDuration);
				}
			}
		}

		return time;
	}

	private IVideoBuffer blendFrames(FrameBlend frameBlend) {
		if (frameBlend == FrameBlend.NONE) throw new IllegalArgumentException();
		if (frameBlend != FrameBlend.FRAME_MIX) throw new UnsupportedOperationException();

		MediaInput input = getMediaInput();
		Time mediaFrameDuration = input.getVideoFrameDuration();

		Time ctxTime = _context.getTime();
		Time mediaTime0 = calcVideoMediaTime(input);
		_context.setTime(ctxTime.add(_context.getVideoFrameDuration()));
		Time mediaTime1 = calcVideoMediaTime(input);
		_context.setTime(ctxTime);

		Time mediaTimeDiff = mediaTime1.subtract(mediaTime0);
		double fpsRatio = mediaTimeDiff.toSecond() / mediaFrameDuration.toSecond();
		if (fpsRatio == 1) {
			return input.getVideoFrame(ctxTime);
		}

		fpsRatio = Math.abs(fpsRatio);


		List<IVideoBuffer> buffers = Util.newList();
		IVideoBuffer accumBuf = null;
		try {
			Resolution resolution = _context.getVideoResolution();
			VideoBounds bounds = resolution.scale(input.getVideoFrameBounds());
			ColorMode ctxColorMode = _context.getColorMode();

			// 不動小数テクスチャ使ってもほとんど差は出ないかも。
			accumBuf = ctxColorMode.isFloat()
					? _support.createVideoBuffer(bounds, ctxColorMode)
					: _support.createVideoBuffer(bounds, ColorMode.RGBA16_FLOAT);
			accumBuf.clear();

			List<Double> weights = Util.newList();
			int maxSources = _accumSuport.getMaxSourcesAtATime();

			if (fpsRatio > 1) {
				List<Integer> intWeights = Util.newList();
				int w = (int)Math.round(255/fpsRatio);
				intWeights.add(w);
				int rest = 255 - w;

				for (int i = 1; rest > 0; ++i) {
					double p = 1.17741*2*i/fpsRatio;
					double q = Math.exp(-p*p/2)/fpsRatio;
					w = Math.min(rest, 2*(int)Math.round(255*q))/2;
					if (w > 0) {
						intWeights.add(w);
						rest -= 2*w;
					} else if (rest > 0) {
						// 端数は中央のフレームに上乗せしてしまう。
						intWeights.set(0, intWeights.get(0)+rest);
						rest = 0;
					}
				}

				for (int n = intWeights.size(), i = -n+1; i < n; ++i) {
					Time t = mediaTime0.add(new Time(mediaFrameDuration.timeValue*i, mediaFrameDuration.timeScale));

					buffers.add(input.getVideoFrame(t));
					weights.add(intWeights.get(Math.abs(i))/255.0);

					if (buffers.size() >= maxSources) {
						_accumSuport.accumulate(buffers, weights, accumBuf);
						for (IVideoBuffer vb : buffers) vb.dispose();
						buffers.clear();
						weights.clear();
					}
				}
			} else {
				long frame = mediaTime0.toFrameNumber(mediaFrameDuration);
				Time t0, t1;
				if (mediaTime0.timeValue >= 0) {
					t0 = Time.fromFrameNumber(frame, mediaFrameDuration);
					t1 = Time.fromFrameNumber(frame+1, mediaFrameDuration);
				} else {
					t0 = Time.fromFrameNumber(frame-1, mediaFrameDuration);
					t1 = Time.fromFrameNumber(frame, mediaFrameDuration);
				}
				long tv0 = mediaTime0.subtract(t0).timeValue;
				long tv1 = t1.subtract(mediaTime0).timeValue;

				buffers.add(input.getVideoFrame(t0));
				weights.add((double)tv1/(tv0+tv1));

				// maxSourcesが1の場合はあるのか？　無いのなら次のifブロックは不要。
				if (maxSources == 1) {
					_accumSuport.accumulate(buffers, weights, accumBuf);
					buffers.get(0).dispose();
					buffers.clear();
					weights.clear();
				}

				buffers.add(input.getVideoFrame(t1));
				weights.add((double)tv0/(tv0+tv1));
			}
			if (!buffers.isEmpty()) {
				_accumSuport.accumulate(buffers, weights, accumBuf);
			}

			IVideoBuffer resultBuffer;
			if (accumBuf.getColorMode() == ctxColorMode) {
				resultBuffer = accumBuf;
				accumBuf = null;
			} else {
				IVideoBuffer copy = null;
				try {
					copy = _support.createVideoBuffer(bounds, ctxColorMode);
					_support.copy(accumBuf, copy);
					resultBuffer = copy;
					copy = null;
				} finally {
					if (copy != null) copy.dispose();
				}
			}
			return resultBuffer;
		} finally {
			for (IVideoBuffer vb : buffers) vb.dispose();
			if (accumBuf != null) accumBuf.dispose();
		}
	}

	private IAudioBuffer getTimeRemappedAudioChunk(MediaInput input) {
		IAudioBuffer ab1 = null;
		IAudioBuffer ab2 = null;
		IObjectArray<Double> times = null;

		try {
			AudioMode audioMode = _context.getAudioMode();
			int sampleRate = audioMode.sampleRate;

			ab1 = _arSupport.createAudioBuffer();

			times = _arrayPools.getObjectArray(ab1.getFrameCount());
			_timeRemap.values(times, audioMode.sampleDuration, _context.getAudioAnimationRate(), _context);


			// 音声データを取得する範囲を求める。
			double min = Double.POSITIVE_INFINITY, max = Double.NEGATIVE_INFINITY;
			Object[] timeArray = times.getArray();
			for (int i = 0, n = times.getLength(); i < n; ++i) {
				min = Math.min(min, (Double) timeArray[i]);
				max = Math.max(max, (Double) timeArray[i]);
			}
			long timeValueMin = (long) Math.floor(min*sampleRate);
			long timeValueMax = (long) Math.ceil(max*sampleRate);


			// 上で求めた範囲の音声データをMediaInputから取得。
			_context.setAudioFrameCount((int)(timeValueMax - timeValueMin) + 1);

			ab2 = input.getAudioChunk(new Time(timeValueMin, sampleRate));
			if (ab2 == null) return null;


			// タイムリマップの計算値に従ってリサンプルする。
			Object data1 = ab1.getData();
			Object data2 = ab2.getData();
			int n1 = ab1.getFrameCount();

			switch (audioMode.dataType) {
				case SHORT: {
					short[] array1 = (short[]) data1, array2 = (short[]) data2;
					for (int i = 0; i < n1; ++i) {
						int j = (int)(Math.round((Double)timeArray[i]*sampleRate) - timeValueMin);
						array1[i*2  ] = array2[j*2  ];
						array1[i*2+1] = array2[j*2+1];
					}
					break;
				}
				case INT: {
					int[] array1 = (int[]) data1, array2 = (int[]) data2;
					for (int i = 0; i < n1; ++i) {
						int j = (int)(Math.round((Double)timeArray[i]*sampleRate) - timeValueMin);
						array1[i*2  ] = array2[j*2  ];
						array1[i*2+1] = array2[j*2+1];
					}
					break;
				}
				case FLOAT: {
					float[] array1 = (float[]) data1, array2 = (float[]) data2;
					for (int i = 0; i < n1; ++i) {
						int j = (int)(Math.round((Double)timeArray[i]*sampleRate) - timeValueMin);
						array1[i*2  ] = array2[j*2  ];
						array1[i*2+1] = array2[j*2+1];
					}
					break;
				}
				default:
					throw new UnsupportedOperationException(
							"unsupported AudioMode.DataType: " + audioMode.dataType);
			}


			IAudioBuffer result = ab1;
			ab1 = null;
			return result;

		} finally {
			if (ab1 != null) ab1.dispose();
			if (ab2 != null) ab2.dispose();
			if (times != null) times.release();
		}
	}

	private IAudioBuffer getRateChangedAudioChunk(MediaInput input, Time time, double rate) {
		IAudioBuffer ab1 = null;
		IAudioBuffer ab2 = null;

		try {
			AudioMode audioMode = _context.getAudioMode();
			int sampleRate = audioMode.sampleRate;

			ab1 = _arSupport.createAudioBuffer();

			long timeValueMin;
			long timeValueMax;
			double duration = input.getDuration().toSecond();

			if (rate >= 0) {
				timeValueMin = (long) Math.floor(time.toSecond() * sampleRate * rate);

				Time t = time.add(new Time(_context.getAudioFrameCount(), sampleRate));
				timeValueMax = (long) Math.ceil(t.toSecond() * sampleRate * rate);

			} else {
				timeValueMax =  (long) Math.ceil(duration * sampleRate + time.toSecond() * sampleRate * rate);

				Time t = time.add(new Time(_context.getAudioFrameCount(), sampleRate));
				timeValueMin = (long) Math.floor(duration * sampleRate + t.toSecond() * sampleRate * rate);
			}

			_context.setAudioFrameCount((int)(timeValueMax - timeValueMin) + 1);

			ab2 = input.getAudioChunk(new Time(timeValueMin, sampleRate));
			if (ab2 == null) return null;


			Object data1 = ab1.getData();
			Object data2 = ab2.getData();
			int n1 = ab1.getFrameCount();

			switch (audioMode.dataType) {
				case SHORT: {
					short[] array1 = (short[]) data1, array2 = (short[]) data2;
					for (int i = 0; i < n1; ++i) {
						int j = (rate >= 0) ? (int)(Math.round((time.toSecond() * sampleRate + i) * rate) - timeValueMin)
											: (int)(Math.round(duration * sampleRate + (time.toSecond() * sampleRate + i) * rate) - timeValueMin); 
						array1[i*2  ] = array2[j*2  ];
						array1[i*2+1] = array2[j*2+1];
					}
					break;
				}
				case INT: {
					int[] array1 = (int[]) data1, array2 = (int[]) data2;
					for (int i = 0; i < n1; ++i) {
						int j = (rate >= 0) ? (int)(Math.round((time.toSecond() * sampleRate + i) * rate) - timeValueMin)
											: (int)(Math.round(duration * sampleRate + (time.toSecond() * sampleRate + i) * rate) - timeValueMin); 
						array1[i*2  ] = array2[j*2  ];
						array1[i*2+1] = array2[j*2+1];
					}
					break;
				}
				case FLOAT: {
					float[] array1 = (float[]) data1, array2 = (float[]) data2;
					for (int i = 0; i < n1; ++i) {
						int j = (rate >= 0) ? (int)(Math.round((time.toSecond() * sampleRate + i) * rate) - timeValueMin)
											: (int)(Math.round(duration * sampleRate + (time.toSecond() * sampleRate + i) * rate) - timeValueMin); 
						array1[i*2  ] = array2[j*2  ];
						array1[i*2+1] = array2[j*2+1];
					}
					break;
				}
				default:
					throw new UnsupportedOperationException(
							"unsupported AudioMode.DataType: " + audioMode.dataType);
			}


			IAudioBuffer result = ab1;
			ab1 = null;
			return result;

		} finally {
			if (ab1 != null) ab1.dispose();
			if (ab2 != null) ab2.dispose();
		}
	}

	public IAudioBuffer renderAudioChunk(boolean rawSource) {
		// エフェクトからの参照時にこのチェックがあるとまずいので。同じチェックはコンポ側でもしている。
		//if (!LayerNature.isAudioEnabled(this)) {
		//	return null;
		//}

		final MediaInput input = getMediaInput();
		if (input == null || !input.isAudioAvailable()) {
			return null;
		}

		WrappedOperation<IAudioBuffer> inputOperation = new WrappedOperation<IAudioBuffer>() {
			public IAudioBuffer execute() {
				if (LayerNature.isTimeRemapEnabled(AbstractMediaLayer.this)) {
					return getTimeRemappedAudioChunk(input);

				} else {
					Time time = _context.getTime().subtract(getStartTime());
					double rate = getRate();
					if (rate == 1.0) {
						return input.getAudioChunk(time);
					} else {
						return getRateChangedAudioChunk(input, time, rate);
					}
				}
			}
		};

		IAudioBuffer ab = _aePipeline.doAudioEffects(
				(!rawSource && _effectsEnabled) ? _effects : Collections.<Effect>emptyList(), inputOperation);

		if (!rawSource) {
			if (_audioLevels.hasKeyframe() || _audioLevels.getExpression() != null
					|| !_audioLevels.getStaticValue().equals(Vec2d.ZERO)) {

				IObjectArray<Vec2d> audioLevels = _arrayPools.getObjectArray(ab.getFrameCount());

				_audioLevels.values(audioLevels, ab.getAudioMode().sampleDuration,
									_context.getAudioAnimationRate(), _context);
				_arSupport.amplify(ab, audioLevels);

				audioLevels.release();
			}

			// インポイントよりも手前とアウトポイント以降はゼロクリアする
			Time time = _context.getTime();
			Time sampleDuration = _context.getAudioMode().sampleDuration;

			int clearFrameCount = (int) getInPoint().subtract(time).toFrameNumber(sampleDuration);
			if (clearFrameCount > 0) {
				_arSupport.clear(ab, 0, clearFrameCount);
			}

			int clearFrameOffset = (int) getOutPoint().subtract(time).toFrameNumber(sampleDuration);
			clearFrameCount = _context.getAudioFrameCount() - clearFrameOffset;
			if (clearFrameCount > 0) {
				_arSupport.clear(ab, clearFrameOffset, clearFrameCount);
			}
		}

		return ab;
	}

	@Override
	public void afterDecode(Project p, LayerComposition c) throws ProjectDecodeException {
		super.afterDecode(p, c);

		for (Effect effect : _effects) {
			effect.afterDecode(p);
		}
	}

	public Object createExpressionElement(CoreContext context) {
		return new MediaLayerExpressionElement(context);
	}

	public class MediaLayerExpressionElement extends TransformableLayerExpressionElement {

		public MediaLayerExpressionElement(CoreContext context) {
			super(context);
		}

		public Object getOpacity()				{ return elem(_opacity); }
		public Object getAudioLevels()			{ return elem(_audioLevels); }
		public Object getTimeRemap()			{ return LayerNature.isTimeRemapEnabled(AbstractMediaLayer.this) ? elem(_timeRemap) : null; }

		public String getCastsShadows()			{ return _castsShadows.name(); }
		public Object getLightTransmission()	{ return elem(_lightTransmission); }
		public boolean isAcceptsShadows()		{ return _acceptsShadows; }
		public boolean isAcceptsLights()		{ return _acceptsLights; }
		public Object getAmbient()				{ return elem(_ambient); }
		public Object getDiffuse()				{ return elem(_diffuse); }
		public Object getSpecular()				{ return elem(_specular); }
		public Object getShininess()			{ return elem(_shininess); }
		public Object getMetal()				{ return elem(_metal); }

		public Object effect(int index) {
			Effect effect = _effects.get(index - 1);
			return context.getExpressionElement(effect);
		}

		public Object effect(String name) {
			for (Effect e : _effects) {
				if (e.getName().equals(name)) {
					return context.getExpressionElement(e);
				}
			}
			return null;
		}
	}

	public static class AnimatableCastsShadows
			extends AbstractAnimatableEnum<CastsShadows> {

		public AnimatableCastsShadows(
				CastsShadows staticValue,
				Collection<Keyframe<CastsShadows>> keyframes,
				String expression) {
		
			super(CastsShadows.class, staticValue, keyframes, expression);
		}
	
		public AnimatableCastsShadows(CastsShadows staticValue) {
			super(CastsShadows.class, staticValue);
		}

	}
	
}
