/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.util.Collections;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;

import javax.vecmath.Matrix4d;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Camera;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.Light;
import ch.kuramo.javie.core.LightLayer;
import ch.kuramo.javie.core.LightType;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.RenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

class LightImpl implements Light {

	private final LightLayer layer;

	private final RenderContext context;

	private final VideoRenderSupport oldSupport;

	private final Map<Time, Vec3d> directionCache = Util.newMap();

	private final Map<Time, Vec3d> positionCache = Util.newMap();

	private final Map<AnimatableValue<?>, Map<Time, ?>> valueCache = Util.newMap();


	LightImpl(LightLayer layer, RenderContext context, VideoRenderSupport oldSupport) {
		this.layer = layer;
		this.context = context;
		this.oldSupport = oldSupport;
	}

	private double[] getModelView3D() {
		double[] mvMatrix = new double[16];

		Camera camera = context.getCamera();
		System.arraycopy(camera.getModelView3D(), 0, mvMatrix, 0, 16);

		oldSupport.setMatrix(mvMatrix, null);
		LayerMatrixUtil.multModelViewMatrix(layer, context, oldSupport);
		oldSupport.getMatrix(mvMatrix, null);

		return mvMatrix;
	}

	private <V> V getValue(AnimatableValue<V> avalue) {
		Time time = context.getTime();

		@SuppressWarnings("unchecked")
		Map<Time, V> map = (Map<Time, V>) valueCache.get(avalue);
		if (map != null) {
			V value = map.get(time);
			if (value != null) {
				return value;
			}
		} else {
			map = Util.newMap();
			valueCache.put(avalue, map);
		}

		V value = avalue.value(context);
		map.put(time, value);

		return value;
	}

	public LightType getType() {
		return layer.getLightType();
	}

	public boolean isCastsShadows() {
		return (layer.getLightType() != LightType.AMBIENT && layer.isCastsShadows());
	}

	public Vec3d getDirectionInCameraView() {
		switch (layer.getLightType()) {
			case AMBIENT:
			case POINT:
				throw new IllegalStateException();
		}

		Time time = context.getTime();
		Vec3d dir = directionCache.get(time);
		if (dir != null) {
			return dir;
		}

		Point3d p0 = new Point3d();
		Point3d p1 = new Point3d(0, 0, 1);

		Matrix4d m = new Matrix4d(getModelView3D());
		m.transpose();
		m.transform(p0);
		m.transform(p1);

		Vector3d v = new Vector3d();
		v.sub(p1, p0);
		v.normalize();

		dir = new Vec3d(v.x, v.y, v.z);
		directionCache.put(time, dir);

		return dir;
	}

	public Vec3d getPositionInCameraView() {
		switch (layer.getLightType()) {
			case AMBIENT:
			case PARALLEL:
				throw new IllegalStateException();
		}

		Time time = context.getTime();
		Vec3d pos = positionCache.get(time);
		if (pos != null) {
			return pos;
		}

		Point3d p = new Point3d();
		Matrix4d m = new Matrix4d(getModelView3D());
		m.transpose();
		m.transform(p);

		pos = new Vec3d(p.x, p.y, p.z);
		positionCache.put(time, pos);

		return pos;
	}

	public double getIntensity() {
		return getValue(layer.getIntensity());
	}

	public Vec3d getAttenuation() {
		switch (layer.getLightType()) {
			case POINT:
			case SPOT:
				return getValue(layer.getAttenuation());
			default:
				throw new IllegalStateException();
		}
	}

	public Color getColor() {
		return getValue(layer.getColor());
	}

	public double getConeAngle() {
		if (layer.getLightType() == LightType.SPOT) {
			return getValue(layer.getConeAngle());
		} else {
			throw new IllegalStateException();
		}
	}

	public double getConeFeather() {
		if (layer.getLightType() == LightType.SPOT) {
			return getValue(layer.getConeFeather());
		} else {
			throw new IllegalStateException();
		}
	}

	public double getShadowDarkness() {
		if (isCastsShadows()) {
			return getValue(layer.getShadowDarkness());
		} else {
			throw new IllegalStateException();
		}
	}

	public double getShadowDiffusion() {
		if (isCastsShadows()) {
			if (layer.getLightType() == LightType.PARALLEL) {
				// 並行ライトの場合、例外を投げるよりもゼロを返す方が都合がいい。
				return 0;
			} else {
				return getValue(layer.getShadowDiffusion());
			}
		} else {
			throw new IllegalStateException();
		}
	}

}

class LightSet {

	private final Set<Light> lights;

	@Inject
	LightSet(RenderContext context, VideoRenderSupport oldSupport) {
		LayerComposition comp = (LayerComposition) context.getComposition();

		// 上のレイヤーから(リスト内では末尾から)順に有効なライトレイヤーを探す。
		Set<Light> lights = Util.newSet();
		List<Layer> layers = comp.getLayers();
		Time time = context.getTime();
		for (ListIterator<Layer> it = layers.listIterator(layers.size()); it.hasPrevious(); ) {
			Layer layer = it.previous();
			if (layer instanceof LightLayer && LayerNature.isVideoEnabled(layer)
					&& !layer.getInPoint().after(time) && layer.getOutPoint().after(time)) {
				lights.add(new LightImpl((LightLayer)layer, context, oldSupport));
			}
		}

		this.lights = Collections.unmodifiableSet(lights);
	}

	Set<Light> getLights() {
		return lights;
	}

}
