/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.List;

import ch.kuramo.javie.api.IAnimatableValue;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IObjectArray;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IAudioEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.AudioEffect;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.services.AudioEffectPipeline;
import ch.kuramo.javie.core.services.AudioRenderSupport;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;

public class AudioEffectPipelineImpl implements AudioEffectPipeline {

	@Inject
	private RenderContext _context;

	@Inject
	private IVideoRenderSupport _vrSupport;

	@Inject
	private AudioRenderSupport _arSupport;

	@Inject
	private IArrayPools _arrayPools;


	public IAudioBuffer doAudioEffects(List<Effect> effects, WrappedOperation<IAudioBuffer> inputOperation) {
		AudioEffectContextImpl impl = new AudioEffectContextImpl(enabledAudioEffectsOnly(effects), inputOperation);
		AudioEffectContextProxy.push(impl);
		try {
			return impl.doPreviousEffect();
		} finally {
			AudioEffectContextProxy.pop();
		}
	}

	private List<AudioEffect> enabledAudioEffectsOnly(List<Effect> effects) {
		List<AudioEffect> audioEffects = Util.newList();
		for (Effect e : effects) {
			if (e instanceof AudioEffect && e.isEnabled()) {
				audioEffects.add((AudioEffect) e);
			}
		}
		return audioEffects;
	}


	private class AudioEffectContextImpl
			extends AbstractEffectContext implements IAudioEffectContext {

		private List<AudioEffect> effects;

		private final WrappedOperation<IAudioBuffer> inputOperation;

		private AudioEffect currentEffect;


		private AudioEffectContextImpl(List<AudioEffect> effects, WrappedOperation<IAudioBuffer> inputOperation) {
			super(_context, _vrSupport, _arSupport);

			this.effects = effects;
			this.inputOperation = inputOperation;
		}

		public <V> IObjectArray<V> values(IAnimatableValue<V> avalue) {
			return values(avalue, _context.getAudioAnimationRate());
		}

		public <V> IObjectArray<V> values(IAnimatableValue<V> avalue, int animationRate) {
			@SuppressWarnings("unchecked")
			AnimatableValue<V> av = (AnimatableValue<V>) avalue;

			animationRate = Math.max(1, Math.min(animationRate, _context.getAudioAnimationRate()));

			IObjectArray<V> objArray = _arrayPools.getObjectArray(_context.getAudioFrameCount());
			av.values(objArray, _context.getAudioMode().sampleDuration, animationRate, _context);
			return objArray;
		}

		public IAudioBuffer doPreviousEffect() {
			if (effects.isEmpty()) {
				IAudioBuffer ab = _context.saveAndExecute(inputOperation);
				if (ab == null) {
					ab = _arSupport.createAudioBuffer();
					_arSupport.clear(ab);
				}
				return ab;
			}

			return executePrevious(new WrappedOperation<IAudioBuffer>() {
				public IAudioBuffer execute() {
					IAudioBuffer buffer = currentEffect.doAudioEffect();
					return (buffer != null) ? buffer : doPreviousEffect();
				}
			});
		}

		private <T> T executePrevious(final WrappedOperation<T> wop) {
			AudioEffect savedCurrent = currentEffect;
			List<AudioEffect> savedEffects = effects;
			try {

				int lastIndex = effects.size() - 1;
				currentEffect = effects.get(lastIndex);
				effects = effects.subList(0, lastIndex);

				return _context.saveAndExecute(wop);

			} finally {
				effects = savedEffects;
				currentEffect = savedCurrent;
			}
		}

		public String getEffectName() {
			if (currentEffect == null) {
				throw new IllegalStateException("no current Effect");
			}
			return currentEffect.getName();
		}

	}

}
