/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.generate;

import javax.media.opengl.GL2;
import javax.vecmath.Point2d;
import javax.vecmath.Vector2d;

import ch.kuramo.javie.api.BlendMode;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IBlendSupport;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.LinearGradient", category=Categories.GENERATE)
public class LinearGradient extends GradientBase {

	@Property
	private IAnimatableVec2d startPoint;

	@Property
	private IAnimatableVec2d endPoint;

	@Property("1,1,1")
	private IAnimatableColor startColor;

	@Property("0,0,0")
	private IAnimatableColor endColor;

	@Property("RGB")
	private IAnimatableEnum<ColorSpace> colorSpace;

	@Property("CLAMP_TO_EDGE")
	private IAnimatableEnum<RepeatMode> repeatMode;

	@Property(value="100", min="0", max="100")
	private IAnimatableDouble opacity;

	@Property
	private IAnimatableEnum<BlendMode> blendMode;


	private final IVideoRenderSupport support;

	@Inject
	public LinearGradient(
			IVideoEffectContext context, IBlendSupport blendSupport,
			IArrayPools arrayPools, IVideoRenderSupport support) {

		super(context, blendSupport, arrayPools);
		this.support = support;
	}

	public IVideoBuffer doVideoEffect() {
		return doGradientEffect(context.value(blendMode), context.value(opacity)/100);
	}

	private class LinearGradientParams {
		private final float translateX, translateY;
		private final float rotateZ;
		private final float scaleX, scaleY;
		private final float texCoord0, texCoord1;
		private final float vertexLeft, vertexTop, vertexRight, vertexBottom;
		private final int texture;

		private LinearGradientParams(VideoBounds bounds, Vec2d startPoint, Vec2d endPoint) {
			Resolution resolution = context.getVideoResolution();

			startPoint = resolution.scale(startPoint);
			endPoint = resolution.scale(endPoint);

			Vector2d startToEnd = new Vector2d(endPoint.x-startPoint.x, endPoint.y-startPoint.y);
			double x = bounds.x;
			double y = bounds.y;
			int w = bounds.width;
			int h = bounds.height;

			Point2d center = new Point2d(x+w*0.5, y+h*0.5);
			double theta = Math.atan2(startToEnd.y, startToEnd.x);
			double gradWidth = w*Math.abs(Math.cos(theta))+h*Math.abs(Math.sin(theta));
			double gradHeight = h*Math.abs(Math.cos(theta))+w*Math.abs(Math.sin(theta));
			double angle = Math.toDegrees(theta);

			translateX = (float)center.x;
			translateY = (float)center.y;
			rotateZ = (float)angle;
			scaleX = (float)(gradWidth/w);
			scaleY = (float)(gradHeight/h);


			Vector2d startToCenter = new Vector2d(center.x-startPoint.x, center.y-startPoint.y);
			Vector2d endToCenter = new Vector2d(center.x-endPoint.x, center.y-endPoint.y);
			double lenOfStartToEnd = startToEnd.length();

			texCoord0 = (float)(-(gradWidth*0.5 - startToCenter.dot(startToEnd)/lenOfStartToEnd)/lenOfStartToEnd);
			texCoord1 = (float)(1+(gradWidth*0.5 + endToCenter.dot(startToEnd)/lenOfStartToEnd)/lenOfStartToEnd);

			vertexLeft = (float)x;
			vertexTop = (float)y;
			vertexRight = (float)(x+w);
			vertexBottom = (float)(y+h);


			Color startColor = context.value(LinearGradient.this.startColor);
			Color endColor = context.value(LinearGradient.this.endColor);
			ColorSpace colorSpace = context.value(LinearGradient.this.colorSpace);
			RepeatMode repeatMode = context.value(LinearGradient.this.repeatMode);
			int texSize = Math.min(Math.max((int)Math.ceil(lenOfStartToEnd), 2), 4000);

			switch (colorSpace) {
				case RGB:
					texture = createRGBGradientTexture(startColor, endColor, texSize, repeatMode);
					break;
				case HSL:
					texture = createHSLGradientTexture(startColor, endColor, false, texSize, repeatMode);
					break;
				case HSL_REVERSE_HUE:
					texture = createHSLGradientTexture(startColor, endColor, true, texSize, repeatMode);
					break;
				default:
					// never reached here
					throw new Error();
			}
		}
	}

	@Override
	protected IVideoBuffer createGradient(final VideoBounds bounds) {
		Vec2d startPoint = context.value(this.startPoint);
		Vec2d endPoint = context.value(this.endPoint);
		if (startPoint.equals(endPoint)) {
			IVideoBuffer buffer = context.createVideoBuffer(bounds);
			buffer.clear();
			return buffer;
		}

		final LinearGradientParams p = new LinearGradientParams(bounds, startPoint, endPoint);
		IVideoBuffer gradient = null;
		try {
			Runnable operation = new Runnable() {
				public void run() {
					GL2 gl = context.getGL().getGL2();

					gl.glActiveTexture(GL2.GL_TEXTURE0);
					gl.glBindTexture(GL2.GL_TEXTURE_1D, p.texture);
					gl.glEnable(GL2.GL_TEXTURE_1D);

					support.ortho2D(bounds);
					gl.glMatrixMode(GL2.GL_MODELVIEW);
					gl.glTranslatef(p.translateX, p.translateY, 0);
					gl.glRotatef(p.rotateZ, 0, 0, 1);
					gl.glScalef(p.scaleX, p.scaleY, 1);
					gl.glTranslatef(-p.translateX, -p.translateY, 0);

					gl.glBegin(GL2.GL_QUADS);
					gl.glTexCoord1f(p.texCoord0);
					gl.glVertex2f(p.vertexLeft, p.vertexTop);
					gl.glTexCoord1f(p.texCoord1);
					gl.glVertex2f(p.vertexRight, p.vertexTop);
					gl.glTexCoord1f(p.texCoord1);
					gl.glVertex2f(p.vertexRight, p.vertexBottom);
					gl.glTexCoord1f(p.texCoord0);
					gl.glVertex2f(p.vertexLeft, p.vertexBottom);
					gl.glEnd();
				}
			};

			gradient = context.createVideoBuffer(bounds);
			gradient.clear();

			support.useFramebuffer(operation, GL2.GL_TEXTURE_BIT | GL2.GL_ENABLE_BIT, gradient);

			IVideoBuffer result = gradient;
			gradient = null;
			return result;

		} finally {
			if (gradient != null) gradient.dispose();
			deleteTextures(p.texture);
		}
	}

}
