/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.generate;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GL2;
import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.BlendMode;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IBlendSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.effects.Texture1D;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.Rectangle", category=Categories.GENERATE)
public class Rectangle {

	public enum Edge {
		NONE,
		THICKNESS,
		P_THICKNESS,				// PはProportionalのP
		P_THICKNESS_P_FEATHER
	}


	@Property
	private IAnimatableVec2d location;

	@Property(value="150", min="0")
	private IAnimatableVec2d size;

	@Property("0.5")
	private IAnimatableVec2d anchor;

	@Property
	private IAnimatableEnum<Edge> edge;

	@Property(value="10", min="0", max="4000")
	private IAnimatableDouble thickness;

	@Property(min="0", max="4000")
	private IAnimatableDouble outerFeather;

	@Property(min="0", max="4000")
	private IAnimatableDouble innerFeather;

	@Property
	private IAnimatableBoolean invert;

	@Property("1,1,1")
	private IAnimatableColor color;

	@Property(value="100", min="0", max="100")
	private IAnimatableDouble opacity;

	@Property
	private IAnimatableEnum<BlendMode> blendMode;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IBlendSupport blendSupport;

	private final IArrayPools arrayPools;

	private final IShaderProgram rectangleSamplerProgram;

	private final IShaderProgram invertSamplerProgram;

	@Inject
	public Rectangle(
			IVideoEffectContext context, IVideoRenderSupport support, IBlendSupport blendSupport,
			IArrayPools arrayPools, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		this.blendSupport = blendSupport;
		this.arrayPools = arrayPools;
		rectangleSamplerProgram = shaders.getProgram(Rectangle.class, "RECTANGLE_SAMPLER");
		invertSamplerProgram = shaders.getProgram(Rectangle.class, "INVERT_SAMPLER");
	}

	private class RectangleParams {
		private final Vec2d center;
		private final boolean invert;
		private final Color color;
		private final double opacity;

		private final double halfWidth;
		private final double halfHeight;
		private final double innerMiddleX;
		private final double innerMiddleY;
		private final double outerMiddleX;
		private final double outerMiddleY;

		private final int[] texture;
		private final int[] texSize = new int[3];

		private RectangleParams() {
			Resolution resolution = context.getVideoResolution();

			Vec2d location = context.value(Rectangle.this.location);
			Vec2d size = context.value(Rectangle.this.size);
			Vec2d anchor = context.value(Rectangle.this.anchor);

			double halfDiag = Math.sqrt(size.x*size.x + size.y*size.y)*0.5;
			center = resolution.scale(new Vec2d(location.x-size.x*(anchor.x-0.5), location.y-size.y*(anchor.y-0.5)));
			size = resolution.scale(size);

			Edge edge = context.value(Rectangle.this.edge);
			invert = context.value(Rectangle.this.invert);
			opacity = context.value(Rectangle.this.opacity) / 100;

			Color color = context.value(Rectangle.this.color);
			double a = color.a;
			this.color = new Color(color.r*a, color.g*a, color.b*a, a);

			double thickness = 0;
			double innerFeather = 0;
			double outerFeather = 0;

			switch (edge) {
				case NONE:
					outerFeather = resolution.scale(context.value(Rectangle.this.outerFeather));
					break;

				case THICKNESS:
					thickness = resolution.scale(context.value(Rectangle.this.thickness));
					innerFeather = resolution.scale(context.value(Rectangle.this.innerFeather));
					outerFeather = resolution.scale(context.value(Rectangle.this.outerFeather));
					break;

				case P_THICKNESS:
					thickness = resolution.scale(context.value(Rectangle.this.thickness)) * halfDiag / 100;
					innerFeather = resolution.scale(context.value(Rectangle.this.innerFeather));
					outerFeather = resolution.scale(context.value(Rectangle.this.outerFeather));
					break;

				case P_THICKNESS_P_FEATHER:
					thickness = resolution.scale(context.value(Rectangle.this.thickness)) * halfDiag / 100;
					innerFeather = resolution.scale(context.value(Rectangle.this.innerFeather)) * halfDiag / 100;
					outerFeather = resolution.scale(context.value(Rectangle.this.outerFeather)) * halfDiag / 100;
					break;
			}

			switch (edge) {
				case NONE:
					halfWidth = size.x*0.5 + outerFeather;
					halfHeight = size.y*0.5 + outerFeather;
					innerMiddleX = 0;
					innerMiddleY = 0;
					outerMiddleX = size.x*0.5;
					outerMiddleY = size.y*0.5;
					break;

				default:
					halfWidth = size.x*0.5 + thickness*0.5 + outerFeather;
					halfHeight = size.y*0.5 + thickness*0.5 + outerFeather;
					innerMiddleX = size.x*0.5 - thickness*0.5;
					innerMiddleY = size.y*0.5 - thickness*0.5;
					outerMiddleX = size.x*0.5 + thickness*0.5;
					outerMiddleY = size.y*0.5 + thickness*0.5;
					break;
			}

			texture = createRectangleTextures(innerMiddleX, innerMiddleY, innerFeather, outerFeather, texSize);
		}
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer original;
		final VideoBounds bounds;

		BlendMode blendMode = context.value(this.blendMode);

		if (blendMode == BlendMode.NONE) {
			bounds = context.getPreviousBounds();
			original = context.createVideoBuffer(bounds);
		} else {
			original = context.doPreviousEffect();
			bounds = original.getBounds();
		}

		if (bounds.isEmpty()) {
			return original;
		}


		RectangleParams params = null;
		IVideoBuffer rectangle = null;
		try {
			final RectangleParams p = params = new RectangleParams();

			rectangle = context.createVideoBuffer(bounds);

			if (p.invert) {
				rectangle.clear(p.color);
			} else {
				rectangle.clear();
			}


			IShaderProgram program = p.invert ? invertSamplerProgram : rectangleSamplerProgram;

			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texInnerX", 0));
			uniforms.add(new GLUniformData("texInnerY", 1));
			uniforms.add(new GLUniformData("texOuter", 2));
			uniforms.add(new GLUniformData("texSize", 4, FloatBuffer.wrap(new float[] {
					p.texSize[0], p.texSize[1], p.texSize[2], p.texSize[2] })));
			uniforms.add(new GLUniformData("texOffset", 4, FloatBuffer.wrap(new float[] {
					(float)(p.texSize[0]*0.5-p.innerMiddleX), (float)(p.texSize[1]*0.5-p.innerMiddleY),
					(float)(p.texSize[2]*0.5-p.outerMiddleX), (float)(p.texSize[2]*0.5-p.outerMiddleY) })));
			uniforms.add(new GLUniformData("center", 2, FloatBuffer.wrap(new float[] {
					(float)(p.center.x-bounds.x), (float)(p.center.y-bounds.y) })));
			uniforms.add(new GLUniformData("color", 4, FloatBuffer.wrap(new float[] {
					(float)p.color.r, (float)p.color.g, (float)p.color.b, (float)p.color.a })));

			Runnable operation = new Runnable() {
				public void run() {
					GL2 gl = context.getGL().getGL2();

					support.ortho2D(bounds);
					gl.glMatrixMode(GL2.GL_MODELVIEW);
					gl.glTranslatef((float)p.center.x, (float)p.center.y, 0);

					gl.glActiveTexture(GL2.GL_TEXTURE0);
					gl.glBindTexture(GL2.GL_TEXTURE_1D, p.texture[0]);
					gl.glActiveTexture(GL2.GL_TEXTURE1);
					gl.glBindTexture(GL2.GL_TEXTURE_1D, p.texture[1]);
					gl.glActiveTexture(GL2.GL_TEXTURE2);
					gl.glBindTexture(GL2.GL_TEXTURE_1D, p.texture[2]);

					double halfWidth = p.halfWidth+2;
					double halfHeight = p.halfHeight+2;
					support.quad2D(-halfWidth, -halfHeight, halfWidth, halfHeight);
				}
			};

			support.useShaderProgram(program, uniforms, operation, GL2.GL_TEXTURE_BIT, rectangle);


			if (blendMode == BlendMode.NONE && p.opacity == 1) {
				IVideoBuffer result = rectangle;
				rectangle = null;
				return result;

			} else {
				return blendSupport.blend(rectangle, original, null, blendMode, p.opacity, context);
			}

		} finally {
			if (original != null) original.dispose();
			if (rectangle != null) rectangle.dispose();
			if (params != null) deleteTextures(params.texture);
		}
	}

	private int[] createRectangleTextures(
			double innerMiddleX, double innerMiddleY, double innerFeather, double outerFeather, int[] texSize) {

		int tex0 = 0, tex1 = 0, tex2 = 0;
		IArray<float[]> data0 = null, data1 = null, data2 = null;
		try {
			if (innerMiddleX <= 0 && innerFeather == 0) {
				tex0 = Texture1D.fromArray(new float[] { 0, 0 }, context);
				texSize[0] = 2;
			} else {
				data0 = createRectangleData(innerFeather);
				tex0 = Texture1D.fromArray(data0, context);
				texSize[0] = data0.getLength();
			}

			if (innerMiddleY <= 0 && innerFeather == 0) {
				tex1 = Texture1D.fromArray(new float[] { 0, 0 }, context);
				texSize[1] = 2;
			} else {
				data1 = createRectangleData(innerFeather);
				tex1 = Texture1D.fromArray(data1, context);
				texSize[1] = data1.getLength();
			}

			data2 = createRectangleData(outerFeather);
			tex2 = Texture1D.fromArray(data2, context);
			texSize[2] = data2.getLength();

			int[] result = new int[] { tex0, tex1, tex2 };
			tex0 = tex1 = tex2 = 0;
			return result;

		} finally {
			if (data0 != null) data0.release();
			if (data1 != null) data1.release();
			if (data2 != null) data2.release();
			if (tex0 != 0) deleteTextures(tex0);
			if (tex1 != 0) deleteTextures(tex1);
			if (tex2 != 0) deleteTextures(tex2);
		}
	}

	private IArray<float[]> createRectangleData(double feather) {
		int halfLen = (int)Math.ceil(feather) + 1;

		IArray<float[]> data = arrayPools.getFloatArray(halfLen*2);
		float[] array = data.getArray();
		int arrayLen = data.getLength();

		for (int i = 0; i < arrayLen; ++i) {
			double r = i + 0.5;
			double d = (r <= halfLen-feather) ? 1
					 : (r >= halfLen+feather) ? 0
					 : 0.5*(1-Math.cos(2*Math.PI*(r-(halfLen+feather))/(4*feather)));

			array[i] = (float) d;
		}

		return data;
	}

	private void deleteTextures(int... texture) {
		GL2 gl = context.getGL().getGL2();
		gl.glDeleteTextures(texture.length, texture, 0);
	}

	private static final String[] createSamplerProgram(boolean invert) {
		return new String[] {
				"uniform sampler1D texInnerX;",
				"uniform sampler1D texInnerY;",
				"uniform sampler1D texOuter;",
				"uniform vec4 texSize;",
				"uniform vec4 texOffset;",
				"uniform vec2 center;",
				"uniform vec4 color;",		// ATIでgl_Colorを使うには頂点シェーダで明示的にglFrontColorを設定する必要がある。
				"",							// ここではそれをせず、uniform変数で色を渡している。
				"void main(void)",
				"{",
				"	vec2 delta = abs(gl_FragCoord.st - center);",
				"	vec4 texCoord = (vec4(delta, delta) + texOffset) / texSize;",
				"",
				"	float a0 = texture1D(texInnerX, texCoord.x).a;",
				"	float a1 = texture1D(texInnerY, texCoord.y).a;",
				"	float a2 = texture1D(texOuter, texCoord.z).a;",
				"	float a3 = texture1D(texOuter, texCoord.w).a;",
				"",
	   invert ? "	gl_FragColor = color * (1.0 - (1.0-a0*a1)*(a2*a3));"
			  : "	gl_FragColor = color * (1.0-a0*a1)*(a2*a3);",
				"}"
		};
	}

	@ShaderSource
	public static final String[] RECTANGLE_SAMPLER = createSamplerProgram(false);

	@ShaderSource
	public static final String[] INVERT_SAMPLER = createSamplerProgram(true);

}
