/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.stylize;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IBlurSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.api.services.IBlurSupport.BlurDimensions;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.Border", category=Categories.STYLIZE)
public class Border {

	public enum Style { SMOOTH, BOX }

	public enum Position { OUTSIDE, INSIDE, BOTH }


	@Property(value="50", min="0", max="100")
	private IAnimatableDouble threshold;

	@Property
	private IAnimatableEnum<Style> style;

	@Property
	private IAnimatableEnum<Position> position;

	@Property(value="3", min="0", max="50")
	private IAnimatableDouble size;

	@Property
	private IAnimatableBoolean accuracy;

	@Property("1,0,0")
	private IAnimatableColor color;

	@Property(value="100", min="0", max="100")
	private IAnimatableDouble opacity;

	@Property
	private IAnimatableBoolean borderOnly;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IBlurSupport blurSupport;

	private final IShaderProgram binarizationProgram;

	private final IShaderProgram outsideProgram;

	private final IShaderProgram outsideBOProgram;

	private final IShaderProgram insideProgram;

	private final IShaderProgram insideBOProgram;

	private final IShaderProgram bothProgram;

	private final IShaderProgram bothBOProgram;

	@Inject
	public Border(
			IVideoEffectContext context, IVideoRenderSupport support,
			IBlurSupport blurSupport, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		this.blurSupport = blurSupport;
		binarizationProgram = shaders.getProgram(Border.class, "BINARIZATION");
		outsideProgram = shaders.getProgram(Border.class, "OUTSIDE");
		outsideBOProgram = shaders.getProgram(Border.class, "OUTSIDE_BORDER_ONLY");
		insideProgram = shaders.getProgram(Border.class, "INSIDE");
		insideBOProgram = shaders.getProgram(Border.class, "INSIDE_BORDER_ONLY");
		bothProgram = shaders.getProgram(Border.class, "BOTH");
		bothBOProgram = shaders.getProgram(Border.class, "BOTH_BORDER_ONLY");
	}

	public VideoBounds getVideoBounds() {
		VideoBounds bounds = context.getPreviousBounds();
		if (bounds.isEmpty()) {
			return bounds;
		}

		Resolution resolution = context.getVideoResolution();
		double size = resolution.scale(context.value(this.size));
		Style style = context.value(this.style);

		if (size > 0) {
			switch (style) {
				case SMOOTH:
					return blurSupport.calcGaussianBlurredBounds(bounds, size, BlurDimensions.BOTH, false);
				case BOX:
					return blurSupport.calcBoxBlurredBounds(bounds, size, BlurDimensions.BOTH, false, 1);
				default:
					throw new RuntimeException("unknown Style: " + style);
			}
		} else {
			return bounds;
		}
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();
		if (source.getBounds().isEmpty()) {
			return source;
		}

		Resolution resolution = context.getVideoResolution();
		double threshold = context.value(this.threshold) / 100;
		Style style = context.value(this.style);
		Position position = context.value(this.position);
		double size = resolution.scale(context.value(this.size));
		boolean accuracy = context.value(this.accuracy);
		Color color = context.value(this.color);
		double opacity = context.value(this.opacity) / 100;
		boolean borderOnly = context.value(this.borderOnly);

		IShaderProgram program;
		switch (position) {
			case OUTSIDE:
				program = borderOnly ? outsideBOProgram : outsideProgram;
				break;
			case INSIDE:
				program = borderOnly ? insideBOProgram : insideProgram;
				break;
			case BOTH:
				program = borderOnly ? bothBOProgram : bothProgram;
				break;
			default:
				throw new RuntimeException("unknown Position: " + position);
		}

		IVideoBuffer binary = null;
		IVideoBuffer blurred = null;
		try {
			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture", 0));
			uniforms.add(new GLUniformData("threshold", (float)threshold));

			if (accuracy && !context.getColorMode().isFloat()) {
				binary = support.createVideoBuffer(source.getBounds(), ColorMode.RGBA16_FLOAT);
			} else {
				binary = support.createVideoBuffer(source.getBounds());
			}

			support.useShaderProgram(binarizationProgram, uniforms, binary, source);

			if (size > 0) {
				switch (style) {
					case SMOOTH:
						blurred = blurSupport.gaussianBlur(binary, size, BlurDimensions.BOTH, false, false);
						break;
					case BOX:
						blurred = blurSupport.boxBlur(binary, size, BlurDimensions.BOTH, false, false, 1);
						break;
					default:
						throw new RuntimeException("unknown Style: " + style);
				}
			} else {
				blurred = binary;
			}

			uniforms.clear();
			uniforms.add(new GLUniformData("texBlurred", 0));
			uniforms.add(new GLUniformData("texSource", 1));
			uniforms.add(new GLUniformData("threshold", (float)threshold));
			uniforms.add(new GLUniformData("size", (float)size));
			uniforms.add(new GLUniformData("color", 4, toFloatBuffer(color.r*opacity, color.g*opacity, color.b*opacity, opacity)));
			return support.useShaderProgram(program, uniforms, null, blurred, source);

		} finally {
			if (blurred != null && blurred != binary) blurred.dispose(); 
			if (binary != null) binary.dispose();
			source.dispose();
		}
	}

	private FloatBuffer toFloatBuffer(double... values) {
		float[] array = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			array[i] = (float)values[i];
		}
		return FloatBuffer.wrap(array);
	}

	@ShaderSource
	public static final String[] BINARIZATION = {
		"uniform sampler2D texture;",
		"uniform float threshold;",
		"",
		"void main(void)",
		"{",
		"	float alpha = texture2D(texture, gl_TexCoord[0].st).a;",
		"	gl_FragColor = (alpha <= threshold) ? vec4(0.0) : vec4(1.0);",
		"}"
	};

	@ShaderSource
	public static final String[] OUTSIDE = {
		"uniform sampler2D texBlurred;",
		"uniform sampler2D texSource;",
		"uniform float threshold;",
		"uniform float size;",
		"uniform vec4 color;",
		"",
		"void main(void)",
		"{",
		"	vec4 source = texture2D(texSource, gl_TexCoord[1].st);",
		"	if (source.a > threshold) {",
		"		gl_FragColor = source + color*(1.0-source.a);",
		"	} else {",
		"		float a = texture2D(texBlurred, gl_TexCoord[0].st).a;",
		"		if (a > 0.00621) {",
		"			a = 1.0+asin(2.0*(a-0.00621)-1.0)*2.0/3.14159265358979323846264;",
		"			a = clamp(a*size*(1.0-size*0.016), 0.0, 1.0);",
		"			gl_FragColor = source + color*(1.0-source.a)*a;",
		"		} else {",
		"			gl_FragColor = source;",
		"		}",
		"	}",
		"}"
	};

	@ShaderSource
	public static final String[] OUTSIDE_BORDER_ONLY = {
		"uniform sampler2D texBlurred;",
		"uniform sampler2D texSource;",
		"uniform float threshold;",
		"uniform float size;",
		"uniform vec4 color;",
		"",
		"void main(void)",
		"{",
		"	vec4 source = texture2D(texSource, gl_TexCoord[1].st);",
		"	if (source.a > threshold) {",
		"		gl_FragColor = color*(1.0-source.a);",
		"	} else {",
		"		float a = texture2D(texBlurred, gl_TexCoord[0].st).a;",
		"		if (a > 0.00621) {",
		"			a = 1.0+asin(2.0*(a-0.00621)-1.0)*2.0/3.14159265358979323846264;",
		"			a = clamp(a*size*(1.0-size*0.016), 0.0, 1.0);",
		"			gl_FragColor = color*(1.0-source.a)*a;",
		"		} else {",
		"			gl_FragColor = vec4(0.0);",
		"		}",
		"	}",
		"}"
	};

	@ShaderSource
	public static final String[] INSIDE = {
		"uniform sampler2D texBlurred;",
		"uniform sampler2D texSource;",
		"uniform float threshold;",
		"uniform float size;",
		"uniform vec4 color;",
		"",
		"void main(void)",
		"{",
		"	vec4 source = texture2D(texSource, gl_TexCoord[1].st);",
		"	if (source.a <= threshold) {",
		"		gl_FragColor = color*(source.a/threshold) + source*(1.0-color.a);",
		"	} else {",
		"		float a = texture2D(texBlurred, gl_TexCoord[0].st).a;",
		"		if (a < 0.99379) {",
		"			a = 1.0-asin(2.0*(a+0.00621)-1.0)*2.0/3.14159265358979323846264;",
		"			a = clamp(a*size*(1.0-size*0.016), 0.0, 1.0);",
		"			gl_FragColor = color*a + source*(1.0-color.a*a);",
		"		} else {",
		"			gl_FragColor = source;",
		"		}",
		"	}",
		"}"
	};

	@ShaderSource
	public static final String[] INSIDE_BORDER_ONLY = {
		"uniform sampler2D texBlurred;",
		"uniform sampler2D texSource;",
		"uniform float threshold;",
		"uniform float size;",
		"uniform vec4 color;",
		"",
		"void main(void)",
		"{",
		"	vec4 source = texture2D(texSource, gl_TexCoord[1].st);",
		"	if (source.a <= threshold) {",
		"		gl_FragColor = color*(source.a/threshold);",
		"	} else {",
		"		float a = texture2D(texBlurred, gl_TexCoord[0].st).a;",
		"		if (a < 0.99379) {",
		"			a = 1.0-asin(2.0*(a+0.00621)-1.0)*2.0/3.14159265358979323846264;",
		"			a = clamp(a*size*(1.0-size*0.016), 0.0, 1.0);",
		"			gl_FragColor = color*a;",
		"		} else {",
		"			gl_FragColor = vec4(0.0);",
		"		}",
		"	}",
		"}"
	};

	@ShaderSource
	public static final String[] BOTH = {
		"uniform sampler2D texBlurred;",
		"uniform sampler2D texSource;",
		"uniform float threshold;",
		"uniform float size;",
		"uniform vec4 color;",
		"",
		"void main(void)",
		"{",
		"	vec4 source = texture2D(texSource, gl_TexCoord[1].st);",
		"	float a = texture2D(texBlurred, gl_TexCoord[0].st).a;",
		"	if (a > 0.00621 && a < 0.99379) {",
		"		if (a < 0.5) {",
		"			a = 1.0+asin(2.0*(a-0.00621)-1.0)*2.0/3.14159265358979323846264;",
		"		} else {",
		"			a = 1.0-asin(2.0*(a+0.00621)-1.0)*2.0/3.14159265358979323846264;",
		"		}",
		"		a = clamp(a*size*(1.0-size*0.016), 0.0, 1.0);",
		"		gl_FragColor = color*a + source*(1.0-color.a*a);",
		"	} else {",
		"		gl_FragColor = source;",
		"	}",
		"}"
	};

	@ShaderSource
	public static final String[] BOTH_BORDER_ONLY = {
		"uniform sampler2D texBlurred;",
		"uniform sampler2D texSource;",
		"uniform float threshold;",
		"uniform float size;",
		"uniform vec4 color;",
		"",
		"void main(void)",
		"{",
		"	vec4 source = texture2D(texSource, gl_TexCoord[1].st);",
		"	float a = texture2D(texBlurred, gl_TexCoord[0].st).a;",
		"	if (a > 0.00621 && a < 0.99379) {",
		"		if (a < 0.5) {",
		"			a = 1.0+asin(2.0*(a-0.00621)-1.0)*2.0/3.14159265358979323846264;",
		"		} else {",
		"			a = 1.0-asin(2.0*(a+0.00621)-1.0)*2.0/3.14159265358979323846264;",
		"		}",
		"		a = clamp(a*size*(1.0-size*0.016), 0.0, 1.0);",
		"		gl_FragColor = color*a;",
		"	} else {",
		"		gl_FragColor = vec4(0.0);",
		"	}",
		"}"
	};

}
