/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.api.services;

import ch.kuramo.javie.api.Color;

/**
 * アンチエイリアスを有効にしてOpenGLの処理を実行するサービスです。
 * 
 * <p>使用例：アンチエイリアスを有効にしてポリゴンを描画します。
 * <blockquote><pre>
 *  &#64;Effect
 *  public class DrawPolygon {
 *  
 *  	private final IVideoEffectContext context;
 *  
 *  	private final IVideoRenderSupport support;
 *  
 *  	private final IAntiAliasSupport aaSupport;
 *  
 *  	&#64;Inject
 *  	public DrawPolygon(IVideoEffectContext context, IVideoRenderSupport support, IAntiAliasSupport aaSupport) {
 *  		this.context = context;
 *  		this.support = support;
 *  		this.aaSupport = aaSupport;
 *  	}
 *  
 *  	public IVideoBuffer doVideoEffect() {
 *  		IVideoBuffer buffer = context.doPreviousEffect();
 *  		final VideoBounds bounds = buffer.getBounds();
 *  		if (bounds.isEmpty()) {
 *  			return buffer;
 *  		}
 *  
 *  		Runnable operation = new Runnable() {
 *  			public void run() {
 *  				support.ortho2D(bounds);
 *  
 *  				<b>// アンチエイリアスを有効にしてポリゴンを描画
 *  				aaSupport.antiAlias(bounds.width, bounds.height, new Runnable() {
 *  					public void run() {
 *  						GL2 gl = context.getGL().getGL2();
 *  						gl.glBegin(GL2.GL_POLYGON);
 *  						gl.glVertex...
 *  						...
 *  						gl.glEnd();
 *  					}
 *  				});</b>
 *  			}
 *  		};
 *  
 *  		try {
 *  			<b>// IVideoRenderSupport.useFramebuffer メソッドを使うと
 *  			// フレームバッファの設定や後始末などの煩わしい処理を記述しなくて済みます。
 *  			support.useFramebuffer(operation, 0, buffer);</b>
 *  
 *  			IVideoBuffer result = buffer;
 *  			buffer = null;
 *  			return result;
 *  
 *  		} finally {
 *  			if (buffer != null) buffer.dispose();
 *  		}
 *  	}
 *  }
 * </pre></blockquote>
 */
public interface IAntiAliasSupport {

	/**
	 * <p>アンチエイリアスを有効にして、引数<code>operation</code>に指定した処理を実行します。
	 * <code>width</code>及び<code>height</code>には作業バッファのサイズを指定します。
	 * この作業バッファ上で<code>operation</code>を実行した後、
	 * 結果の画像を作業バッファから現在のフレームバッファに転送します。
	 * <code>depthTest</code>に<code>true</code>を指定した場合、
	 * デプスバッファ用の作業バッファも用意し、デプステストを有効にして<code>operation</code>を実行します。
	 * <code>clearColor</code>に<code>null</code>以外の値を指定すると、
	 * 指定した色で作業バッファを塗りつぶした後に<code>operation</code>を実行します。</p>
	 * 
	 * <p>このメソッドを呼び出す前にOpenGLのフレームバッファを適切に設定しておく必要があります。</p>
	 * 
	 * <p>アンチエイリアスの現在の実装はMSAAです。この実装にハードウエアが対応していない場合、アンチエイリアスされません。</p>
	 * 
	 * @param width			作業バッファの幅
	 * @param height		作業バッファの高さ
	 * @param depthTest		デプステストを行う場合は true そうでない場合は false
	 * @param clearColor	処理前に塗りつぶす色
	 * @param operation		アンチエイリアスを有効にして行う処理
	 * 
	 * @see #antiAlias(int, int, Runnable)
	 */
	void antiAlias(int width, int height, boolean depthTest, Color clearColor, Runnable operation);

	/**
	 * アンチエイリアスを有効にして、引数<code>operation</code>に指定した処理を実行します。
	 * これは、<code>antiAlias(width, height, false, Color.COLORLESS_TRANSPARENT, operation)</code>と同じです。
	 * 
	 * @param width			作業バッファの幅
	 * @param height		作業バッファの高さ
	 * @param operation		アンチエイリアスを有効にして行う処理
	 * 
	 * @see #antiAlias(int, int, boolean, Color, Runnable)
	 */
	void antiAlias(int width, int height, Runnable operation);

}
