/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.AnimatableVec2d;

public class AnimatableVec2dElementDelegate extends AnimatableValueElementDelegate<Vec2d> {

	protected final double increment;

	protected final int displayPrecision;

	protected final int editorPrecision;

	protected final String unit;

	protected Vec2dCellEditor editor;


	public AnimatableVec2dElementDelegate(
			AnimatableValueElement element, String name, AnimatableVec2d avalue) {

		this(element, name, avalue, null);
	}

	public AnimatableVec2dElementDelegate(
			AnimatableValueElement element, String name, AnimatableVec2d avalue, String unit) {

		this(element, name, avalue, 1, 1, 4, unit);
	}

	public AnimatableVec2dElementDelegate(
			AnimatableValueElement element, String name, AnimatableVec2d avalue,
			double increment, int displayPrecision, int editorPrecision, String unit) {

		super(element, name, avalue);
		this.increment = increment;
		this.displayPrecision = displayPrecision;
		this.editorPrecision = editorPrecision;
		this.unit = unit;
	}

	protected void drawValue(GC gc, int x, int y, int height, boolean focused) {
		if (editor != null) {
			return;
		}

		x = drawValue(gc, x, y, height, focused, value.x, displayPrecision, unit);
		x = drawSeparator(gc, x, y, height, focused);
		x = drawValue(gc, x, y, height, focused, value.y, displayPrecision, unit);
	}

	protected int drawValue(GC gc, int x, int y, int height, int valueIndex, double value) {
		if (editor == null) {
			throw new IllegalStateException("no editor");
		}

		if (this.valueIndex != valueIndex) {
			return drawValue(gc, x, y, height, false, value, displayPrecision, unit);

		} else {
			Rectangle bounds = editor.getTextBounds();
			if (unit != null) {
				setValueForeground(gc, false);
				return drawString(gc, x + bounds.width + (WIN32 ? 0 : 3), y, height, unit);
			} else {
				return x + bounds.width;
			}
		}
	}

	protected int drawSeparator(GC gc, int x, int y, int height, boolean focused) {
		setForeground(gc, focused);
		x = drawString(gc, x + 2, y, height, ",");
		return x + 4;
	}

	protected boolean canDragGestureEdit() {
		return true;
	}

	protected Vec2d dragGesture(double dx, double dy) {
		double[] orig = { originalValue.x, originalValue.y };
		double[] array = { value.x, value.y };

		array[valueIndex] = orig[valueIndex] + increment * dx;

		return new Vec2d(array[0], array[1]);
	}

	public boolean canEdit(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			return canInPlaceEdit;
		}
		return false;
	}

	public CellEditor getCellEditor(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			if (editor == null) {
				editor = new Vec2dCellEditor(element.viewer.getTree(), SWT.SINGLE | SWT.BORDER);
			}
			return editor;
		}
		return null;
	}

	public Object getCellEditorValue(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			double[] array = { value.x, value.y };
			return formatValue(array[valueIndex], editorPrecision);
		}
		return null;
	}

	public void setCellEditorValue(int columnIndex, Object value) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			double[] array = { this.value.x, this.value.y };
			if (value.equals(formatValue(array[valueIndex], editorPrecision))) {
				return;
			}
			try {
				array[valueIndex] = Double.valueOf((String) value);
				modifyValue(new Vec2d(array[0], array[1]));
			} catch (NumberFormatException e) {
				element.viewer.getTree().getDisplay().beep();
			}
		}
	}

	protected class Vec2dCellEditor extends TextCellEditor {

		private boolean activated;


		protected Vec2dCellEditor(Composite parent, int style) {
			super(parent, style);
		}

		protected Control createControl(final Composite parent) {
			final Composite composite = new Composite(parent, SWT.NO_FOCUS | (COCOA ? SWT.NO_BACKGROUND : 0));
			composite.setFont(parent.getFont());
			composite.setBackground(parent.getBackground());
			composite.setLayout(new FormLayout());

			super.createControl(composite);
			text.setBackground(text.getDisplay().getSystemColor(SWT.COLOR_WHITE));

			Rectangle valueBounds = valueArea.get(valueIndex);
			int left = valueBounds.x - valueArea.get(0).x + (WIN32 ? 3 : 0);
			int right = left + Math.max(60, valueBounds.width + 30);

			FormData data = new FormData();
			data.top = new FormAttachment(0, 0);
			data.bottom = new FormAttachment(100, 0);
			data.left = new FormAttachment(0, left);
			data.right = new FormAttachment(0, right);
			text.setLayoutData(data);

			composite.addMouseListener(new MouseAdapter() {
				public void mouseDown(MouseEvent e) {
					parent.forceFocus();

					if (e.button == 1) {
						for (int i = 0, n = valueArea.size(); i < n; ++i) {
							Rectangle r = valueArea.get(i);
							if (r.contains(e.x, e.y)) {
								valueIndex = (i < valueIndex) ? i : i+1;
								parent.update();
								openInPlaceEditor();
								break;
							}
						}
					}
				}
			});

			composite.addPaintListener(new PaintListener() {
				public void paintControl(PaintEvent e) {
					GC gc = e.gc;
					int x = WIN32 ? 5 : 2;
					int height = composite.getBounds().height;

					x = drawValue(gc, x, 0, height, 0, value.x);
					x = drawSeparator(gc, x, 0, height, false);
					x = drawValue(gc, x, 0, height, 1, value.y);
				}
			});

			return composite;
		}

		private Rectangle getTextBounds() {
			return text.getBounds();
		}

		public void activate() {
			super.activate();
			activated = true;
		}

		public void deactivate() {
			super.deactivate();
			if (activated) {
				editor = null;
				dispose();
			}
		}

	}

}
