/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.wizards;

import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.widgets.FrameRateComboUtil;
import ch.kuramo.javie.app.widgets.GridBuilder;
import ch.kuramo.javie.core.FrameDuration;
import ch.kuramo.javie.core.TimeCode;

public class CompositionWizardPage extends WizardPage {

	private static final boolean COCOA = SWT.getPlatform().equals("cocoa");

	// TODO この最大値は暫定。グラフィックカード毎に異なるはず。
	private static final int MAX_SIZE = 8192;

	private CompositionSettings _settings;

	private Text _nameText;

	private Text _widthText;

	private Text _heightText;

	private Button _freezeRatio;

	private Combo _fpsCombo;

	private Text _durationText;

	private Label _duration2;

	private Combo _colorModeCombo;

	private Button _fpsPreserved;

	private Text _mblurAngleText;

	private Text _mblurPhaseText;

	private Text _mblurSamplesText;

	private ModifyListener _modifyListener;

	private int _widthRatio;

	private int _heightRatio;


	public CompositionWizardPage(CompositionSettings settings, boolean newComp) {
		super("CompositionWizardPage", newComp ? "新規コンポジション" : "コンポジション設定", null);
		_settings = settings;
	}

	public void createControl(Composite parent) {
		String[] colorModeItems = {
				"8 bit/チャンネル",
				"16 bit/チャンネル",
				"16 bit/チャンネル（浮動小数点）",
				"32 bit/チャンネル（浮動小数点）"		
		};

		GridBuilder gb = new GridBuilder(parent, 16, true);

							gb.hSpan(4).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "名前:");
		_nameText =			gb.hSpan(12).hAlign(SWT.FILL).hGrab().text(SWT.BORDER, _settings.name);

							gb.hSpan(16).size(10, 10).composite(SWT.NULL);

							gb.hSpan(4).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "幅:");
		_widthText =		gb.hSpan(3).hAlign(SWT.FILL).hGrab().text(SWT.BORDER, String.valueOf(_settings.size.width));
		_freezeRatio =		gb.span(9, 2).hAlign(SWT.LEFT).hGrab().button(SWT.CHECK, "縦横比を固定");

							gb.hSpan(4).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "高さ:");
		_heightText =		gb.hSpan(3).hAlign(SWT.FILL).hGrab().tabAfter(_widthText)
								.text(SWT.BORDER, String.valueOf(_settings.size.height));

							gb.hSpan(4).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "フレームレート:");
		_fpsCombo = 		gb.hSpan(3).hAlign(SWT.FILL).hGrab().combo(SWT.NULL, 0,
								FrameRateComboUtil.getComboItems(), FrameRateComboUtil.toComboItem(_settings.frameDuration));
							gb.hSpan(9).hGrab().label(SWT.NULL, "フレーム／秒");

							gb.hSpan(4).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "デュレーション:");
		_durationText =		gb.hSpan(3).hAlign(SWT.FILL).hGrab()
								.text(SWT.BORDER, TimeCode.toTimeCode(_settings.duration, _settings.frameDuration));
		_duration2 =		gb.hSpan(9).hAlign(SWT.FILL).hGrab().label(SWT.NULL, "");

							// Macでは何となく詰まった感じで見栄えが悪いのでスペースを少しだけあける。
							if (COCOA) gb.hSpan(16).size(0, 0).composite(SWT.NULL);

							gb.hSpan(4).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "色深度:");
		_colorModeCombo =	gb.hSpan(12).hGrab().combo(SWT.READ_ONLY, colorModeItems, _settings.colorMode.ordinal());

							gb.hSpan(16).size(10, 10).composite(SWT.NULL);

							gb.hSpan(4).size(10, 10).composite(SWT.NULL);
		_fpsPreserved =		gb.hSpan(12).hAlign(SWT.LEFT).hGrab().button(SWT.CHECK, "ネスト時にフレームレートを保持");

							gb.hSpan(16).size(10, 10).composite(SWT.NULL);


		Group mblurGroup =	gb.hSpan(16).hAlign(SWT.FILL).hGrab().group(SWT.NONE, "モーションブラー");
		mblurGroup.setLayout(new FillLayout(SWT.HORIZONTAL));
		GridBuilder gb2 = new GridBuilder(mblurGroup, 16, true);

							gb2.hSpan(4).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "シャッター角度:");
		_mblurAngleText = 	gb2.hSpan(3).hAlign(SWT.FILL).hGrab().text(SWT.BORDER, String.valueOf(_settings.motionBlurShutterAngle));
							gb2.hSpan(9).hGrab().label(SWT.NULL, "°");

							gb2.hSpan(4).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "シャッターフェーズ:");
		_mblurPhaseText =	gb2.hSpan(3).hAlign(SWT.FILL).hGrab().text(SWT.BORDER, String.valueOf(_settings.motionBlurShutterPhase));
							gb2.hSpan(9).hGrab().label(SWT.NULL, "°");

							gb2.hSpan(4).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "サンプル数:");
		_mblurSamplesText =	gb2.hSpan(3).hAlign(SWT.FILL).hGrab().text(SWT.BORDER, String.valueOf(_settings.motionBlurSamples));
							gb2.hSpan(9).hGrab().label(SWT.NULL, "サンプル／フレーム");


		_modifyListener = new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				doValidate((Control) e.getSource());
			}
		};

		SelectionListener selectionListener = new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				doValidate((Control) e.getSource());
			}
			public void widgetDefaultSelected(SelectionEvent e) {
				doValidate((Control) e.getSource());
			}
		};

		_nameText.addModifyListener(_modifyListener);

		_widthText.addModifyListener(_modifyListener);
		_heightText.addModifyListener(_modifyListener);

		_freezeRatio.addSelectionListener(selectionListener);

		_fpsCombo.addModifyListener(_modifyListener);
		_fpsCombo.addSelectionListener(selectionListener);

		_durationText.addModifyListener(_modifyListener);

		_mblurAngleText.addModifyListener(_modifyListener);
		_mblurPhaseText.addModifyListener(_modifyListener);
		_mblurSamplesText.addModifyListener(_modifyListener);


		Composite grid = gb.getComposite();
		grid.setTabList(gb.getTabList());

		Composite grid2 = gb2.getComposite();
		grid2.setTabList(gb2.getTabList());

		doValidate(null);
		setControl(grid);
	}

	private void doValidate(Control control) {
		setPageComplete(false);


		String wStr = _widthText.getText().trim();
		int width;
		try {
			width = Integer.parseInt(wStr);
		} catch (NumberFormatException e) {
			width = 0;
		}

		String hStr = _heightText.getText().trim();
		int height;
		try {
			height = Integer.parseInt(hStr);
		} catch (NumberFormatException e) {
			height = 0;
		}

		boolean frozen = _freezeRatio.getSelection();
		if (frozen) {
			if (control == _widthText && width > 0) {
				height = Math.max(1, width * _heightRatio / _widthRatio);
				_heightText.removeModifyListener(_modifyListener);
				_heightText.setText(String.valueOf(height));
				_heightText.addModifyListener(_modifyListener);
			} else if (control == _heightText && height > 0) {
				width = Math.max(1, height * _widthRatio / _heightRatio);
				_widthText.removeModifyListener(_modifyListener);
				_widthText.setText(String.valueOf(width));
				_widthText.addModifyListener(_modifyListener);
			}
		} else if ((control == null || control == _widthText || control == _heightText) && width > 0 && height > 0) {
			int gcd = gcd(width, height);
			_widthRatio = width / gcd;
			_heightRatio = height / gcd;
			_freezeRatio.setText(String.format("縦横比を%d:%dに固定", _widthRatio, _heightRatio));
		}


		String fpsStr = _fpsCombo.getText().trim();
		double fps;
		try {
			fps = Double.parseDouble(fpsStr);
		} catch (NumberFormatException e) {
			fps = 0;
		}

		String durationStr = _durationText.getText().trim();
		long frames = 0;
		if (fps >= 1 && durationStr.length() > 0) {
			Time frameDuration = FrameRateComboUtil.toFrameDuration(fpsStr);
			frames = TimeCode.parseTimeCode(durationStr, frameDuration);

			String timeCode = TimeCode.toTimeCode(Time.fromFrameNumber(frames, frameDuration), frameDuration);
			int base = (int)Math.round((double)frameDuration.timeScale / frameDuration.timeValue);
			boolean drop = (timeCode.indexOf(';') != -1);
			_duration2.setText(String.format("= %s (ベース %d%s) , %dフレーム", timeCode, base, drop ? "ドロップ" : "", frames));

			if (control == _fpsCombo && !durationStr.matches("[\\d]+")) {
				_durationText.removeModifyListener(_modifyListener);
				_durationText.setText(timeCode);
				_durationText.addModifyListener(_modifyListener);
			}
		}


		if (_nameText.getText().trim().length() == 0) {
			setErrorMessage("名前が入力されていません。");
			return;
		}
		if (wStr.length() == 0) {
			setErrorMessage("幅が入力されていません。");
			return;
		}
		if (width < 1 || width > MAX_SIZE) {
			setErrorMessage(String.format("幅には1から%dまでの整数を入力してください。", MAX_SIZE));
			return;
		}
		if (hStr.length() == 0) {
			setErrorMessage("高さが入力されていません。");
			return;
		}
		if (height < 1 || height > MAX_SIZE) {
			setErrorMessage(String.format("高さには1から%dまでの整数を入力してください。", MAX_SIZE));
			return;
		}
		if (fpsStr.length() == 0) {
			setErrorMessage("フレームレートが入力されていません。");
			return;
		}
		// TODO フレームレートの上限はどうする？
		if (fps < 1) {
			setErrorMessage("フレームレートには1以上の数値を入力してください。");
			return;
		}
		if (durationStr.length() == 0) {
			setErrorMessage("デュレーションが入力されていません。");
			return;
		}
		// TODO デュレーションの上限はどうする？
		if (frames <= 0) {
			setErrorMessage("デュレーションは1フレーム以上必要です。");
			return;
		}

		String angleStr = _mblurAngleText.getText().trim();
		double angle = Double.NaN;
		try {
			angle = Double.parseDouble(angleStr);
		} catch (NumberFormatException e) {
		}
		if (Double.isNaN(angle) || angle < 0 || angle > 720) {
			setErrorMessage("シャッター角度には0から720までの数値を入力してください。");
			return;
		}

		String phaseStr = _mblurPhaseText.getText().trim();
		double phase = Double.NaN;
		try {
			phase = Double.parseDouble(phaseStr);
		} catch (NumberFormatException e) {
		}
		if (Double.isNaN(phase) || phase < -360 || phase > 360) {
			setErrorMessage("シャッターフェーズには-360から360までの数値を入力してください。");
			return;
		}

		String samplesStr = _mblurSamplesText.getText().trim();
		int samples = 0;
		try {
			samples = Integer.parseInt(samplesStr);
		} catch (NumberFormatException e) {
		}
		if (samples < 2 || samples > 64) {
			setErrorMessage("サンプル数には2から64までの整数を入力してください。");
			return;
		}


		setErrorMessage(null);
		setPageComplete(true);
	}

	private int gcd(int x, int y) {
		while (y != 0) {
			int t = x % y;
			x = y;
			y = t;
		}
		return x;
	}

	public CompositionSettings getResult() {
		if (!isPageComplete()) {
			return null;
		}

		CompositionSettings result = new CompositionSettings();

		Time frameDuration = FrameRateComboUtil.toFrameDuration(_fpsCombo.getText().trim());
		long frames = TimeCode.parseTimeCode(_durationText.getText().trim(), frameDuration);

		result.name = _nameText.getText().trim();
		result.size = new Size2i(Integer.parseInt(_widthText.getText().trim()),
								Integer.parseInt(_heightText.getText().trim()));
		result.frameDuration = frameDuration;
		result.duration = Time.fromFrameNumber(frames, frameDuration);
		result.colorMode = ColorMode.values()[_colorModeCombo.getSelectionIndex()];
		result.frameDurationPreserved = _fpsPreserved.getSelection();
		result.motionBlurShutterAngle = Double.parseDouble(_mblurAngleText.getText().trim());
		result.motionBlurShutterPhase = Double.parseDouble(_mblurPhaseText.getText().trim());
		result.motionBlurSamples = Integer.parseInt(_mblurSamplesText.getText().trim());
		return result;
	}

}

class CompositionSettings implements Cloneable {

	String name = "コンポ";

	Size2i size = new Size2i(640, 480);

	Time frameDuration = FrameDuration.FPS_29_97;

	Time duration = Time.fromFrameNumber(1800, frameDuration);

	ColorMode colorMode = ColorMode.RGBA8;

	boolean frameDurationPreserved = false;

	double motionBlurShutterAngle = 180;

	double motionBlurShutterPhase = -90;

	int motionBlurSamples = 16;


	CompositionSettings() {
	}

	public CompositionSettings clone() {
		try {
			return (CompositionSettings) super.clone();
		} catch (CloneNotSupportedException e) {
			throw new Error(e);
		}
	}

}
