/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;

import org.mozilla.javascript.Scriptable;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CoreContext;
import ch.kuramo.javie.core.ExpressionScope;
import ch.kuramo.javie.core.Expressionee;
import ch.kuramo.javie.core.Expressioner;
import ch.kuramo.javie.core.WrappedOperation;

import com.google.inject.Inject;
import com.google.inject.Injector;

public abstract class AbstractCoreContextProxy<I extends CoreContext> implements CoreContext {

	private final Class<I> _implClass;

	@Inject
	private Injector _injector;


	public AbstractCoreContextProxy(Class<I> implClass) {
		_implClass = implClass;
	}

	public AbstractCoreContextProxy() {
		Type type = getClass().getGenericSuperclass();
		if (!(type instanceof ParameterizedType)) {
			throw new UnsupportedOperationException();
		}

		@SuppressWarnings("unchecked")
		Class<I> clazz = (Class<I>) ((ParameterizedType) type).getActualTypeArguments()[0];

		_implClass = clazz;
	}


	protected abstract ThreadLocal<I> getThreadLocal();


	protected I getImpl() {
		I impl = getThreadLocal().get();
		if (impl == null) {
			throw new IllegalStateException(
					_implClass.getSimpleName() + " of current thread is not activated.");
		}
		return impl;
	}

	public void activate() {
		I impl = getThreadLocal().get();
		if (impl != null) {
			throw new IllegalStateException(
					_implClass.getSimpleName() + " of current thread has already been activated.");
		}
		impl = _injector.getInstance(_implClass);
		impl.activate();
		getThreadLocal().set(impl);
	}

	public void deactivate() {
		I impl = getImpl();
		impl.deactivate();
		getThreadLocal().set(null);
	}

	public boolean isActive() {
		I impl = getThreadLocal().get();
		return (impl != null && impl.isActive());
	}

	public void reset() {
		getImpl().reset();
	}

	public <T> T saveAndExecute(WrappedOperation<T> operation) {
		return getImpl().saveAndExecute(operation);
	}

	public Time getTime() {
		return getImpl().getTime();
	}

	public void setTime(Time time) {
		getImpl().setTime(time);
	}

	public Composition getComposition() {
		return getImpl().getComposition();
	}

	public void setComposition(Composition comp) {
		getImpl().setComposition(comp);
	}

	public ExpressionScope createInitialExpressionScope(Composition comp) {
		return getImpl().createInitialExpressionScope(comp);
	}

	public <T> T getExpressionElement(Expressionee exprnee) {
		return getImpl().<T>getExpressionElement(exprnee);
	}

	public boolean checkCyclicEvaluation(Expressioner<?> expr) {
		return getImpl().checkCyclicEvaluation(expr);
	}

	public <T> T evaluateExpression(Expressioner<T> expr) {
		return getImpl().evaluateExpression(expr);
	}

	public Scriptable toNativeJavaObject(Object javaObject, Class<?> defaultHint) {
		return toNativeJavaObject(javaObject, defaultHint);
	}

}
