/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.keying;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.effects.TexCoords;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.ChromaKey", category=Categories.KEYING)
public class ChromaKey {

	@Property("1,1,1")
	private IAnimatableColor keyColor;

	@Property(value="0", min="0", max="100")
	private IAnimatableDouble similarity;

	@Property(value="10", min="0", max="100")
	private IAnimatableDouble blend;

	@Property(value="0", min="0", max="100")
	private IAnimatableDouble threshold;

	@Property(value="0", min="0", max="100")
	private IAnimatableDouble cutoff;

	@Property
	private IAnimatableEnum<Smoothing> smoothing;

	@Property
	private IAnimatableBoolean maskOnly;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram keyingProgram;

	private final IShaderProgram maskOnlyProgram;

	private final IShaderProgram smoothingLowProgram;

	private final IShaderProgram smoothingHighProgram;

	@Inject
	public ChromaKey(IVideoEffectContext context, IVideoRenderSupport support, IShaderRegistry shaders) {
		this.context = context;
		this.support = support;

		keyingProgram = shaders.getProgram(KeyingShaders.class, "CHROMA_KEY");
		maskOnlyProgram = shaders.getProgram(KeyingShaders.class, "CHROMA_KEY_MASK_ONLY");
		smoothingLowProgram = shaders.getProgram(KeyingShaders.class, "CHROMA_KEY_SMOOTHING_LOW");
		smoothingHighProgram = shaders.getProgram(KeyingShaders.class, "CHROMA_KEY_SMOOTHING_HIGH");
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer input = context.doPreviousEffect();

		VideoBounds bounds = input.getBounds();
		if (bounds.isEmpty()) {
			return input;
		}

		Color keyColor = context.value(this.keyColor);
		double similarity = context.value(this.similarity) / 100 * 0.4;	// 0.4というのはPEとほぼ同じにするための係数
		double blend = context.value(this.blend) / 100 * 0.4;
		double threshold = context.value(this.threshold) / 100;
		double cutoff = context.value(this.cutoff) / 100;
		Smoothing smoothing = context.value(this.smoothing);
		boolean maskOnly = context.value(this.maskOnly);

		double r = keyColor.r;
		double g = keyColor.g;
		double b = keyColor.b;
		double u = -0.14713*r -0.28886*g +0.436*b;
		double v =  0.615*r -0.51499*g -0.10001*b;


		Set<IVideoBuffer> tmpBuffers = new HashSet<IVideoBuffer>();
		tmpBuffers.add(input);
		try {
			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture", 0));
			uniforms.add(new GLUniformData("keyUV", 2, FloatBuffer.wrap(new float[] { (float)u, (float)v })));
			uniforms.add(new GLUniformData("similarity", (float)similarity));
			uniforms.add(new GLUniformData("blend", (float)blend));
			uniforms.add(new GLUniformData("t_minus_c", (float)(threshold-cutoff)));
			uniforms.add(new GLUniformData("cutoff", (float)cutoff));

			IVideoBuffer keyedOut = null;
			IVideoBuffer smoothMask = null;

			keyedOut = context.createVideoBuffer(bounds);
			tmpBuffers.add(keyedOut);

			IShaderProgram program = maskOnly ? maskOnlyProgram : keyingProgram;

			if (smoothing == Smoothing.NONE || maskOnly) {
				support.useShaderProgram(program, uniforms, keyedOut, input);

			} else {
				smoothMask = context.createVideoBuffer(bounds);
				tmpBuffers.add(smoothMask);

				support.useShaderProgramMRT(program, uniforms, new IVideoBuffer[] { keyedOut, smoothMask }, input);
			}

			float[] neib;
			switch (smoothing) {
				case LOW:
					program = smoothingLowProgram;
					neib = TexCoords.neighboringOffset8(bounds.width, bounds.height);
					break;

				case HIGH:
					program = smoothingHighProgram;
					neib = TexCoords.neighboringOffset24(bounds.width, bounds.height);
					break;

				default:
					tmpBuffers.remove(keyedOut);
					return keyedOut;
			}

			uniforms.clear();
			uniforms.add(new GLUniformData("tex0", 0));
			uniforms.add(new GLUniformData("tex1", 1));
			uniforms.add(new GLUniformData("neib[0]", 2, FloatBuffer.wrap(neib)));

			support.useShaderProgram(program, uniforms, input, keyedOut, smoothMask);

			tmpBuffers.remove(input);
			return input;

		} finally {
			for (IVideoBuffer vb : tmpBuffers) {
				vb.dispose();
			}
		}
	}

}
