/*
 * Copyright (c) 2010,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import java.util.List;
import java.util.regex.Pattern;

import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;

import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.app.widgets.FontUtil;
import ch.kuramo.javie.app.widgets.GridBuilder;
import ch.kuramo.javie.app.widgets.InPlaceEditorShell;
import ch.kuramo.javie.core.AnimatableString;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.Font;
import ch.kuramo.javie.core.services.FontList;

import com.google.inject.Inject;
import com.ibm.icu.text.Normalizer;
import com.ibm.icu.text.Transliterator;

public class TextFontElement extends LayerAnimatableValueElement.ForSimpleProperty {

	public TextFontElement(Element parent, TextLayer layer) {
		super(parent, layer, "font", "フォント");
	}

	protected AnimatableValueElementDelegate<?> createDelegate() {
		AnimatableString avalue = PropertyUtil.getProperty(layer, getProperty());
		return new TextFontElementDelegate(this, name, avalue);
	}

}

class TextFontElementDelegate extends AnimatableStringElementDelegate {

	private static final Transliterator H2K_TRANSLITERATOR = Transliterator.getInstance("Hiragana-Katakana");
	private static final Pattern SPACE_PATTERN = Pattern.compile("\\s");

	private static Point defaultEditorSize = new Point(200, 300);

	@Inject
	private FontList fontList;


	TextFontElementDelegate(
			AnimatableValueElement element, String name, AnimatableString avalue) {

		super(element, name, avalue);
		InjectorHolder.getInjector().injectMembers(this);
	}

	protected void drawValue(GC gc, int x, int y, int height, boolean focused) {
		Font font = fontList.get(value);
		drawValue(gc, x, y, height, focused, (font != null)
				? (font.familyName + " " + font.subfamilyName) : "      ");
	}

	protected void openInPlaceEditor() {
		final Shell shell = InPlaceEditorShell.create(
				element.viewer.getTree(), valueArea.get(valueIndex),
				defaultEditorSize.x, defaultEditorSize.y, true);

		shell.setMinimumSize(100, 100);

		FillLayout fillLayout = new FillLayout();
		fillLayout.marginWidth = 5;
		fillLayout.marginHeight = 5;
		shell.setLayout(fillLayout);

		GridBuilder gb = new GridBuilder(shell, 1, false);
		gb.getLayout().marginWidth = 0;
		gb.getLayout().marginHeight = 0;

		final Text searchText = gb.hAlign(SWT.FILL).hGrab().text(SWT.SEARCH | SWT.ICON_SEARCH, "");
		FontUtil.setCompatibleFont(searchText);

		final Tree fontTree = gb.align(SWT.FILL, SWT.FILL).grab(true, true)
				.tree(SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER | SWT.FULL_SELECTION);
		final TreeViewer fontViewer = new TreeViewer(fontTree);

		// layout() を実行してレイアウトを確定させておかないと、
		// 選択中の要素がツリーの表示領域内に正しく収まらない場合がある。
		shell.layout();

		shell.addDisposeListener(new DisposeListener() {
			public void widgetDisposed(DisposeEvent e) {
				if (!shell.isDisposed()) {
					defaultEditorSize = shell.getSize();
				}
			}
		});

		searchText.addKeyListener(new KeyAdapter() {
			public void keyPressed(KeyEvent e) {
				if (e.keyCode == SWT.ARROW_DOWN) {
					fontTree.setFocus();
				}
			}
		});


		fontViewer.setContentProvider(new FontNameContetProvider());
		fontViewer.setLabelProvider(FontNameLabelProvider.getInstance());
		fontViewer.setInput(fontList.listFamilyNames());

		final String relation = Util.randomId();
		fontViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				Object element = ((TreeSelection) event.getSelection()).getFirstElement();
				if (element instanceof Font) {
					modifyValue(((Font) element).psName, relation);
				}
			}
		});
		fontViewer.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {
				Object element = ((TreeSelection) event.getSelection()).getFirstElement();
				if (element instanceof Font) {
					modifyValue(((Font) element).psName, relation);
					if (!shell.isDisposed()) {
						shell.setVisible(false);
					}
				} else {
					fontViewer.setExpandedState(element,
							!fontViewer.getExpandedState(element));
				}
			}
		});


		final String[] text = new String[] { "" };
		searchText.addModifyListener(new ModifyListener() {
			private long lasttime;
			private boolean scheduled;
			public void modifyText(ModifyEvent e) {
				text[0] = searchText.getText();
				text[0] = Normalizer.normalize(text[0], Normalizer.NFKC);
				text[0] = H2K_TRANSLITERATOR.transliterate(text[0]);
				text[0] = SPACE_PATTERN.matcher(text[0]).replaceAll("").toLowerCase();

				if (!scheduled) {
					long time = System.currentTimeMillis();
					if (time >= lasttime+500) {
						fontViewer.refresh(false);
						lasttime = time;
					} else {
						scheduled = true;
						e.display.timerExec((int)(lasttime+500-time), new Runnable() {
							public void run() {
								fontViewer.refresh(false);
								lasttime = System.currentTimeMillis();
								scheduled = false;
							}
						});
					}
				}
			}
		});
		fontViewer.addFilter(new ViewerFilter() {
			public boolean select(Viewer viewer, Object parentElement, Object element) {
				if (text[0].length() == 0 || element instanceof Font) {
					return true;
				}
				String name = (String) element;
				name = Normalizer.normalize(name, Normalizer.NFKC);
				name = H2K_TRANSLITERATOR.transliterate(name);
				name = SPACE_PATTERN.matcher(name).replaceAll("").toLowerCase();
				return name.contains(text[0]);
			}
		});


		Font currentFont = fontList.get(value);
		if (currentFont != null) {
			List<String> familyNames = fontList.listFamilyNames();
			fontViewer.setExpandedState(currentFont.familyName, true);
			fontViewer.reveal(familyNames.get(familyNames.size()-1));
			fontViewer.reveal(currentFont.familyName);
			fontViewer.setSelection(new TreeSelection(new TreePath(
					new Object[] { currentFont.familyName, currentFont })));
		}


		shell.open();
	}

	public boolean canEdit(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			return false;
		}
		return super.canEdit(columnIndex);
	}

	private class FontNameContetProvider implements ITreeContentProvider {

		public void dispose() {
			// nothing to do
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			// nothing to do
		}

		public Object[] getElements(Object inputElement) {
			return ((List<?>) inputElement).toArray();
		}

		public boolean hasChildren(Object element) {
			return (element instanceof String);
		}

		public Object[] getChildren(Object parentElement) {
			return (parentElement instanceof String)
				? fontList.getFamily((String) parentElement).toArray() : null;
		}

		public Object getParent(Object element) {
			return (element instanceof Font)
				? ((Font) element).familyName : null;
		}
	}

	private static class FontNameLabelProvider extends LabelProvider {

		private static final FontNameLabelProvider instance = new FontNameLabelProvider();

		public static FontNameLabelProvider getInstance() {
			return instance;
		}

		public String getText(Object element) {
			return (element instanceof Font)
				? ((Font) element).subfamilyName : super.getText(element);
		}
	}

}
