/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.mmd2javie;

import java.io.File;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.handlers.HandlerUtil;
import org.mozilla.javascript.ContextFactory;

import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.UIUtil;
import ch.kuramo.javie.app.project.NewLayerCompositionOperation;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.views.ProjectView;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Folder;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.services.ProjectElementFactory;

import com.google.inject.Inject;

public class MMD2JavieHandler extends AbstractHandler {

	@Inject
	private ProjectElementFactory elementFactory;

	@Inject
	private ContextFactory contextFactory;


	public MMD2JavieHandler() {
		InjectorHolder.getInjector().injectMembers(this);
	}

	public Object execute(ExecutionEvent event) throws ExecutionException {
		IWorkbenchWindow window = HandlerUtil.getActiveWorkbenchWindowChecked(event);

		ProjectManager pm = ProjectManager.forWorkbenchWindow(window);
		if (pm == null) {
			return null;
		}

		File file = showFileDialog(window);
		if (file == null) {
			return null;
		}

		Project project = null;
		try {
			project = elementFactory.newProject();

			ProgressMonitorDialog dialog = new ProgressMonitorDialog(window.getShell()) {
				protected int getShellStyle() {
					return super.getShellStyle() | SWT.SHEET;
				}
			};

			MMD2Javie mmd2javie = new MMD2Javie(contextFactory);
			if (mmd2javie.run(file, project, dialog)) {
				CompositionItem compItem = (CompositionItem) project.getItems().get(0);

				ProjectView projectView = UIUtil.showView(window.getActivePage(), ProjectView.ID);
				Folder parent = projectView.getItemTreeInsertionPoint();
				pm.postOperation(new NewLayerCompositionOperation(pm, parent,
						compItem.getName(), (LayerComposition)compItem.getComposition(), "MMD2AE JSXファイルの読み込み"));
			}

		} finally {
			if (project != null) {
				for (Item i : project.getItems()) {
					i.dispose();
				}
			}
		}

		return null;
	}

	private File showFileDialog(IWorkbenchWindow window) {
		String[] filterNames = new String[] { "JSXファイル (*.jsx)", "すべてのファイル" };
		String[] filterExtensions = new String[] { "*.jsx", "*" };

		String platform = SWT.getPlatform();
		if (platform.equals("win32") || platform.equals("wpf")) {
			filterExtensions[filterExtensions.length-1] = "*.*";
		}

		FileDialog dialog = new FileDialog(window.getShell(), SWT.OPEN | SWT.SHEET);
		dialog.setFilterNames(filterNames);
		dialog.setFilterExtensions(filterExtensions);

		if (dialog.open() == null) {
			return null;
		}

		return new File(dialog.getFilterPath(), dialog.getFileName());
	}

}
