/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.swt.SWT;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.actions.RetargetAction;
import org.eclipse.ui.actions.ActionFactory.IWorkbenchAction;
import org.eclipse.ui.application.ActionBarAdvisor;
import org.eclipse.ui.application.IActionBarConfigurer;

import ch.kuramo.javie.app.actions.AVIOutputAction;
import ch.kuramo.javie.app.actions.NewCompositionAction;
import ch.kuramo.javie.app.actions.NewFileItemsAction;
import ch.kuramo.javie.app.actions.NewFolderAction;
import ch.kuramo.javie.app.actions.NewImageSequenceAction;
import ch.kuramo.javie.app.actions.NewProjectAction;
import ch.kuramo.javie.app.actions.NewSolidColorItemAction;
import ch.kuramo.javie.app.actions.OpenProjectAction;
import ch.kuramo.javie.app.actions.QTMovieOutputAction;
import ch.kuramo.javie.app.actions.RedoAction;
import ch.kuramo.javie.app.actions.SaveAction;
import ch.kuramo.javie.app.actions.SaveAsAction;
import ch.kuramo.javie.app.actions.SaveCopyAction;
import ch.kuramo.javie.app.actions.ShowConsoleViewAction;
import ch.kuramo.javie.app.actions.ShowProjectViewAction;
import ch.kuramo.javie.app.actions.UndoAction;
import ch.kuramo.javie.app.player.GLCanvasFactory;
import ch.kuramo.javie.core.EffectDescriptor;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.EffectRegistry;

import com.google.inject.Inject;

public class ApplicationActionBarAdvisor extends ActionBarAdvisor {

	private static final boolean COCOA = SWT.getPlatform().equals("cocoa");


	private NewProjectAction newProjectAction;

	private NewCompositionAction newCompositionAction;

	private NewSolidColorItemAction newSolidColorItemAction;

	private NewFolderAction newFolderAction;

	private OpenProjectAction openProjectAction;

	private SaveAction saveAction;

	private SaveAsAction saveAsAction;

	private SaveCopyAction saveCopyAction;

	private NewFileItemsAction newFileItemsAction;

	private NewImageSequenceAction newImageSequenceAction;

	private RetargetAction aviOutputAction;

	private RetargetAction qtMovieOutputAction;

	private RetargetAction sequenceOutputAction;

	private RetargetAction waveOutputAction;

	private IWorkbenchAction quitAction;

	private UndoAction undoAction;

	private RedoAction redoAction;

	private IWorkbenchAction cutAction;

	private IWorkbenchAction copyAction;

	private IWorkbenchAction pasteAction;

	private IWorkbenchAction deleteAction;

	private RetargetAction duplicateAction;

	private IWorkbenchAction selectAllAction;

	private RetargetAction newLightLayerAction;

	private RetargetAction newCameraLayerAction;

	private RetargetAction newNullLayerAction;

	private RetargetAction newTextLayerAction;

	private RetargetAction splitLayersAction;

	private IWorkbenchAction newWindowAction;

	private ShowProjectViewAction showProjectViewAction;

	private ShowConsoleViewAction showConsoleViewAction;

	private IWorkbenchAction preferencesAction;

	private IWorkbenchAction aboutAction;

	private MenuManager recentProjectsMenu;

	@Inject
	private EffectRegistry effectRegistry;

	private final List<MenuManager> effectCategorySubMenus = Util.newList();

	private final List<Action> defaultCategoryEffectActions = Util.newList();


	public ApplicationActionBarAdvisor(IActionBarConfigurer configurer) {
		super(configurer);
		InjectorHolder.getInjector().injectMembers(this);
	}

	protected void makeActions(IWorkbenchWindow window) {
		newProjectAction = new NewProjectAction(window);
		register(newProjectAction);

		newCompositionAction = new NewCompositionAction(window);
		register(newCompositionAction);

		newSolidColorItemAction = new NewSolidColorItemAction(window);
		register(newSolidColorItemAction);

		newFolderAction = new NewFolderAction(window);
		register(newFolderAction);

		openProjectAction = new OpenProjectAction(window);
		register(openProjectAction);

		saveAction = new SaveAction(window);
		register(saveAction);

		saveAsAction = new SaveAsAction(window);
		register(saveAsAction);

		saveCopyAction = new SaveCopyAction(window);
		register(saveCopyAction);

		newFileItemsAction = new NewFileItemsAction(window);
		register(newFileItemsAction);

		newImageSequenceAction = new NewImageSequenceAction(window);
		register(newImageSequenceAction);

		if (AVIOutputAction.isAvailable()) {
			aviOutputAction = new RetargetAction(CommandIds.AVI_OUTPUT, "AVI...");
			register(aviOutputAction);
			window.getPartService().addPartListener(aviOutputAction);
		}

		if (QTMovieOutputAction.isAvailable()) {
			qtMovieOutputAction = new RetargetAction(CommandIds.QTMOVIE_OUTPUT, "QuickTime ムービー...");
			register(qtMovieOutputAction);
			window.getPartService().addPartListener(qtMovieOutputAction);
		}

		sequenceOutputAction = new RetargetAction(CommandIds.SEQUENCE_OUTPUT, "イメージシーケンス...");
		register(sequenceOutputAction);
		window.getPartService().addPartListener(sequenceOutputAction);

		waveOutputAction = new RetargetAction(CommandIds.WAVE_OUTPUT, "Wave...");
		register(waveOutputAction);
		window.getPartService().addPartListener(waveOutputAction);

		if (!COCOA) {
			quitAction = ActionFactory.QUIT.create(window);
			register(quitAction);
		}

		undoAction = new UndoAction(window);
		register(undoAction);

		redoAction = new RedoAction(window);
		register(redoAction);

		cutAction = ActionFactory.CUT.create(window);
		register(cutAction);

		copyAction = ActionFactory.COPY.create(window);
		register(copyAction);

		pasteAction = ActionFactory.PASTE.create(window);
		register(pasteAction);

		deleteAction = ActionFactory.DELETE.create(window);
		register(deleteAction);

		duplicateAction = new RetargetAction(CommandIds.DUPLICATE, "Duplicate");
		duplicateAction.setActionDefinitionId(CommandIds.DUPLICATE);
		register(duplicateAction);
		window.getPartService().addPartListener(duplicateAction);

		selectAllAction = ActionFactory.SELECT_ALL.create(window);
		register(selectAllAction);

		newLightLayerAction = new RetargetAction(CommandIds.NEW_LIGHT_LAYER, "ライト");
		register(newLightLayerAction);
		window.getPartService().addPartListener(newLightLayerAction);

		newCameraLayerAction = new RetargetAction(CommandIds.NEW_CAMERA_LAYER, "カメラ");
		register(newCameraLayerAction);
		window.getPartService().addPartListener(newCameraLayerAction);

		newNullLayerAction = new RetargetAction(CommandIds.NEW_NULL_LAYER, "ヌルオブジェクト");
		register(newNullLayerAction);
		window.getPartService().addPartListener(newNullLayerAction);

		newTextLayerAction = new RetargetAction(CommandIds.NEW_TEXT_LAYER, "テキスト");
		register(newTextLayerAction);
		window.getPartService().addPartListener(newTextLayerAction);

		splitLayersAction = new RetargetAction(CommandIds.SPLIT_LAYERS, "レイヤーを分割");
		splitLayersAction.setActionDefinitionId(CommandIds.SPLIT_LAYERS);
		register(splitLayersAction);
		window.getPartService().addPartListener(splitLayersAction);

		GLCanvasFactory glCanavsFactory = GLCanvasFactory.getFactory();
		if (!glCanavsFactory.isPoolMode()) {
			newWindowAction = ActionFactory.OPEN_NEW_WINDOW.create(window);
			register(newWindowAction);
		}

		showProjectViewAction = new ShowProjectViewAction(window);
		register(showProjectViewAction);

		showConsoleViewAction = new ShowConsoleViewAction(window);
		register(showConsoleViewAction);

		preferencesAction = ActionFactory.PREFERENCES.create(window);
		register(preferencesAction);

		aboutAction = ActionFactory.ABOUT.create(window);
		register(aboutAction);


		recentProjectsMenu = RecentProjects.createMenuManager(window);

		makeEffectActions(window);
	}

	private void makeEffectActions(IWorkbenchWindow window) {
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IExtensionPoint point = registry.getExtensionPoint("ch.kuramo.javie.api.effectCategory");
		if (point == null) {
			return;
		}

		Map<String, MenuManager> subMenus = Util.newMap();

		for (IExtension ext : point.getExtensions()) {
			for (IConfigurationElement cfgElem : ext.getConfigurationElements()) {
				if ("effect-category".equals(cfgElem.getName())) {
					String category = cfgElem.getAttribute("category");
					String name = cfgElem.getAttribute("name").replaceAll("&", "&&");

					MenuManager subMenu = new MenuManager(name);
					subMenus.put(category, subMenu);
					effectCategorySubMenus.add(subMenu);
				}
			}
		}

		Collections.sort(effectCategorySubMenus, new Comparator<MenuManager>() {
			public int compare(MenuManager o1, MenuManager o2) {
				return o1.getMenuText().compareTo(o2.getMenuText());
			}
		});

		for (EffectDescriptor ed : effectRegistry.getEffectDescriptors()) {
			RetargetAction action = new RetargetAction("EFFECT." + ed.getType(), ed.getLabel().replaceAll("&", "&&"));
			//register(action);
			window.getPartService().addPartListener(action);

			MenuManager subMenu = subMenus.get(ed.getCategory());
			if (subMenu != null) {
				subMenu.add(action);
			} else {
				defaultCategoryEffectActions.add(action);
			}
		}

		Collections.sort(defaultCategoryEffectActions, new Comparator<Action>() {
			public int compare(Action o1, Action o2) {
				return o1.getText().compareTo(o2.getText());
			}
		});
	}

	protected void fillMenuBar(IMenuManager menuBar) {
		MenuManager fileMenu = new MenuManager("&File", IWorkbenchActionConstants.M_FILE);
		MenuManager fileNewMenu = new MenuManager("新規");
		MenuManager inputMenu = new MenuManager("読み込み", "ch.kuramo.javie.menu.input");
		MenuManager outputMenu = new MenuManager("書き出し");
		MenuManager editMenu = new MenuManager("&Edit", IWorkbenchActionConstants.M_EDIT);
		MenuManager layerMenu = new MenuManager("&Layer" /* TODO idが必要？ */);
		MenuManager layerNewMenu = new MenuManager("新規");
		MenuManager effectMenu = new MenuManager("&Effect" /* TODO idが必要？ */);
		MenuManager windowMenu = new MenuManager("&Window", IWorkbenchActionConstants.M_WINDOW);
		MenuManager helpMenu = new MenuManager("&Help", IWorkbenchActionConstants.M_HELP);

		menuBar.add(fileMenu);
		menuBar.add(editMenu);
		menuBar.add(layerMenu);
		menuBar.add(effectMenu);
		menuBar.add(windowMenu);
		menuBar.add(new GroupMarker(IWorkbenchActionConstants.MB_ADDITIONS));
		menuBar.add(helpMenu);

		// File
		fileMenu.add(fileNewMenu);
		fileMenu.add(openProjectAction);
		fileMenu.add(recentProjectsMenu);
		fileMenu.add(new Separator());
		fileMenu.add(saveAction);
		fileMenu.add(saveAsAction);
		fileMenu.add(saveCopyAction);
		fileMenu.add(new Separator());
		fileMenu.add(inputMenu);
		fileMenu.add(outputMenu);
		if (quitAction != null) {
			fileMenu.add(new Separator());
			fileMenu.add(quitAction);
		}

		// File/New
		fileNewMenu.add(newProjectAction);
		fileNewMenu.add(newCompositionAction);
		fileNewMenu.add(newSolidColorItemAction);
		fileNewMenu.add(newFolderAction);

		// File/Input
		inputMenu.add(newFileItemsAction);
		inputMenu.add(newImageSequenceAction);
		inputMenu.add(new Separator());
		inputMenu.add(new GroupMarker(IWorkbenchActionConstants.MB_ADDITIONS));

		// File/Output
		if (aviOutputAction != null) outputMenu.add(aviOutputAction);
		if (qtMovieOutputAction != null) outputMenu.add(qtMovieOutputAction);
		outputMenu.add(sequenceOutputAction);
		outputMenu.add(waveOutputAction);

		// Edit
		editMenu.add(undoAction);
		editMenu.add(redoAction);
		editMenu.add(new Separator());
		editMenu.add(cutAction);
		editMenu.add(copyAction);
		editMenu.add(pasteAction);
		editMenu.add(deleteAction);
		editMenu.add(duplicateAction);
		editMenu.add(selectAllAction);

		// Layer
		layerMenu.add(layerNewMenu);
		layerMenu.add(splitLayersAction);

		// Layer/New
		layerNewMenu.add(newLightLayerAction);
		layerNewMenu.add(newCameraLayerAction);
		layerNewMenu.add(newNullLayerAction);
		layerNewMenu.add(newTextLayerAction);

		// Effect
		for (MenuManager effectCategoryMenu : effectCategorySubMenus) {
			effectMenu.add(effectCategoryMenu);
		}
		for (Action effectAction : defaultCategoryEffectActions) {
			effectMenu.add(effectAction);
		}

		// Window
		if (newWindowAction != null) {
			windowMenu.add(newWindowAction);
			windowMenu.add(new Separator());
		}
		windowMenu.add(showProjectViewAction);
		windowMenu.add(showConsoleViewAction);
		windowMenu.add(new Separator());
		windowMenu.add(preferencesAction);

		// Help
		helpMenu.add(new GroupMarker(IWorkbenchActionConstants.MB_ADDITIONS));
		helpMenu.add(new Separator());
		helpMenu.add(aboutAction);
	}

	protected void fillStatusLine(IStatusLineManager statusLine) {
		GLCanvasFactory glCanavsFactory = GLCanvasFactory.getFactory();
		if (glCanavsFactory.isPoolMode()) {
			statusLine.add(glCanavsFactory.createControlContribution());
		}
	}

}
