/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.actions;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeColumnViewerLabelProvider;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;

import ch.kuramo.javie.app.ColorUtil;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.project.ReplaceLayerItemOperation;
import ch.kuramo.javie.app.views.ItemLabelProvider;
import ch.kuramo.javie.app.views.ItemTreeContentProvider;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.app.views.layercomp.LayerElement;
import ch.kuramo.javie.app.widgets.FontUtil;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Folder;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.ItemLayer;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.MediaInputPlaceholder;
import ch.kuramo.javie.core.MediaItem;
import ch.kuramo.javie.core.MediaItemLayer;

public class ReplaceLayerItemAction extends Action {

	private final LayerCompositionView view;

	private ItemLayer<?> itemLayer;

	public ReplaceLayerItemAction(LayerCompositionView view, TreeViewer viewer) {
		super("ソースの置き換え...");
		this.view = view;

		//setId(CommandIds.REPLACE_LAYER_ITEM);
		//setActionDefinitionId(CommandIds.REPLACE_LAYER_ITEM);
		//setImageDescriptor(Activator.getImageDescriptor("/icons/replace_layer_item.png"));

		updateState(viewer.getSelection());

		viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				updateState(event.getSelection());
			}
		});
	}

	private void updateState(ISelection selection) {
		setEnabled(false);
		itemLayer = null;

		Object element = null;

		if (selection instanceof IStructuredSelection) {
			IStructuredSelection ss = (IStructuredSelection) selection;
			if (ss.size() == 1) {
				element = ss.getFirstElement();
			}
		}

		if (element instanceof LayerElement) {
			Layer layer = ((LayerElement) element).layer;
			if (layer instanceof ItemLayer<?>) {
				itemLayer = (ItemLayer<?>) layer;
				setEnabled(true);
			}
		}
	}

	public void run() {
		if (itemLayer != null) {
			ProjectManager pm = ProjectManager.forWorkbenchWindow(view.getSite().getWorkbenchWindow());
			LayerComposition comp = (LayerComposition)view.getCompositionItem().getComposition();

			ReplaceLayerItemDialog dialog = new ReplaceLayerItemDialog(view.getSite().getShell(), pm, comp, itemLayer);
			if (dialog.open() == IDialogConstants.OK_ID) {
				@SuppressWarnings("unchecked")
				ItemLayer<Item> layer = (ItemLayer<Item>)itemLayer;
				pm.postOperation(new ReplaceLayerItemOperation<Item>(pm, layer, dialog.getSelectedItem()));
			}
		}
	}

}

class ReplaceLayerItemDialog extends Dialog {

	private static final boolean COCOA = SWT.getPlatform().equals("cocoa");


	private final ProjectManager projectManager;

	private final LayerComposition composition;

	private final ItemLayer<?> layer;

	private TreeViewer treeViewer;

	private Label messageLabel;

	private Item selectedItem;


	ReplaceLayerItemDialog(Shell parentShell, ProjectManager pm, LayerComposition composition, ItemLayer<?> layer) {
		super(parentShell);
		this.projectManager = pm;
		this.composition = composition;
		this.layer = layer;
	}

	protected Point getInitialSize() {
		return new Point(700, 400);
	}

	protected boolean isResizable() {
		return true;
	}

	protected void configureShell(Shell newShell) {
		super.configureShell(newShell);
		newShell.setText("ソースの置き換え");
	}

	protected Control createDialogArea(Composite parent) {
		Composite composite = (Composite) super.createDialogArea(parent);

		GridLayout layout = new GridLayout(1, false);
		layout.marginWidth = 10;
		layout.marginHeight = 0;
		layout.marginTop = 10;
		layout.verticalSpacing = 10;
		composite.setLayout(layout);


		treeViewer = new TreeViewer(composite, SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);
		treeViewer.setContentProvider(new ItemTreeContentProvider());

		Tree tree = treeViewer.getTree();
		tree.setHeaderVisible(true);
		tree.setLinesVisible(!COCOA);

		FontUtil.setCompatibleFont(tree);
		tree.setBackground(ColorUtil.tableBackground());

		if (COCOA) {
			final Color lineColor = ColorUtil.tableRowLine();

			tree.addListener(SWT.EraseItem, new Listener() {
				public void handleEvent(Event event) {
					int x = event.x;
					int y = event.y + event.height - 1;
					GC gc = event.gc;
					gc.setForeground(lineColor);
					gc.drawLine(x, y, x + event.width, y);
				}
			});
		}


		createViewerColumn(200, null, "名前");

		// TODO ラベルのアイコン
		Image labelIcon = null;
		createViewerColumn( 25, labelIcon, "");

		createViewerColumn( 90, null, "種類");
		createViewerColumn( 60, null, "サイズ");
		createViewerColumn( 90, null, "ﾃﾞｭﾚｰｼｮﾝ");
		createViewerColumn(600, null, "ファイルパス");


		treeViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				update((TreeSelection)event.getSelection());
			}
		});

		treeViewer.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {
				if (selectedItem != null) {
					setReturnCode(OK);
					close();
				}
			}
		});


		messageLabel = new Label(composite, SWT.NULL);


		GridData gridData = new GridData();
		gridData.horizontalAlignment = SWT.FILL;
		gridData.verticalAlignment = SWT.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		tree.setLayoutData(gridData);

		gridData = new GridData();
		gridData.horizontalAlignment = SWT.FILL;
		messageLabel.setLayoutData(gridData);


		treeViewer.setInput(projectManager);
		return composite;
	}

	public void create() {
		super.create();
		Button okButton = getButton(IDialogConstants.OK_ID);
		okButton.setEnabled(false);
	}

	private TreeViewerColumn createViewerColumn(int width, Image image, String name) {
		TreeViewerColumn viewerColumn = new TreeViewerColumn(treeViewer, SWT.NONE);
		TreeColumn column = viewerColumn.getColumn();
		column.setWidth(width);
		column.setImage(image);
		column.setText(name);
		column.setMoveable(true);

		viewerColumn.setLabelProvider(new TreeColumnViewerLabelProvider(new ItemLabelProvider()));

		return viewerColumn;
	}

	private void update(TreeSelection selection) {
		Button okButton = getButton(IDialogConstants.OK_ID);
		okButton.setEnabled(false);
		selectedItem = null;

		// SWT.SINGLE を指定しているので、これは無いはずだが。
		if (selection.size() != 1) {
			messageLabel.setText("");
			return;
		}

		Object element = selection.getFirstElement();
		if (element instanceof Folder) {
			messageLabel.setText("");
			return;
		}

		Item item = (Item) element;

		if (item == layer.getItem()) {
			messageLabel.setText(String.format("“%s” は現在のソースです。", item.getName()));
			return;
		}

		if (layer instanceof MediaItemLayer) {
			if (!(item instanceof MediaItem)) {
				messageLabel.setText("ソースを置き換えようとしているレイヤーは、この種類のアイテムをソースにすることができません。");
				return;
			}

			if (item instanceof CompositionItem && ReplaceLayerItemOperation
					.checkCyclicComposition(composition, (CompositionItem)item)) {

				messageLabel.setText(String.format(
						"コンポジションの循環が発生するため、“%s” をソースにすることはできません。", item.getName()));
				return;
			}

			if (((MediaItem)item).getMediaInput() instanceof MediaInputPlaceholder) {
				messageLabel.setText(String.format(
						"ファイルが見つからないため、“%s” をソースにすることはできません。", item.getName()));
				return;
			}
		} else {
			// MediaItemLayer以外のItemLayerはまだ無い。
			return;
		}

		selectedItem = item;
		messageLabel.setText("");
		okButton.setEnabled(true);
	}

	public Item getSelectedItem() {
		return selectedItem;
	}

}
