/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import java.util.List;
import java.util.Map;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;

import ch.kuramo.javie.api.BlendMode;
import ch.kuramo.javie.app.EnumLabels;
import ch.kuramo.javie.app.ImageUtil;
import ch.kuramo.javie.app.project.ModifyLayerBlendModeOperation;
import ch.kuramo.javie.app.project.ModifyLayerParentOperation;
import ch.kuramo.javie.app.project.ModifyLayerTrackMatteOperation;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.project.RenameLayerOperation;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.app.widgets.WSWin32;
import ch.kuramo.javie.core.CameraLayer;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.ItemLayer;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.LightLayer;
import ch.kuramo.javie.core.LightType;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.NullLayer;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.TrackMatte;
import ch.kuramo.javie.core.TransformableLayer;
import ch.kuramo.javie.core.Util;

public class LayerElement extends Element {

	private static final int[] UNDERLINE_DASH = new int[] { 2, 2 };

	private static final boolean COCOA = SWT.getPlatform().equals("cocoa");

	private static final boolean WIN32 = SWT.getPlatform().equals("win32");


	public final Layer layer;

	private final LayerComposition composition;

	private final List<Element> children = Util.newList();

	private final Map<Class<? extends Element>, Element> childrenMap = Util.newMap();

	private final SwitchGroup showHideColumnSwitches = new SwitchGroup();

	private final SwitchGroup switchesColumnSwitches = new SwitchGroup();

	private TextCellEditor nameEditor;

	private Rectangle modeArea;

	private Rectangle trackMatteArea;

	private Rectangle parentArea;


	public LayerElement(TreeViewer viewer, Layer layer) {
		super(viewer);
		this.layer = layer;
		this.composition = (LayerComposition) viewer.getData(LayerCompositionView.LAYER_COMPOSITION);

		if (WSWin32.isXPThemed()) {
			showHideColumnSwitches.setMarginLeft(11);
			switchesColumnSwitches.setMarginLeft(11);
		}
	}

	public void createSwitches() {
		showHideColumnSwitches.removeAll();
		switchesColumnSwitches.removeAll();

		ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);

		if (LayerNature.isVideoNature(layer)) {
			showHideColumnSwitches.add(new VideoSwitch(pm, layer));
		} else {
			showHideColumnSwitches.addPlaceholder();
		}

		if (LayerNature.isAudioNature(layer)) {
			showHideColumnSwitches.add(new AudioSwitch(pm, layer));
		} else {
			showHideColumnSwitches.addPlaceholder();
		}

		switchesColumnSwitches.add(new ShySwitch(pm, layer));

		if (LayerNature.isCTCRNature(layer)) {
			switchesColumnSwitches.add(new CTCRSwitch(pm, layer));
		} else {
			switchesColumnSwitches.addPlaceholder();
		}

		if (LayerNature.isQualityNature(layer)) {
			switchesColumnSwitches.add(new QualitySwitch(pm, layer));
		} else {
			switchesColumnSwitches.addPlaceholder();
		}

		if (layer instanceof EffectableLayer) {
			switchesColumnSwitches.add(new EffectsSwitch(pm, (EffectableLayer) layer));
		} else {
			switchesColumnSwitches.addPlaceholder();
		}

		if (LayerNature.isFrameBlendNature(layer)) {
			switchesColumnSwitches.add(new FrameBlendSwitch(pm, layer));
		} else {
			switchesColumnSwitches.addPlaceholder();
		}

		if (LayerNature.isMotionBlurNature(layer)) {
			switchesColumnSwitches.add(new MotionBlurSwitch(pm, layer));
		} else {
			switchesColumnSwitches.addPlaceholder();
		}

		if (layer instanceof TransformableLayer && LayerNature.isThreeDNature(layer)) {
			switchesColumnSwitches.add(new ThreeDSwitch(pm, (TransformableLayer) layer));
		} else {
			switchesColumnSwitches.addPlaceholder();
		}
	}

	public Image getColumnImage(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				if (layer instanceof ItemLayer<?>) {
					return ImageUtil.getItemIcon(((ItemLayer<?>) layer).getItem());
				}
				if (layer instanceof LightLayer) {
					return ImageUtil.getLightIcon();
				}
				if (layer instanceof CameraLayer) {
					return ImageUtil.getCameraIcon();
				}
				if (layer instanceof NullLayer) {
					return ImageUtil.getNullIcon();
				}
				if (layer instanceof TextLayer) {
					return ImageUtil.getTextIcon();
				}

				// TODO その他のレイヤーのアイコン
				return PlatformUI.getWorkbench().getSharedImages().getImage(ISharedImages.IMG_OBJ_ELEMENT);

			default:
				return null;
		}
	}

	public String getColumnText(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL: {
				String name;
				if (layer instanceof ItemLayer<?>) {
					String itemName = ((ItemLayer<?>) layer).getItem().getName();
					if (isLayerNameMode()) {
						name = layer.getName();
						if (name.equals(itemName)) {
							name = String.format("[%s]", name);
						}
					} else {
						name = itemName;
					}
				} else {
					name = layer.getName();
				}
				return name;
			}

			default:
				return null;
		}
	}

	private boolean isLayerNameMode() {
		return Boolean.TRUE.equals(viewer.getData(LayerCompositionView.LAYER_NAME_MODE));
	}

	private MediaLayer getTrackMatteLayer() {
		if (layer instanceof MediaLayer) {
			List<Layer> layers = composition.getLayers();
			int layerIndex = layers.indexOf(layer);
			if (layerIndex + 1 < layers.size()) {
				Layer matteLayer = layers.get(layerIndex + 1);
				if (matteLayer instanceof MediaLayer) {
					return (MediaLayer) matteLayer;
				}
			}
		}
		return null;
	}

	private <E extends Element> E getChildElement(Class<E> clazz) {
		@SuppressWarnings("unchecked")
		E element = (E) childrenMap.get(clazz);
		if (element == null) {
			try {
				element = clazz.getConstructor(getClass()).newInstance(this);
			} catch (Exception e) {
				throw new JavieRuntimeException(e);
			}
			childrenMap.put(clazz, element);
		}
		return element;
	}

	private void prepareChildren() {
		children.clear();

		if (LayerNature.isTimeRemapEnabled(layer)) {
			children.add(getChildElement(TimeRemapElement.class));
		}

		if (layer instanceof TextLayer) {
			children.add(getChildElement(TextElement.class));
		}

		if (layer instanceof EffectableLayer /*&& !((EffectableLayer) layer).getEffects().isEmpty()*/) {
			children.add(getChildElement(EffectsElement.class));
		}

		if (layer instanceof TransformableLayer && LayerNature.isVideoNature(layer)) {
			children.add(getChildElement(TransformElement.class));

		} else if (layer instanceof LightLayer) {
			if (((LightLayer) layer).getLightType() != LightType.AMBIENT) {
				children.add(getChildElement(LightTransformElement.class));
			}
			children.add(getChildElement(LightOptionsElement.class));

		} else if (layer instanceof CameraLayer) {
			children.add(getChildElement(CameraTransformElement.class));
			children.add(getChildElement(CameraOptionsElement.class));
		}

		if (layer instanceof MediaLayer && LayerNature.isThreeD(layer)) {
			children.add(getChildElement(MaterialOptionsElement.class));
		}

		if (layer instanceof MediaLayer && LayerNature.isAudioNature(layer)) {
			children.add(getChildElement(AudioElement.class));
		}

		childrenMap.values().retainAll(children);
	}

	@Override
	public boolean hasChildren() {
		prepareChildren();
		return (children.size() > 0);
	}

	@Override
	public Element[] getChildren() {
		// getChildren が呼ばれるより前に必ず hasChildren が呼ばれるようではあるが、
		// 念のため prepareChildren を実行しておく。
		prepareChildren();
		return children.toArray(new Element[children.size()]);
	}

	public void paintColumn(Event event) {
		switch (event.index) {
			case LayerCompositionView.SHOWHIDE_COL:
				showHideColumnSwitches.paint(event);
				break;

			case LayerCompositionView.VALUE_COL:
				switchesColumnSwitches.paint(event);
				break;

			case LayerCompositionView.MODE_COL:
				if (layer instanceof MediaLayer) {
					modeArea = drawString(event,
							EnumLabels.getLabel(((MediaLayer) layer).getBlendMode()));
				}
				break;

			case LayerCompositionView.TRACKMATTE_COL:
				if (layer instanceof MediaLayer && getTrackMatteLayer() != null) {
					TrackMatte trackMatte = ((MediaLayer) layer).getTrackMatte();
					trackMatteArea = drawString(event,
							trackMatte != TrackMatte.NONE ? EnumLabels.getLabel(trackMatte) : "      ");
				} else {
					trackMatteArea = null;
				}
				break;

			case LayerCompositionView.PARENT_COL: {
				String parentName;
				Layer parent = composition.getParentLayer(layer);
				if (parent == null) {
					parentName = "      ";
				} else {
					parentName = getName(parent);
				}
				parentArea = drawString(event, parentName);
				break;
			}

			case LayerCompositionView.TIMELINE_COL:
				TimelineManager tm = (TimelineManager) viewer.getData(LayerCompositionView.TIMELINE_MANAGER);
				tm.drawLayer(event, layer);
				break;

			default:
				super.paintColumn(event);
				break;
		}
	}

	private Rectangle drawString(Event event, String str) {
		GC gc = event.gc;
		int x = event.x + 5;
		int y = event.y;
		int height = ((TreeItem) event.item).getBounds(event.index).height;
		boolean focused = ((event.detail & SWT.SELECTED) != 0 && ((Control) event.widget).isFocusControl());

		gc.setForeground(gc.getDevice().getSystemColor(
				focused ? SWT.COLOR_WHITE : SWT.COLOR_DARK_BLUE));

		Point extent = gc.textExtent(str, SWT.DRAW_TRANSPARENT);

		y += (height - extent.y) / 2;
		gc.drawString(str, x, y, true);

		Rectangle area = new Rectangle(x, y, extent.x, extent.y);

		y += extent.y;
		gc.setLineDash(UNDERLINE_DASH);
		if (COCOA) {
			gc.drawLine(x, y - 1, x + extent.x, y - 1);
		} else if (WIN32) {
			gc.drawLine(x, y, x + extent.x - 2, y);
		} else {
			gc.drawLine(x, y, x + extent.x, y);
		}

		return area;
	}

	public void updateCursor(MouseEvent event, int columnIndex) {
		Rectangle area;

		switch (columnIndex) {
			case LayerCompositionView.TIMELINE_COL:
				TimelineManager tm = (TimelineManager) viewer.getData(LayerCompositionView.TIMELINE_MANAGER);
				tm.updateCursor(event, layer);
				return;

			default:
				super.updateCursor(event, columnIndex);
				return;

			case LayerCompositionView.MODE_COL:
				area = modeArea;
				break;

			case LayerCompositionView.TRACKMATTE_COL:
				area = trackMatteArea;
				break;

			case LayerCompositionView.PARENT_COL:
				area = parentArea;
				break;
		}

		Cursor cursor = null;
		if (area != null && area.contains(event.x, event.y)) {
			cursor = event.display.getSystemCursor(SWT.CURSOR_HAND);
		}
		viewer.getTree().setCursor(cursor);
	}

	public void mouseDown(MouseEvent event, int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.SHOWHIDE_COL:
				showHideColumnSwitches.mouseDown(event);
				break;

			case LayerCompositionView.VALUE_COL:
				switchesColumnSwitches.mouseDown(event);
				break;

			case LayerCompositionView.MODE_COL:
				if (/*layer instanceof MediaLayer &&*/
						modeArea != null && modeArea.contains(event.x, event.y)) {
					showModeMenu();
				}
				break;

			case LayerCompositionView.TRACKMATTE_COL:
				if (/*layer instanceof MediaLayer && getTrackMatteLayer() != null &&*/
						trackMatteArea != null && trackMatteArea.contains(event.x, event.y)) {
					showTrackMatteMenu();
				}
				break;

			case LayerCompositionView.PARENT_COL:
				if (parentArea != null && parentArea.contains(event.x, event.y)) {
					showParentMenu();
				}
				break;

			case LayerCompositionView.TIMELINE_COL:
				TimelineManager tm = (TimelineManager) viewer.getData(LayerCompositionView.TIMELINE_MANAGER);
				tm.mouseDown(event, layer);
				break;

			default:
				super.mouseDown(event, columnIndex);
				break;
		}
	}

	private Menu createPopUpMenu(Rectangle area) {
		return createPopUpMenu(area.x-5, area.y+area.height+2);
	}

	private void showModeMenu() {
		Menu menu = createPopUpMenu(modeArea);

		final BlendMode currentMode = ((MediaLayer) layer).getBlendMode();

		for (final BlendMode mode : BlendMode.values()) {
			if (mode == BlendMode.NONE) continue;
			if (EnumLabels.insertSeparator(mode)) {
				new MenuItem(menu, SWT.SEPARATOR);
			}
			Action action = new Action(EnumLabels.getLabel(mode), IAction.AS_RADIO_BUTTON) {
				public void run() {
					if (currentMode != mode) {
						ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);
						pm.postOperation(new ModifyLayerBlendModeOperation(pm, (MediaLayer) layer, mode));
					}
				}
			};
			action.setChecked(currentMode == mode);
			new ActionContributionItem(action).fill(menu, -1);
		}

		menu.setVisible(true);
	}

	private void showTrackMatteMenu() {
		Menu menu = createPopUpMenu(trackMatteArea);

		final TrackMatte current = ((MediaLayer) layer).getTrackMatte();

		for (final TrackMatte tm : TrackMatte.values()) {
			if (EnumLabels.insertSeparator(tm)) {
				new MenuItem(menu, SWT.SEPARATOR);
			}

			String label;
			if (tm == TrackMatte.NONE) {
				label = EnumLabels.getLabel(tm);
			} else {
				label = String.format("%s - %s", EnumLabels.getLabel(tm), getName(getTrackMatteLayer()));
			}

			Action action = new Action(label.replaceAll("&", "&&"), IAction.AS_RADIO_BUTTON) {
				public void run() {
					if (current != tm) {
						ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);
						pm.postOperation(new ModifyLayerTrackMatteOperation(pm, (MediaLayer) layer, tm));
					}
				}
			};
			action.setChecked(current == tm);
			new ActionContributionItem(action).fill(menu, -1);
		}

		menu.setVisible(true);
	}

	private void showParentMenu() {
		Menu menu = createPopUpMenu(parentArea);

		final Layer currentParent = composition.getParentLayer(layer);

		Action action = new Action(getName(null), IAction.AS_RADIO_BUTTON) {
			public void run() {
				if (currentParent != null) {
					ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);
					pm.postOperation(new ModifyLayerParentOperation(pm, layer, null));
				}
			}
		};
		action.setChecked(currentParent == null);
		new ActionContributionItem(action).fill(menu, -1);

		new MenuItem(menu, SWT.SEPARATOR);

		for (final Layer otherLayer : composition.getLayers()) {
			action = new Action(getName(otherLayer).replaceAll("&", "&&"), IAction.AS_RADIO_BUTTON) {
				public void run() {
					if (currentParent != otherLayer) {
						ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);
						pm.postOperation(new ModifyLayerParentOperation(pm, layer, otherLayer));
					}
				}
			};
			action.setEnabled(otherLayer != layer);
			action.setChecked(currentParent == otherLayer);
			new ActionContributionItem(action).fill(menu, 2);
		}

		menu.setVisible(true);
	}

	private String getName(Layer layer) {
		if (layer == null) {
			return "なし";
		} else if (!(layer instanceof ItemLayer<?>) || isLayerNameMode()) {
			return layer.getName();
		} else {
			return ((ItemLayer<?>) layer).getItem().getName();
		}
	}

	public boolean canEdit(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				return Boolean.TRUE.equals(viewer.getData(LayerCompositionView.EDIT_ELEMENT_NAME));

			default:
				return false;
		}
	}

	public CellEditor getCellEditor(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				if (nameEditor == null) {
					nameEditor = new TextCellEditor(viewer.getTree(), SWT.SINGLE | SWT.BORDER);
					Control control = nameEditor.getControl();
					control.setBackground(control.getDisplay().getSystemColor(SWT.COLOR_WHITE));
				}
				return nameEditor;

			default:
				return null;
		}
	}

	public Object getCellEditorValue(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				return layer.getName();

			default:
				return null;
		}
	}

	public void setCellEditorValue(int columnIndex, Object value) {
		ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);

		switch (columnIndex) {
			case LayerCompositionView.NAME_COL: {
				String newName = (String) value;
				if (newName.length() == 0) {
					if (layer instanceof ItemLayer<?>) {
						newName = ((ItemLayer<?>) layer).getItem().getName();
					} else {
						// TODO ItemLayer以外の場合、どうするのがいいか？　以下は暫定
						if (layer instanceof TextLayer) newName = "テキスト";
						else if (layer instanceof NullLayer) newName = "ヌル";
						else if (layer instanceof CameraLayer) newName = "カメラ";
						else if (layer instanceof LightLayer) newName = "ライト";
					}
				}
				pm.postOperation(new RenameLayerOperation(pm, layer, newName));
				break;
			}
		}
	}

}
