/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.wizards;

import java.util.Arrays;

import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.PaletteData;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.ColorDialog;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.app.widgets.GridBuilder;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.output.MacOSXQTMovieOutput;
import ch.kuramo.javie.core.output.MacOSXQTMovieOutput.AudioCompressorSettings;
import ch.kuramo.javie.core.output.MacOSXQTMovieOutput.VideoCompressorSettings;
import ch.kuramo.javie.core.output.Output.VideoChannels;

public class QTMovieOutputSettingsWizardPage extends WizardPage {

	private static boolean videoEnabledDefault = true;

	private static VideoChannels videoChannelsDefault = VideoChannels.RGB;

	private static Color colorMatteDefault = Color.BLACK;

	private static VideoCompressorSettings vcSettingsDefault;

	private static boolean audioEnabledDefault = true;

	private static AudioMode audioModeDefault = AudioMode.STEREO_48KHZ_INT16;

	private static AudioCompressorSettings acSettingsDefault;


	private final double compositionFrameRate;

	private boolean videoEnabled = videoEnabledDefault;

	private VideoChannels videoChannels = videoChannelsDefault;

	private Color colorMatte = colorMatteDefault;

	private VideoCompressorSettings vcSettings;

	private boolean audioEnabled = audioEnabledDefault;

	private AudioMode audioMode = audioModeDefault;

	private AudioCompressorSettings acSettings;

	private Image colorMatteImage;


	public QTMovieOutputSettingsWizardPage(Composition composition) {
		super("QTMovieOutputSettingsWizardPage");
		setTitle("出力設定");

		compositionFrameRate = 1/composition.getFrameDuration().toSecond();
		vcSettings = MacOSXQTMovieOutput.doVideoCompressorSettings(vcSettingsDefault, compositionFrameRate, false);

		acSettings = MacOSXQTMovieOutput.doAudioCompressorSettings(acSettingsDefault, audioMode.sampleRate, false);
	}

	public void dispose() {
		if (colorMatteImage != null) {
			colorMatteImage.dispose();
			colorMatteImage = null;
		}
		super.dispose();
	}

	public void createControl(Composite parent) {
		GridBuilder gb = new GridBuilder(parent, 1, false);
		((GridLayout) gb.getComposite().getLayout()).verticalSpacing = 0;

		createVideoGroup(gb);
		gb.size(10, 10).composite(SWT.NULL);
		createAudioGroup(gb);

		Composite grid = gb.getComposite();
		grid.setTabList(gb.getTabList());

		setControl(grid);
		doValidate();
	}

	private void createVideoGroup(GridBuilder gb) {
		String[] channelItems = {
				"RGB", "アルファ", "RGB + アルファ（ストレート）", "RGB + アルファ（乗算済み）"
		};

		final Button enabledCheck = gb.hAlign(SWT.LEFT).hGrab().button(SWT.CHECK, "ビデオ出力");
		enabledCheck.setSelection(videoEnabled);

		Group group = gb.hAlign(SWT.FILL).group(SWT.NONE, null);
		group.setLayout(new FillLayout(SWT.HORIZONTAL));
		GridBuilder gb2 = new GridBuilder(group, 6, false);
		GridLayout layout = (GridLayout) gb2.getComposite().getLayout();
		layout.marginWidth = 15;

		final Label chLabel = gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "出力チャンネル:");
		final Combo chCombo = gb2.hSpan(2).hAlign(SWT.LEFT).combo(SWT.READ_ONLY, 0,
										channelItems, channelItems[videoChannels.ordinal()]);
		gb2.hSpan(1).size(10, 10).composite(SWT.NULL);
		final Label matteLabel = gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "カラーマット:");
		final Button matteButton = gb2.hSpan(1).hAlign(SWT.LEFT).button(SWT.PUSH | SWT.FLAT, "");

		final Label vcompLabel = gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "ビデオ圧縮:");
		final Button vcompButton = gb2.hSpan(1).hAlign(SWT.LEFT).button(SWT.PUSH, "設定...");
		final Label vcompAsText = gb2.span(4, 2).hAlign(SWT.FILL).label(SWT.NULL, vcSettings.text);
		gb2.hSpan(2).vGrab().size(10, 10).composite(SWT.NULL);


		enableControls(videoEnabled, chLabel, chCombo, vcompLabel, vcompButton, vcompAsText);
		enableControls(videoEnabled && videoChannels == VideoChannels.RGBA_PREMULTIPLIED, matteLabel, matteButton);
		updateColorMatteImage(matteButton, toRGB(colorMatte));

		enabledCheck.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				videoEnabled = enabledCheck.getSelection();
				enableControls(videoEnabled, chLabel, chCombo, vcompLabel, vcompButton, vcompAsText);
				enableControls(videoEnabled && videoChannels == VideoChannels.RGBA_PREMULTIPLIED, matteLabel, matteButton);
				doValidate();
			}
		});

		chCombo.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				videoChannels = VideoChannels.values()[chCombo.getSelectionIndex()];
				enableControls(videoChannels == VideoChannels.RGBA_PREMULTIPLIED, matteLabel, matteButton);
				doValidate();
			}
		});

		matteButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				ColorDialog dialog = new ColorDialog(getShell());
				dialog.setRGB(toRGB(colorMatte));
				RGB rgb = dialog.open();
				if (rgb != null) {
					colorMatte = toColor(rgb);
					updateColorMatteImage((Button)e.widget, rgb);
					doValidate();
				}
				doValidate();
			}
		});

		vcompButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				VideoCompressorSettings newSettings = MacOSXQTMovieOutput.doVideoCompressorSettings(
																vcSettings, vcSettings.frameRate, true);
				if (newSettings != null) {
					vcSettings = newSettings;
					vcompAsText.setText(newSettings.text);
					doValidate();
				}
			}
		});


		Composite grid2 = gb2.getComposite();
		grid2.setTabList(gb2.getTabList());
	}

	private void createAudioGroup(GridBuilder gb) {
		String[] audioItems = {
				"48kHz, 16bit", "48kHz, 32bit", "48kHz, 32bit浮動小数",
				"96kHz, 16bit", "96kHz, 32bit", "96kHz, 32bit浮動小数",
		};

		final Button enabledCheck = gb.hAlign(SWT.LEFT).hGrab().button(SWT.CHECK, "オーディオ出力");
		enabledCheck.setSelection(audioEnabled);

		Group group = gb.hAlign(SWT.FILL).group(SWT.NONE, null);
		group.setLayout(new FillLayout(SWT.HORIZONTAL));
		GridBuilder gb3 = new GridBuilder(group, 3, false);
		GridLayout layout = (GridLayout) gb3.getComposite().getLayout();
		layout.marginWidth = 15;

		final Label amodeLabel = gb3.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "内部処理形式:");
		final Combo amodeCombo = gb3.hSpan(2).hAlign(SWT.LEFT).combo(SWT.READ_ONLY, 0,
													audioItems, audioItems[audioMode.ordinal()]);

		final Label acompLabel = gb3.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "オーディオ圧縮:");
		final Button acompButton = gb3.hSpan(1).hAlign(SWT.LEFT).button(SWT.PUSH, "設定...");
		final Label acompAsText = gb3.span(1, 2).hAlign(SWT.FILL).label(SWT.NULL, acSettings.text);
		gb3.hSpan(2).vGrab().size(10, 10).composite(SWT.NULL);


		enableControls(audioEnabled, amodeLabel, amodeCombo, acompLabel, acompButton, acompAsText);

		enabledCheck.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				audioEnabled = enabledCheck.getSelection();
				enableControls(audioEnabled, amodeLabel, amodeCombo, acompLabel, acompButton, acompAsText);
				doValidate();
			}
		});

		amodeCombo.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				audioMode = AudioMode.values()[amodeCombo.getSelectionIndex()];
				doValidate();
			}
		});

		acompButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				AudioCompressorSettings newSettings = MacOSXQTMovieOutput.doAudioCompressorSettings(
																acSettings, acSettings.sampleRate, true);
				if (newSettings != null) {
					acSettings = newSettings;
					acompAsText.setText(newSettings.text);
					doValidate();
				}
			}
		});


		Composite grid3 = gb3.getComposite();
		grid3.setTabList(gb3.getTabList());
	}

	private void enableControls(boolean enabled, Control... controls) {
		org.eclipse.swt.graphics.Color disabledColor
				= getShell().getDisplay().getSystemColor(SWT.COLOR_DARK_GRAY);

		for (Control c : controls) {
			if (c instanceof Label) {
				c.setForeground(enabled ? c.getParent().getForeground() : disabledColor);
			} else {
				c.setEnabled(enabled);
			}
		}
	}

	private RGB toRGB(Color color) {
		return new RGB((int)(color.r*255), (int)(color.g*255), (int)(color.b*255));
	}

	private Color toColor(RGB rgb) {
		return new Color(rgb.red/255.0, rgb.green/255.0, rgb.blue/255.0, 1.0);
	}

	private void updateColorMatteImage(Button button, RGB rgb) {
		PaletteData palette = new PaletteData(new RGB[] { new RGB(0, 0, 0), rgb });
		byte[] data = new byte[64];
		Arrays.fill(data, (byte) 255);
		ImageData imageData = new ImageData(32, 16, 1, palette, 1, data);

		Image image = new Image(button.getDisplay(), imageData);
		button.setImage(image);
		if (colorMatteImage != null) {
			colorMatteImage.dispose();
		}
		colorMatteImage = image;
	}

	private void doValidate() {
		setPageComplete(false);

		if (!videoEnabled && !audioEnabled) {
			setErrorMessage("ビデオ出力とオーディオ出力の少なくともどちらか一方は有効にする必要があります。");
			return;
		}

		if (videoEnabled) {
			switch (videoChannels) {
				case RGBA_STRAIGHT:
				case RGBA_PREMULTIPLIED:
					if (vcSettings.depth < 32) {
						setErrorMessage("“RGB + アルファ” を出力するには、ビデオ圧縮の階調を “約 1670 万色以上” にする必要があります。");
						return;
					}
					break;
			}
		}

		String warning = null;

		if (videoEnabled) {
			String fpsVideoComp = String.format("%.02f", vcSettings.frameRate);
			String fpsComposition = String.format("%.02f", compositionFrameRate);
			if (warning == null && !fpsVideoComp.equals(fpsComposition)) {
				warning = "ビデオ圧縮のフレームレートがコンポジションのフレームレートと一致していません。コンポジションのフレームレートは"
							+ String.format(" %.02f です。", compositionFrameRate);
			}
		}

		if (audioEnabled) {
			if (warning == null && acSettings.sampleRate != audioMode.sampleRate) {
				warning = "内部処理形式のサンプルレートとオーディオ圧縮のサンプルレートが一致していません。";
			}
		}

		setMessage(warning, IMessageProvider.WARNING);
		setErrorMessage(null);
		setPageComplete(true);
	}

	public void saveDefaults() {
		videoEnabledDefault = videoEnabled;
		videoChannelsDefault = videoChannels;
		colorMatteDefault = colorMatte;
		vcSettingsDefault = vcSettings;
		audioEnabledDefault = audioEnabled;
		audioModeDefault = audioMode;
		acSettingsDefault = acSettings;
	}

	public boolean isVideoEnabled() {
		return videoEnabled;
	}

	public VideoChannels getVideoChannels() {
		return videoChannels;
	}

	public Color getColorMatte() {
		return colorMatte;
	}

	public VideoCompressorSettings getVideoCompressorSettings() {
		return vcSettings;
	}

	public boolean isAudioEnabled() {
		return audioEnabled;
	}

	public AudioMode getAudioMode() {
		return audioMode;
	}

	public AudioCompressorSettings getAudioCompressorSettings() {
		return acSettings;
	}

}
