/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;

import javax.media.opengl.GL2;
import javax.media.opengl.GLUniformData;
import javax.media.opengl.glu.GLU;
import javax.media.opengl.glu.GLUtessellator;
import javax.media.opengl.glu.GLUtessellatorCallback;
import javax.media.opengl.glu.GLUtessellatorCallbackAdapter;
import javax.vecmath.Matrix4d;
import javax.vecmath.Point2d;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;
import javax.vecmath.Vector4d;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.BlendMode;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Quality;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.ShaderType;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.services.IAccumulationSupport;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IBlurSupport;
import ch.kuramo.javie.api.services.IConvolutionSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Camera;
import ch.kuramo.javie.core.CastsShadows;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.Light;
import ch.kuramo.javie.core.LightType;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.MotionBlur;
import ch.kuramo.javie.core.TrackMatte;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.VideoLayerRenderer;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.VideoLayerRenderer.CRLayerRenderer;
import ch.kuramo.javie.core.VideoLayerRenderer.GeneralLayerRenderer;
import ch.kuramo.javie.core.VideoLayerRenderer.MatteLayerRenderers;
import ch.kuramo.javie.core.VideoLayerRenderer.NormalLayerRenderer;
import ch.kuramo.javie.core.services.GLGlobal;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;

public class VideoLayerComposer {

	private static final Logger logger = LoggerFactory.getLogger(VideoLayerComposer.class);


	private final RenderContext context;

	private final IVideoRenderSupport support;

	private final IAccumulationSupport accumSupport;

	private final IConvolutionSupport convolution;

	private final IBlurSupport blurSupport;

	private final IArrayPools arrayPools;

	private final IShaderRegistry shaders;

	private final GLGlobal glGlobal;

	private final IShaderProgram normalBlend3DProgram;

	private final IShaderProgram normalBlend2DProgram;

	private final IShaderProgram mblur2Accum2Program;

	private final IShaderProgram matteMultiplyProgram;

	private final IShaderProgram diffusionConvolutionProgram;

	private final IShaderProgram shadowMultiplyProgram;

	private final int maxTexImageUnits;


	private final LayerComposition composition;

	private final Camera camera;

	private final Set<Light> lights;

	private final boolean lightsCastShadows;

	private boolean mblurWithShadows;

	private final Resolution resolution;

	private final int ssScale;

	private final VideoBounds bounds;

	private final VideoBounds boundsX;

	private final Time baseTime;


	private final List<ComposeGroup> groups = Util.newList();

	private ComposeGroup currentGroup;

	private final Map<String, MatteLayerRenderers> matteLayerRenderers = Util.newMap();

	private final Map<String, IVideoBuffer> matteBufferCache = Util.newMap();

	private final Map<AnimatableValue<?>, Map<Time, ?>> valueCache = Util.newMap();


	@Inject
	VideoLayerComposer(
			RenderContext context, IVideoRenderSupport support, IAccumulationSupport accumSupport,
			IConvolutionSupport convolution, IBlurSupport blurSupport,
			IArrayPools arrayPools, IShaderRegistry shaders, GLGlobal glGlobal) {

		this.context = context;
		this.support = support;
		this.accumSupport = accumSupport;
		this.convolution = convolution;
		this.blurSupport = blurSupport;
		this.arrayPools = arrayPools;
		this.shaders = shaders;
		this.glGlobal = glGlobal;

		maxTexImageUnits = glGlobal.getMaxTextureImageUnits();
		createReadTextureFunction();

		normalBlend3DProgram = shaders.getProgram(VideoLayerComposer.class, "NORMAL_BLEND_3D");
		normalBlend2DProgram = shaders.getProgram(VideoLayerComposer.class, "NORMAL_BLEND_2D");
		mblur2Accum2Program = shaders.getProgram(VideoLayerComposer.class, "MBLUR2_ACCUM2");
		matteMultiplyProgram = shaders.getProgram(VideoLayerComposer.class, "MATTE_MULTIPLY");
		diffusionConvolutionProgram = shaders.getProgram(VideoLayerComposer.class, "DIFFUSION_CONVOLUTION");
		shadowMultiplyProgram = shaders.getProgram(VideoLayerComposer.class, "SHADOW_MULTIPLY");

		composition = (LayerComposition) context.getComposition();
		camera = context.getCamera();
		lights = context.getLights();
		resolution = context.getVideoResolution();
		ssScale = (resolution.scale < 1) ? 1 : composition.getSuperSampling().ordinal() + 1;
		bounds = new VideoBounds(camera.getViewportSize());
		boundsX = new VideoBounds(bounds.width*ssScale, bounds.height*ssScale);
		baseTime = context.getTime();

		boolean lightsCastShadows = false;
		for (Light light : lights) {
			if (light.isCastsShadows()) {
				lightsCastShadows = true;
				break;
			}
		}
		this.lightsCastShadows = lightsCastShadows;
	}

	void addLayerRenderer(VideoLayerRenderer r) {
		if (r instanceof MatteLayerRenderers) {
			MatteLayerRenderers mlr = (MatteLayerRenderers) r;
			matteLayerRenderers.put(mlr.getMatteId(), mlr);

		} else if (r instanceof GeneralLayerRenderer) {
			boolean threeD = LayerNature.isThreeD(r.getLayer());
			if (currentGroup == null || currentGroup.threeD != threeD) {
				currentGroup = new ComposeGroup(threeD);
				groups.add(currentGroup);
			}
			currentGroup.add((GeneralLayerRenderer) r);

		} else {
			throw new IllegalArgumentException();
		}
	}

	void addLayerRenderers(List<VideoLayerRenderer> renderers) {
		for (VideoLayerRenderer r : renderers) {
			addLayerRenderer(r);
		}
	}

	IVideoBuffer compose() {
		IVideoBuffer composeBuffer = null;
		try {
			composeBuffer = support.createVideoBuffer(bounds);
			composeBuffer.clear();

			for (ComposeGroup group : groups) {
				if (group.threeD) {
					IVideoBuffer buffer = composeBuffer;
					composeBuffer = compose3D(group, buffer);
					if (composeBuffer != buffer) {
						buffer.dispose();
					}
				} else {
					compose2D(group, composeBuffer);
				}
			}

			IVideoBuffer result = composeBuffer;
			composeBuffer = null;
			return result;

		} finally {
			if (composeBuffer != null) composeBuffer.dispose();

			for (IVideoBuffer vb : matteBufferCache.values()) vb.dispose();
			matteBufferCache.clear();
		}
	}

	private IVideoBuffer compose3D(final ComposeGroup group, IVideoBuffer composeBuffer) {
		boolean mblurEnabled = false;
		if (composition.isMotionBlurEnabled()) {
			for (GeneralLayerRenderer r : group.renderers) {
				if (LayerNature.getMotionBlur(r.getLayer()) != MotionBlur.NONE) {
					mblurEnabled = true;
					break;
				}
			}
		}

		mblurWithShadows = false;
		if (mblurEnabled && lightsCastShadows) {
			for (GeneralLayerRenderer r : group.renderers) {
				if (LayerNature.getMotionBlur(r.getLayer()) != MotionBlur.NONE
						&& r.getLayer().getCastsShadows() != CastsShadows.OFF) {
					mblurWithShadows = true;
					break;
				}
			}
		}

		Set<IVideoBuffer> tmpBuffers = Util.newSet();
		final Map<GeneralLayerRenderer, Entry3D> cache = Util.newMap();
		try {
			final IVideoBuffer composeBuffer2;
			IVideoBuffer composeBuffer3;

			if (ssScale > 1) {
				composeBuffer2 = magnifyForSuperSampling(composeBuffer);
				tmpBuffers.add(composeBuffer2);
			} else {
				composeBuffer2 = composeBuffer;
			}

			if (mblurEnabled) {
				composeBuffer3 = motionBlur1(composeBuffer2, new MotionBlurSampler1() {
					public IVideoBuffer sample() {
						IVideoBuffer buffer = null;
						try {
							buffer = support.createVideoBuffer(boundsX);
							support.copy(composeBuffer2, buffer);
							compose3D(group, buffer, cache);

							IVideoBuffer result = buffer;
							buffer = null;
							return result;

						} finally {
							if (buffer != null) buffer.dispose();
						}
					}

					public double getOpacity() {
						return 1.0;
					}
				});
				if (composeBuffer3 != composeBuffer2) {
					tmpBuffers.add(composeBuffer3);
				}
			} else {
				compose3D(group, composeBuffer2, cache);
				composeBuffer3 = composeBuffer2;
			}

			if (ssScale > 1) {
				downSample(composeBuffer3, composeBuffer);
				return composeBuffer;
			} else {
				tmpBuffers.remove(composeBuffer3);
				return composeBuffer3;
			}

		} finally {
			for (Entry3D entry : cache.values()) entry.dispose();
			for (IVideoBuffer vb : tmpBuffers) vb.dispose();
		}
	}

	private void compose3D(ComposeGroup group, IVideoBuffer composeBuffer, Map<GeneralLayerRenderer, Entry3D> cache) {
		// 本質的には順序は結果に無関係だが、プレンドやシャドウの計算で
		// 誤差の発生の仕方が異なるかもしれないので、もとの順序を維持しておく。
		Map<GeneralLayerRenderer, Entry3D> entryMap = Util.newLinkedHashMap();
		try {
			for (ListIterator<GeneralLayerRenderer> it = group.renderers.listIterator(); it.hasNext(); ) {
				int i = it.nextIndex();
				GeneralLayerRenderer r = it.next();
				Entry3D entry = render3DLayer(r, i, cache);
				if (entry != null) {
					entryMap.put(r, entry);
				}
			}

			transformAndLighting(Util.newList(entryMap.values()));

			// シャドウのみのレイヤーは取り除く。
			// レイヤーの平面上に視点が乗っている場合、レイヤーは視点から見えないので取り除く。
			// ただし、あとで必要になるかもしれない（モーションブラー有効時）のでキャッシュに入れておく。
			for (Iterator<Map.Entry<GeneralLayerRenderer, Entry3D>> it = entryMap.entrySet().iterator(); it.hasNext(); ) {
				Map.Entry<GeneralLayerRenderer, Entry3D> e = it.next();
				Entry3D entry = e.getValue();
				// TODO 1e-3が妥当な値かどうかわからない。1e-5ではダメなケースがあることは確認した。
				//      平面上に完全に視点が乗っていなくても、投影面上での面積はほとんどなく
				//      消えてしまっても問題ないはずなので、少々大きめの値でも大丈夫だろうか？
				if (entry.castsShadows == CastsShadows.ONLY
						|| /*entry.plane.w == 0*/ Math.abs(entry.plane.w) < 1e-3) {
					cache.put(e.getKey(), entry);
					it.remove();
				}
			}

			Entry3D[] entries = entryMap.values().toArray(new Entry3D[entryMap.size()]);
			Arrays.sort(entries, Entry3D.partitioningOrderComparator);

			List<Polygon> polygons = Util.newList();
			for (Entry3D entry : entries) {
				polygons.add(new Polygon(entry));
			}

			polygons = partitionPolygons(polygons);

			List<Triangle> triangles = Util.newList();
			List<IVideoBuffer> buffers = Util.newList();

			for (ListIterator<Polygon> it = polygons.listIterator(); it.hasNext(); ) {
				Polygon p = it.next();
				BlendMode blendMode = p.entry.layer.getBlendMode();

				// ステンシルアルファとステンシルルミナンスは、3Dレイヤーでは機能しない。
				switch (blendMode) {
					case STENCIL_ALPHA:
					case STENCIL_LUMA:
						blendMode = BlendMode.NORMAL;
						break;
				}

				if (blendMode == BlendMode.NORMAL) {
					int sourceIndex = buffers.indexOf(p.entry.lighted);
					int matteIndex = (p.entry.matte != null) ? buffers.indexOf(p.entry.matte) : -2;

					// buffersのサイズが2増える状況でmaxTexImageUnitsまでの残りが1の場合、
					// そのまま続行することができないので、イテレータをひとつ巻き戻した上で既にbuffersに入っている分だけ処理する。
					if (sourceIndex == -1 && matteIndex == -1 && buffers.size() == maxTexImageUnits-1) {
						it.previous();

					} else {
						if (sourceIndex == -1) {
							sourceIndex = buffers.size();
							buffers.add(p.entry.lighted);
						}
						if (matteIndex == -1) {
							matteIndex = buffers.size();
							buffers.add(p.entry.matte);
						}

						triangles.addAll(p.toTriangles(sourceIndex, matteIndex));

						if (buffers.size() < maxTexImageUnits && it.hasNext()) {
							continue;
						}
					}
				}

				if (!buffers.isEmpty()) {
					normalBlend3D(triangles, buffers, composeBuffer);
					triangles.clear();
					buffers.clear();
				}

				if (blendMode != BlendMode.NORMAL) {
					customBlend3D(p, composeBuffer);
				}
			}

		} finally {
			for (Map.Entry<GeneralLayerRenderer, Entry3D> e : entryMap.entrySet()) {
				cache.put(e.getKey(), e.getValue());
			}
		}
	}

	private Entry3D render3DLayer(final GeneralLayerRenderer r, final int indexInGroup, final Map<GeneralLayerRenderer, Entry3D> cache) {
		return context.saveAndExecute(new WrappedOperation<Entry3D>() {
			public Entry3D execute() {
				MediaLayer layer = r.getLayer();
				MotionBlur mblur = composition.isMotionBlurEnabled()
						? LayerNature.getMotionBlur(layer) : MotionBlur.NONE;

				Entry3D entry = cache.remove(r);
				IVideoBuffer source = null;
				IVideoBuffer sourceCR = null;
				IVideoBuffer diffusion = null;
				IVideoBuffer matte = null;
				IVideoBuffer tmp = null;

				try {
					switch (mblur) {
						case NONE:
							if (entry != null) {
								if (mblurWithShadows && layer.isAcceptsShadows()) {
									entry.disposeLighted();
								}
								return entry;
							}
							context.setTime(baseTime);
							break;

						case TRANSFORM:
							if (r instanceof NormalLayerRenderer) {
								if (entry != null) {
									source = entry.source;
									diffusion = entry.diffusion;
									entry.disposeLighted();
									entry = null;
								}
								break;
							}
							// fall through
						default:
							if (entry != null) {
								entry.dispose();
								entry = null;
							}
							break;
					}

					CastsShadows castsShadows = layer.getCastsShadows();
					double[] mvMatrix = new double[16];
					Point3d[] vertices;
					Point2d[] texCoords;
					double opacity;

					System.arraycopy(camera.getModelView3D(), 0, mvMatrix, 0, 16);

					if (r instanceof NormalLayerRenderer) {
						NormalLayerRenderer nr = (NormalLayerRenderer) r;

						Time time0 = context.getTime();
						Time time1 = (mblur == MotionBlur.TRANSFORM) ? baseTime : time0;

						VideoBounds b;
						try {
							context.setTime(time1);
							b = nr.calcBounds(true);
						} finally {
							context.setTime(time0);
						}
						if (b.isEmpty()) {
							return null;
						}

						opacity = nr.getOpacity();
						if (opacity <= 0) {
							return null;
						}

						nr.multModelViewMatrix(mvMatrix);

						double expand = 16;		// TODO 拡張するサイズはどの程度必要か？

						vertices = new Point3d[4];
						vertices[0] = new Point3d(b.x        -expand, b.y         -expand, 0);
						vertices[1] = new Point3d(b.x+b.width+expand, b.y         -expand, 0);
						vertices[2] = new Point3d(b.x+b.width+expand, b.y+b.height+expand, 0);
						vertices[3] = new Point3d(b.x        -expand, b.y+b.height+expand, 0);

						if (!intersectsWithFrustum(vertices, mvMatrix) && castsShadows == CastsShadows.ON) {
							castsShadows = CastsShadows.ONLY;
						}
						if (!lightsCastShadows && castsShadows == CastsShadows.ONLY) {
							return null;
						}

						if (source == null) {
							try {
								context.setTime(time1);
								tmp = nr.render(true, composition.isFrameBlendEnabled());
								if (!tmp.getBounds().equals(b)) {
									logger.warn("bounds differ");
								}

								int[] shadowExpansions = null;
								if (lightsCastShadows && castsShadows != CastsShadows.OFF) {
									shadowExpansions = getShadowExpansions(layer);
								}
								if (shadowExpansions != null) {
									source = expand(tmp, shadowExpansions);
									tmp.dispose();
									tmp = null;
								} else {
									source = tmp;
									tmp = null;
								}
							} finally {
								context.setTime(time0);
							}
						}

						if (!source.getBounds().equals(b)) {
							b = source.getBounds();
							vertices[0] = new Point3d(b.x        -expand, b.y         -expand, 0);
							vertices[1] = new Point3d(b.x+b.width+expand, b.y         -expand, 0);
							vertices[2] = new Point3d(b.x+b.width+expand, b.y+b.height+expand, 0);
							vertices[3] = new Point3d(b.x        -expand, b.y+b.height+expand, 0);
						}

						Matrix4d m = new Matrix4d(mvMatrix);
						m.transpose();
						m.transform(vertices[0]);
						m.transform(vertices[1]);
						m.transform(vertices[2]);
						m.transform(vertices[3]);

						double ex = expand/b.width;
						double ey = expand/b.height;
						texCoords = new Point2d[4];
						texCoords[0] = new Point2d( -ex,  -ey);
						texCoords[1] = new Point2d(1+ex,  -ey);
						texCoords[2] = new Point2d(1+ex, 1+ey);
						texCoords[3] = new Point2d( -ex, 1+ey);

					} else {
						CRLayerRenderer crr = (CRLayerRenderer) r;

						opacity = crr.getOpacity();
						if (opacity <= 0) {
							return null;
						}

						crr.multModelViewMatrix(mvMatrix);

						vertices = unProjectViewport(mvMatrix);
						if (vertices != null) {
							Matrix4d m = new Matrix4d(mvMatrix);
							m.transpose();
							for (Point3d pt : vertices) {
								m.transform(pt);
							}
						} else if (castsShadows == CastsShadows.ON) {
							castsShadows = CastsShadows.ONLY;
						}

						if (lightsCastShadows && castsShadows != CastsShadows.OFF) {
							tmp = crr.render();
							int[] shadowExpansions = getShadowExpansions(layer);
							if (shadowExpansions != null) {
								source = expand(tmp, shadowExpansions);
								tmp.dispose();
								tmp = null;
							} else {
								source = tmp;
								tmp = null;
							}
						}
						if (castsShadows != CastsShadows.ONLY) {
							sourceCR = crr.render(true, bounds, camera.getProjection3D(), mvMatrix);
						}

						if (source == null && sourceCR == null) {
							return null;
						}

						texCoords = null;
					}

					matte = renderMatte(r);

					entry = new Entry3D(indexInGroup, layer, castsShadows,
										source, sourceCR, diffusion, matte,
										mvMatrix, vertices, texCoords, opacity);
					source = null;
					sourceCR = null;
					diffusion = null;
					matte = null;
					return entry;

				} finally {
					if (source != null) source.dispose();
					if (sourceCR != null) sourceCR.dispose();
					if (diffusion != null) diffusion.dispose();
					if (matte != null) matte.dispose();
					if (tmp != null) tmp.dispose();
				}
			}
		});
	}

	private int[] getShadowExpansions(MediaLayer layer) {
		int left = getValue(layer.getShadowExpansionLeft());
		int top = getValue(layer.getShadowExpansionTop());
		int right = getValue(layer.getShadowExpansionRight());
		int bottom = getValue(layer.getShadowExpansionBottom());

		if (left > 0 || top > 0 || right > 0 || bottom > 0) {
			return new int[] { left, top, right, bottom };
		} else {
			return null;
		}
	}

	private IVideoBuffer expand(IVideoBuffer buffer, int[] expansions) {
		int left = expansions[0];
		int top = expansions[1];
		int right = expansions[2];
		int bottom = expansions[3];

		VideoBounds b = buffer.getBounds();
		b = new VideoBounds(b.x-left, b.y-top, b.width+left+right, b.height+top+bottom);

		IVideoBuffer buf2 = null;
		try {
			buf2 = support.createVideoBuffer(b);
			buf2.clear();
			support.copy(buffer, buf2);

			IVideoBuffer result = buf2;
			buf2 = null;
			return result;

		} finally {
			if (buf2 != null) buf2.dispose();
		}
	}

	private void transformAndLighting(List<Entry3D> entries) {
		if (lights.isEmpty()) {
			for (Entry3D entry : entries) {
				if (entry.castsShadows == CastsShadows.ONLY) continue;
				if (entry.lighted != null) continue;

				if (entry.texCoords != null) {
					setTextureFilter(entry.source, entry.layer);
					entry.lighted = transformWithoutLighting(entry);
				} else {
					entry.lighted = entry.sourceCR;
				}
			}
			return;
		}

		List<Entry3D> casterEntries = Util.newList();
		if (lightsCastShadows) {
			for (Entry3D entry : entries) {
				if (entry.castsShadows != CastsShadows.OFF) {
					casterEntries.add(entry);
				}
			}
		}

		entries = Util.newList(entries);
		for (Iterator<Entry3D> it = entries.iterator(); it.hasNext(); ) {
			Entry3D entry = it.next();
			if (entry.castsShadows == CastsShadows.ONLY || entry.lighted != null) {
				it.remove();
			}
		}

		// キー：影を受けるレイヤー
		// 　値：マップ（影を受けない場合、空のマップ）
		//		キー：Light
		//		　値：List<Caster>
		Map<Entry3D, Map<Light, List<Caster>>> map = Util.newMap();
		for (Entry3D entry : entries) {
			Map<Light, List<Caster>> castingLights = Util.newMap();
			if (entry.layer.isAcceptsShadows()) {
				for (Light light : lights) {
					List<Caster> actual;
					if (light.isCastsShadows() && (actual = findActualCasters(entry, light, casterEntries)) != null) {
						castingLights.put(light, actual);
					}
				}
			}
			map.put(entry, castingLights);
		}

		Set<Entry3D> all = Util.newSet(entries);
		all.addAll(casterEntries);
		for (Entry3D entry : all) {
			if (entry.source != null && (entry.diffusion == null ||
					(entry.texCoords != null && entry.castsShadows != CastsShadows.ONLY))) {
				setTextureFilter(entry.source, entry.layer);
			}
		}

		for (Entry3D entry : entries) {
			if (entry.layer.isAcceptsLights()) {
				IVideoBuffer buffer = null;
				try {
					buffer = support.createVideoBuffer(bounds);
					buffer.clear();

					Map<Light, List<Caster>> castingLights = map.get(entry);
					Set<Light> ambientLights = Util.newSet();

					for (Light light : lights) {
						if (light.getType() == LightType.AMBIENT) {
							ambientLights.add(light);
						} else {
							IVideoBuffer shadow = null;
							try {
								List<Caster> casters = castingLights.get(light);
								if (casters != null) {
									shadow = renderShadows(entry, light, casters);
								}
								parallelOrPointLight(entry, light, buffer, shadow);
							} finally {
								if (shadow != null) shadow.dispose();
							}
						}
					}

					if (!ambientLights.isEmpty()) {
						ambientLights(entry, ambientLights, buffer);
					}

					entry.lighted = buffer;
					buffer = null;

				} finally {
					if (buffer != null) buffer.dispose();
				}

			} else if (entry.layer.isAcceptsShadows() && !map.get(entry).isEmpty()) {
				IVideoBuffer buffer = null;
				try {
					if (entry.texCoords != null) {
						buffer = transformWithoutLighting(entry);
					} else {
						buffer = support.createVideoBuffer(bounds);
						support.copy(entry.sourceCR, buffer);
					}

					for (Map.Entry<Light, List<Caster>> e : map.get(entry).entrySet()) {
						IVideoBuffer shadow = null;
						try {
							shadow = renderShadows(entry, e.getKey(), e.getValue());
							shadowMultiply(entry, e.getKey(), buffer, shadow);
						} finally {
							if (shadow != null) shadow.dispose();
						}
					}

					entry.lighted = buffer;
					buffer = null;

				} finally {
					if (buffer != null) buffer.dispose();
				}

			} else {
				entry.lighted = (entry.texCoords != null)
						? transformWithoutLighting(entry) : entry.sourceCR;
			}
		}
	}

	private IVideoBuffer createDiffusionMap(IVideoBuffer source) {
		IVideoBuffer diffusion = null;
		IVideoBuffer tmpbuf = null;
		GL2 gl = context.getGL().getGL2();
		gl.glPushAttrib(GL2.GL_TEXTURE_BIT | GL2.GL_COLOR_BUFFER_BIT);
		try {
			diffusion = support.createVideoBuffer(source.getBounds());
			support.copy(source, diffusion);
			diffusion.setTextureFilter(TextureFilter.MIPMAP);

			gl.glActiveTexture(GL2.GL_TEXTURE0);

			int[] params = new int[2];
			for (int level = 0; ; ++level) {
				gl.glBindTexture(GL2.GL_TEXTURE_2D, diffusion.getTexture());
				gl.glGetTexLevelParameteriv(GL2.GL_TEXTURE_2D, level, GL2.GL_TEXTURE_WIDTH, params, 0);
				gl.glGetTexLevelParameteriv(GL2.GL_TEXTURE_2D, level, GL2.GL_TEXTURE_HEIGHT, params, 1);

				VideoBounds b = new VideoBounds(params[0], params[1]);

				// TODO サイズがもう少し大きくてもクリアしてしまった方がいいのかもしれない。
				if (b.width == 1 && b.height == 1) {
					gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
							GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_2D, diffusion.getTexture(), level);
					gl.glClearColor(0, 0, 0, 0);
					gl.glClear(GL2.GL_COLOR_BUFFER_BIT);
					break;

				} else {
					tmpbuf = support.createVideoBuffer(b);

					// ミップマップレベルの縮小比率と合わせると、
					// ブラーの半径は pow(2,level)*(2 or 4 or 8) になる。
					double radius = (level == 0) ? 2 : (level == 1) ? 4 : 8;
					float[] kernel = blurSupport.createGaussianBlurKernel(radius);

					//diffusion.setTextureWrapMode(TextureWrapMode.CLAMP_TO_EDGE);
					diffusionConvolution(gl, kernel, true, b, diffusion.getTexture(), level, tmpbuf.getTexture(), 0);

					//tmpbuf.setTextureWrapMode(TextureWrapMode.CLAMP_TO_EDGE);
					diffusionConvolution(gl, kernel, false, b, tmpbuf.getTexture(), 0, diffusion.getTexture(), level);

					tmpbuf.dispose();
					tmpbuf = null;
				}
			}

			//diffusion.setTextureWrapMode(TextureWrapMode.CLAMP_TO_BORDER);

			IVideoBuffer result = diffusion;
			diffusion = null;
			return result;

		} finally {
			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_2D, 0, 0);
			gl.glPopAttrib();

			if (diffusion != null) diffusion.dispose();
			if (tmpbuf != null) tmpbuf.dispose();
		}
	}

	private void diffusionConvolution(
			final GL2 gl, float[] kernel, boolean horz, final VideoBounds b,
			int srcTex, int srcLevel, int dstTex, int dstLevel) {

		int ksize = kernel.length;
		float[] offset = new float[ksize*2];
		if (horz) {
			for (int i = 0; i < ksize; ++i) {
				offset[i*2] = (float)(i - ksize/2) / b.width;
				offset[i*2+1] = 0;
			}
		} else {
			for (int i = 0; i < ksize; ++i) {
				offset[i*2] = 0;
				offset[i*2+1] = (float)(i - ksize/2) / b.height;
			}
		}

		gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
				GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_2D, dstTex, dstLevel);
		gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);
		gl.glBindTexture(GL2.GL_TEXTURE_2D, srcTex);

		final Set<GLUniformData> uniforms = Util.newSet();
		uniforms.add(new GLUniformData("texture", 0));
		uniforms.add(new GLUniformData("ksize", ksize));
		uniforms.add(new GLUniformData("kernel[0]", 1, FloatBuffer.wrap(kernel)));
		uniforms.add(new GLUniformData("offset[0]", 2, FloatBuffer.wrap(offset)));
		uniforms.add(new GLUniformData("lod", (float)srcLevel));

		diffusionConvolutionProgram.useProgram(new Runnable() {
			public void run() {
				for (GLUniformData data : uniforms) {
					data.setLocation(diffusionConvolutionProgram.getUniformLocation(data.getName()));
					gl.glUniform(data);
				}
				support.ortho2D(b);
				support.quad2D(b, new double[][] { {0, 0}, {1, 0}, {1, 1}, {0, 1} });
			}
		});
	}

	private List<Caster> findActualCasters(Entry3D entry, Light light, List<Entry3D> casterEntries) {
		List<Caster> actualCasters = Util.newList();

		Vec3d lightDir;
		Vec3d lightPos;
		switch (light.getType()) {
			case PARALLEL:
				lightDir = light.getDirectionInCameraView();
				lightPos = new Vec3d(
						entry.vertices[0].x-lightDir.x*1000000,
						entry.vertices[0].y-lightDir.y*1000000,
						entry.vertices[0].z-lightDir.z*1000000);
				break;

			case POINT:
			case SPOT:
				lightDir = null;
				lightPos = light.getPositionInCameraView();
				break;

			default:
				throw new IllegalArgumentException();
		}

		Vector4d lightPos4d = new Vector4d(lightPos.x, lightPos.y, lightPos.z, 1);
		if (Math.signum(entry.plane.dot(lightPos4d)) * Math.signum(entry.plane.w) == -1
				&& getValue(entry.layer.getLightTransmission()) == 0) {
			return null;
		}

		Polygon polygon = new Polygon(entry);
		for (Entry3D ce : casterEntries) {
			if (ce == entry) continue;

			Entry3D ceOrig = ce;

			if (ce.texCoords == null) {
				VideoBounds b = ce.source.getBounds();

				double expand = 16;		// TODO 拡張するサイズはどの程度必要か？

				Point3d[] vertices = new Point3d[4];
				vertices[0] = new Point3d(b.x        -expand, b.y         -expand, 0);
				vertices[1] = new Point3d(b.x+b.width+expand, b.y         -expand, 0);
				vertices[2] = new Point3d(b.x+b.width+expand, b.y+b.height+expand, 0);
				vertices[3] = new Point3d(b.x        -expand, b.y+b.height+expand, 0);

				Matrix4d m = new Matrix4d(ce.mvMatrix);
				m.transpose();
				m.transform(vertices[0]);
				m.transform(vertices[1]);
				m.transform(vertices[2]);
				m.transform(vertices[3]);

				ce = new Entry3D(ce.indexInGroup, ce.layer, ce.castsShadows,
						ce.source, ce.sourceCR, ce.diffusion, ce.matte,
						ce.mvMatrix, vertices, null, ce.opacity);
			}

			Polygon p = new Polygon(ce);
			List<Polygon> near = Util.newList();
			List<Polygon> far = Util.newList();
			partitionPolygon(polygon, p, near, far, lightPos);
			if (near.isEmpty()) continue;


			Vector3d partitionLine = null;
			boolean partitionExpand = false;
			if (!far.isEmpty()) {
				if (ssScale > 1) {
					List<Point3d> partitionPoints = Util.newList(near.get(0).vertices);
					partitionPoints.retainAll(far.get(0).vertices);

					Point3d p0 = partitionPoints.get(0);
					Point3d p1 = partitionPoints.get(1);
					double[] identity = new double[] { 1,0,0,0,0,1,0,0,0,0,1,0,0,0,0,1 };
					double[] projection = camera.getProjection3D();
					int[] viewport = new int[] { 0, 0, bounds.width, bounds.height };
					double[] winpos = new double[6];

					GLU glu = context.getGLU();
					glu.gluProject(p0.x, p0.y, p0.z, identity, 0, projection, 0, viewport, 0, winpos, 0);
					glu.gluProject(p1.x, p1.y, p1.z, identity, 0, projection, 0, viewport, 0, winpos, 3);

					partitionLine = new Vector3d(winpos[4]-winpos[1], winpos[0]-winpos[3],
											(winpos[3]-winpos[0])*winpos[1]-(winpos[4]-winpos[1])*winpos[0]);
					partitionLine.scale(1/Math.sqrt(partitionLine.x*partitionLine.x + partitionLine.y*partitionLine.y));

					double signMult = Math.signum(ce.plane.dot(lightPos4d)) * Math.signum(ce.plane.w);
					if (signMult == 1) {
						p = near.get(0);
					} else if (signMult == -1) {
						partitionExpand = true;
					}
				} else {
					p = near.get(0);
				}
			}


			Polygon target = polygon;

			Point3d p1 = p.vertices.get(p.vertices.size()-1);
			for (Point3d p2 : p.vertices) {
				Vector3d v0, v2;
				if (lightDir != null) {
					v0 = new Vector3d(lightDir.x, lightDir.y, lightDir.z);
					v2 = new Vector3d(p2);
					v2.sub(p1);
				} else {
					v0 = new Vector3d(lightPos.x, lightPos.y, lightPos.z);
					v2 = new Vector3d(p2);
					v0.sub(p1);
					v2.sub(p1);
				}

				Vector3d n = new Vector3d();
				n.cross(v0, v2);
				n.normalize();
				Vector4d plane = new Vector4d(n.x, n.y, n.z, -n.dot(new Vector3d(p1)));

				double d = 0;
				for (Point3d pt : p.vertices) {
					double dd = plane.dot(new Vector4d(pt.x, pt.y, pt.z, 1.0));
					if (Math.abs(dd) > Math.abs(d)) {
						d = dd;
					}
				}

				near.clear();
				far.clear();

				if (!partitionPolygon(plane, Math.signum(d), target, near, far)) {
					// target平面がplane上に重なっている場合。つまり、target平面が光線と並行な場合。
					// nearとfarが逆になっているのは、ライトから遠い方を残すため。
					partitionPolygon(p, target, far, near, lightPos);
				}

				if (near.isEmpty()) {
					target = null;
					break;
				} else {
					target = near.get(0);
				}

				p1 = p2;
			}

			if (target != null) {
				actualCasters.add(new Caster(ce, target.vertices, partitionLine, partitionExpand));

				if (light.getShadowDiffusion() > 0 && ce.diffusion == null) {
					ceOrig.diffusion = ce.diffusion = createDiffusionMap(ce.source);
				}
			}
		}

		return actualCasters.isEmpty() ? null : actualCasters;
	}

	private IVideoBuffer transformWithoutLighting(final Entry3D entry) {
		IVideoBuffer buffer = null;
		try {
			buffer = support.createVideoBuffer(bounds);
			buffer.clear();

			Runnable operation = new Runnable() {
				public void run() {
					GL2 gl = context.getGL().getGL2();

					gl.glViewport(0, 0, bounds.width, bounds.height);
					gl.glMatrixMode(GL2.GL_PROJECTION);
					gl.glLoadMatrixd(camera.getProjection3D(), 0);
					gl.glMatrixMode(GL2.GL_MODELVIEW);
					gl.glLoadIdentity();

					gl.glColor4f(1, 1, 1, 1);

					// 連続ラスタライズではないので頂点は必ず4つ
					gl.glBegin(GL2.GL_QUADS);
					for (int i = 0; i < 4; ++i) {
						Point2d tc = entry.texCoords[i];
						Point3d vert = entry.vertices[i];
						gl.glTexCoord2f((float)tc.x, (float)tc.y);
						gl.glVertex3f((float)vert.x, (float)vert.y, (float)vert.z);
					}
					gl.glEnd();
				}
			};

			int pushAttribs = GL2.GL_CURRENT_BIT;
			support.useFramebuffer(operation, pushAttribs, buffer, entry.source);

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

	private IVideoBuffer renderShadows(Entry3D entry, Light light, List<Caster> casters) {
		IVideoBuffer shadow = null;
		try {
			shadow = support.createVideoBuffer(bounds);
			shadow.clear(Color.WHITE);

			List<Caster> unpartitionedCasters = Util.newList();
			List<Caster> partitionedCasters = Util.newList();
			for (Caster caster : casters) {
				if (caster.partitionLine == null) {
					unpartitionedCasters.add(caster);
				} else {
					partitionedCasters.add(caster);
				}
			}

			for (int i = 0, n = unpartitionedCasters.size(); i < n; ) {
				List<Caster> sublist = unpartitionedCasters.subList(i, Math.min(n, maxTexImageUnits));
				i += sublist.size();

				renderShadow(entry, light, sublist, false, shadow);
			}

			for (int i = 0, n = partitionedCasters.size(); i < n; ) {
				List<Caster> sublist = partitionedCasters.subList(i, Math.min(n, maxTexImageUnits));
				i += sublist.size();

				renderShadow(entry, light, sublist, true, shadow);
			}

			IVideoBuffer result = shadow;
			shadow = null;
			return result;

		} finally {
			if (shadow != null) shadow.dispose();
		}
	}

	private void renderShadow(Entry3D entry, Light light, List<Caster> casters, boolean partitioned, IVideoBuffer shadow) {
		IArray<float[]> attribs = null;
		try {
			int numVertices = 0;
			for (Caster caster : casters) {
				numVertices += (caster.vertices.size()-2)*3;
			}

			attribs = arrayPools.getFloatArray(numVertices*4);
			float[] array = attribs.getArray();
			int k = 0;

			List<IVideoBuffer> buffers = Util.newList();
			float[] casterPlanes = new float[casters.size()*4];
			float[] casterMvTxInvs = new float[casters.size()*16];
			float[] casterOpacs = new float[casters.size()];
			float[] casterTrans = new float[casters.size()];
			double diffusion = light.getShadowDiffusion();		// 並行ライトでは常に0
			float[] partitionLines = partitioned ? new float[casters.size()*3] : null;
			float[] partitionExpands = partitioned ? new float[casters.size()] : null;

			for (Caster caster : casters) {
				int i = buffers.size();

				Point3d[] vertices = new Point3d[] {
						caster.vertices.get(0),
						caster.vertices.get(1),
						null
				};
				for (int j = 2; j < caster.vertices.size(); ++j, vertices[1] = vertices[2]) {
					vertices[2] = caster.vertices.get(j);
					for (Point3d vert : vertices) {
						array[k++] = (float)vert.x;
						array[k++] = (float)vert.y;
						array[k++] = (float)vert.z;
						array[k++] = i + 0.5f;
					}
				}

				if (diffusion > 0.0) {
					buffers.add(caster.entry.diffusion);
				} else {
					buffers.add(caster.entry.source);
				}

				casterPlanes[i*4  ] = (float)caster.entry.plane.x;
				casterPlanes[i*4+1] = (float)caster.entry.plane.y;
				casterPlanes[i*4+2] = (float)caster.entry.plane.z;
				casterPlanes[i*4+3] = (float)caster.entry.plane.w;

				// 影を落とすレイヤーのレイヤー座標 -> 影を落とすレイヤーのテクスチャ座標
				VideoBounds casterBounds = caster.entry.source.getBounds();
				Matrix4d m = new Matrix4d(
						1.0/casterBounds.width, 0, 0, -casterBounds.x/casterBounds.width,
						0, 1.0/casterBounds.height, 0, -casterBounds.y/casterBounds.height,
						0, 0, 1, 0,
						0, 0, 0, 1);

				// 影を落とすレイヤーのカメラ視点の座標 -> レイヤー座標
				Matrix4d casterMvInv = new Matrix4d(caster.entry.mvMatrix);
				casterMvInv.transpose();
				casterMvInv.invert();
				m.mul(casterMvInv);

				for (int col = 0; col < 4; ++col) {
					for (int row = 0; row < 4; ++row) {
						casterMvTxInvs[i*16+col*4+row] = (float)m.getElement(row, col);
					}
				}

				casterOpacs[i] = (float)caster.entry.opacity;
				casterTrans[i] = (float)(getValue(caster.entry.layer.getLightTransmission()) / 100);

				if (partitioned) {
					partitionLines[i*3  ] = (float)caster.partitionLine.x;
					partitionLines[i*3+1] = (float)caster.partitionLine.y;
					partitionLines[i*3+2] = (float)caster.partitionLine.z;
					partitionExpands[i] = caster.partitionExpand ? 1 : 0;
				}
			}

			Set<GLUniformData> uniforms = Util.newSet();
			for (int i = 0; i < buffers.size(); ++i) {
				uniforms.add(new GLUniformData("texture"+i, i));
			}
			uniforms.add(new GLUniformData("casterPlanes[0]", 4, FloatBuffer.wrap(casterPlanes)));
			uniforms.add(new GLUniformData("casterMvTxInvs[0]", 4, 4, FloatBuffer.wrap(casterMvTxInvs)));
			uniforms.add(new GLUniformData("casterOpacs[0]", 1, FloatBuffer.wrap(casterOpacs)));
			uniforms.add(new GLUniformData("casterTrans[0]", 1, FloatBuffer.wrap(casterTrans)));
			uniforms.add(new GLUniformData("viewport", 2, FloatBuffer.wrap(new float[] { bounds.width, bounds.height })));

			if (partitioned) {
				uniforms.add(new GLUniformData("partitionLines[0]", 3, FloatBuffer.wrap(partitionLines)));
				uniforms.add(new GLUniformData("partitionExpands[0]", 1, FloatBuffer.wrap(partitionExpands)));
			}

			final IShaderProgram program;
			if (light.getType() == LightType.PARALLEL) {
				Vec3d lightDir = light.getDirectionInCameraView();
				uniforms.add(new GLUniformData("lightDir", 4, FloatBuffer.wrap(new float[] {
						(float)lightDir.x, (float)lightDir.y, (float)lightDir.z, 0 })));

				program = getShadowOfParallelLightProgram(partitioned);

			} else {
				Vec3d lightPos = light.getPositionInCameraView();
				uniforms.add(new GLUniformData("lightPos", 4, FloatBuffer.wrap(new float[] {
						(float)lightPos.x, (float)lightPos.y, (float)lightPos.z, 1 })));

				if (diffusion > 0.0) {
					uniforms.add(new GLUniformData("diffusion", (float)(diffusion*0.5)));
					program = getShadowOfPointLightProgram(true, partitioned);
				} else {
					program = getShadowOfPointLightProgram(false, partitioned);
				}
			}

			final ByteBuffer directAttribs = ByteBuffer.allocateDirect(attribs.getLength()*4);	// この4は float のバイト数
			directAttribs.order(ByteOrder.nativeOrder());
			directAttribs.asFloatBuffer().put(attribs.getArray(), 0, attribs.getLength());

			Runnable operation = new Runnable() {
				public void run() {
					GL2 gl = context.getGL().getGL2();

					gl.glViewport(0, 0, bounds.width, bounds.height);
					gl.glMatrixMode(GL2.GL_PROJECTION);
					gl.glLoadMatrixd(camera.getProjection3D(), 0);
					gl.glMatrixMode(GL2.GL_MODELVIEW);
					gl.glLoadIdentity();

					gl.glEnable(GL2.GL_BLEND);
					gl.glBlendFuncSeparate(GL2.GL_ZERO, GL2.GL_SRC_COLOR, GL2.GL_ZERO, GL2.GL_ONE);
					gl.glBlendEquation(GL2.GL_FUNC_ADD);

					int attrLoc = program.getAttributeLocation("attr");
					try {
						gl.glEnableVertexAttribArray(attrLoc);

						gl.glVertexAttribPointer(attrLoc, 4, GL2.GL_FLOAT, false, 4*4, directAttribs);
												// ひとつの頂点あたり配列要素4つ　　　　// ストライドの要素数 x floatのバイト数

						gl.glDrawArrays(GL2.GL_TRIANGLES, 0, directAttribs.capacity()/(4*4));

					} finally {
						gl.glDisableVertexAttribArray(attrLoc);
					}
				}
			};

			int pushAttribs = GL2.GL_ENABLE_BIT | GL2.GL_COLOR_BUFFER_BIT;

			support.useShaderProgram(program, uniforms, operation, pushAttribs,
					shadow, buffers.toArray(new IVideoBuffer[buffers.size()]));

		} finally {
			if (attribs != null) attribs.release();
		}
	}

	private void parallelOrPointLight(final Entry3D entry, Light light, IVideoBuffer buffer, IVideoBuffer shadow) {
		// TODO 次の条件の場合、シェーダ内の計算のほとんどを省略できる。
		//
		//      (lightIntensity == 0)		// lightIntensityは負の値を取れるが、その場合は？
		//   || (lightColor.r == 0 && lightColor.g == 0 && lightColor.b == 0)
		//   || (diffuse == 0 && specular == 0)
		//   || (back && lightTransmission == 0);

		final boolean cr = (entry.texCoords == null);

		Runnable operation = new Runnable() {
			public void run() {
				GL2 gl = context.getGL().getGL2();

				gl.glViewport(0, 0, bounds.width, bounds.height);
				gl.glMatrixMode(GL2.GL_PROJECTION);
				gl.glLoadMatrixd(camera.getProjection3D(), 0);
				gl.glMatrixMode(GL2.GL_MODELVIEW);
				gl.glLoadIdentity();

				gl.glEnable(GL2.GL_BLEND);
				gl.glBlendFuncSeparate(GL2.GL_ONE, GL2.GL_ONE, GL2.GL_ONE, GL2.GL_ZERO);
				gl.glBlendEquation(GL2.GL_FUNC_ADD);

				gl.glBegin(GL2.GL_POLYGON);
				for (int i = 0; i < entry.vertices.length; ++i) {
					if (!cr) {
						Point2d tc = entry.texCoords[i];
						gl.glTexCoord2f((float)tc.x, (float)tc.y);
					}
					Point3d vert = entry.vertices[i];
					gl.glVertex3f((float)vert.x, (float)vert.y, (float)vert.z);
				}
				gl.glEnd();
			}
		};

		Set<GLUniformData> uniforms = Util.newSet();
		uniforms.add(new GLUniformData("source", 0));

		IVideoBuffer source;
		if (cr) {
			source = entry.sourceCR;
			VideoBounds b = entry.sourceCR.getBounds();
			uniforms.add(new GLUniformData("sourceOffset", 2, FloatBuffer.wrap(new float[] { (float)(-b.x), (float)(-b.y) })));
			uniforms.add(new GLUniformData("sourceSize", 2, FloatBuffer.wrap(new float[] { b.width, b.height })));
		} else {
			source = entry.source;
		}

		uniforms.add(new GLUniformData("normal", 3, FloatBuffer.wrap(
				new float[] { (float)entry.plane.x, (float)entry.plane.y, (float)entry.plane.z })));
		uniforms.add(new GLUniformData("material", 4, FloatBuffer.wrap(new float[] {
				(float)(getValue(entry.layer.getDiffuse()) / 100),
				(float)(getValue(entry.layer.getSpecular()) / 100),
				getValue(entry.layer.getShininess()).floatValue(),
				(float)(getValue(entry.layer.getMetal()) / 100)
		})));

		Vec3d lightPos;
		switch (light.getType()) {
			case PARALLEL: {
				Vec3d lightDir = light.getDirectionInCameraView();
				uniforms.add(new GLUniformData("lightVec", 3, FloatBuffer.wrap(
						new float[] { (float)-lightDir.x, (float)-lightDir.y, (float)-lightDir.z })));

				lightPos = new Vec3d(
						entry.vertices[0].x-lightDir.x*1000000,
						entry.vertices[0].y-lightDir.y*1000000,
						entry.vertices[0].z-lightDir.z*1000000);
				break;
			}

			case SPOT: {
				Vec3d lightDir = light.getDirectionInCameraView();
				double spotCutoff = Math.cos(Math.toRadians(light.getConeAngle() / 2));
				double spotSlope = 1 / Math.atan(Math.PI/2*light.getConeFeather()/100) / (1-spotCutoff);
				uniforms.add(new GLUniformData("spotVec", 3, FloatBuffer.wrap(
						new float[] { (float)-lightDir.x, (float)-lightDir.y, (float)-lightDir.z })));
				uniforms.add(new GLUniformData("spotCutoff", (float)spotCutoff));
				uniforms.add(new GLUniformData("spotSlope", (float)spotSlope));
			}
			// fall through

			case POINT: {
				lightPos = light.getPositionInCameraView();
				Vec3d lightAtten = light.getAttenuation();
				uniforms.add(new GLUniformData("lightPos", 3, FloatBuffer.wrap(
						new float[] { (float)lightPos.x, (float)lightPos.y, (float)lightPos.z })));
				uniforms.add(new GLUniformData("lightAttenuation", 3, FloatBuffer.wrap(
						new float[] { (float)lightAtten.x, (float)(lightAtten.y/10000), (float)(lightAtten.z/10000000) })));
				break;
			}

			default:
				throw new IllegalArgumentException();
		}

		double lightIntensity = light.getIntensity() / 100;
		Color lightColor = light.getColor();
		uniforms.add(new GLUniformData("lightIntensity", (float)lightIntensity));
		uniforms.add(new GLUniformData("lightColor", 3, FloatBuffer.wrap(
				new float[] { (float)lightColor.r, (float)lightColor.g, (float)lightColor.b })));

		double lightSignum = Math.signum(entry.plane.dot(new Vector4d(lightPos.x, lightPos.y, lightPos.z, 1)));
		boolean back = (lightSignum * Math.signum(entry.plane.w) == -1);
		if (back) {
			double lightTransmission = getValue(entry.layer.getLightTransmission()) / 100;
			uniforms.add(new GLUniformData("lightTransmission", (float)lightTransmission));
		}

		IShaderProgram program = getParallelOrPointLightProgram(light.getType(), shadow != null, back, cr);
		int pushAttribs = GL2.GL_ENABLE_BIT | GL2.GL_COLOR_BUFFER_BIT;

		if (shadow == null) {
			support.useShaderProgram(program, uniforms, operation, pushAttribs, buffer, source);

		} else {
			uniforms.add(new GLUniformData("shadow", 1));
			uniforms.add(new GLUniformData("shadowSize", 2, FloatBuffer.wrap(new float[] { bounds.width, bounds.height })));

			double shadowDarkness = light.getShadowDarkness() / 100;
			uniforms.add(new GLUniformData("shadowDarkness", (float)shadowDarkness));

			support.useShaderProgram(program, uniforms, operation, pushAttribs, buffer, source, shadow);
		}
	}

	private void ambientLights(final Entry3D entry, Set<Light> lights, IVideoBuffer buffer) {
		double[] ambient = new double[3];
		for (Light light : lights) {
			double intensity = light.getIntensity() / 100;
			Color color = light.getColor();
			ambient[0] += color.r*intensity;
			ambient[1] += color.g*intensity;
			ambient[2] += color.b*intensity;
		}
		double materialAmbient = getValue(entry.layer.getAmbient()) / 100;
		ambient[0] *= materialAmbient;
		ambient[1] *= materialAmbient;
		ambient[2] *= materialAmbient;


		final boolean cr = (entry.texCoords == null);

		Runnable operation = new Runnable() {
			public void run() {
				GL2 gl = context.getGL().getGL2();

				gl.glViewport(0, 0, bounds.width, bounds.height);
				gl.glMatrixMode(GL2.GL_PROJECTION);
				gl.glLoadMatrixd(camera.getProjection3D(), 0);
				gl.glMatrixMode(GL2.GL_MODELVIEW);
				gl.glLoadIdentity();

				gl.glEnable(GL2.GL_BLEND);
				gl.glBlendFuncSeparate(GL2.GL_ONE, GL2.GL_ONE, GL2.GL_ONE, GL2.GL_ZERO);
				gl.glBlendEquation(GL2.GL_FUNC_ADD);

				gl.glBegin(GL2.GL_POLYGON);
				for (int i = 0; i < entry.vertices.length; ++i) {
					if (!cr) {
						Point2d tc = entry.texCoords[i];
						gl.glTexCoord2f((float)tc.x, (float)tc.y);
					}
					Point3d vert = entry.vertices[i];
					gl.glVertex3f((float)vert.x, (float)vert.y, (float)vert.z);
				}
				gl.glEnd();
			}
		};

		Set<GLUniformData> uniforms = Util.newSet();
		uniforms.add(new GLUniformData("source", 0));

		IVideoBuffer source;
		if (cr) {
			source = entry.sourceCR;
			VideoBounds b = entry.sourceCR.getBounds();
			uniforms.add(new GLUniformData("sourceOffset", 2, FloatBuffer.wrap(new float[] { (float)(-b.x), (float)(-b.y) })));
			uniforms.add(new GLUniformData("sourceSize", 2, FloatBuffer.wrap(new float[] { b.width, b.height })));
		} else {
			source = entry.source;
		}

		uniforms.add(new GLUniformData("ambient", 3, FloatBuffer.wrap(
				new float[] { (float)ambient[0], (float)ambient[1], (float)ambient[2] })));

		IShaderProgram program = getAmbientLightsProgram(cr);
		int pushAttribs = GL2.GL_ENABLE_BIT | GL2.GL_COLOR_BUFFER_BIT;

		support.useShaderProgram(program, uniforms, operation, pushAttribs, buffer, source);
	}

	private void shadowMultiply(final Entry3D entry, Light light, IVideoBuffer buffer, IVideoBuffer shadow) {
		Runnable operation = new Runnable() {
			public void run() {
				GL2 gl = context.getGL().getGL2();

				gl.glViewport(0, 0, bounds.width, bounds.height);
				gl.glMatrixMode(GL2.GL_PROJECTION);
				gl.glLoadMatrixd(camera.getProjection3D(), 0);
				gl.glMatrixMode(GL2.GL_MODELVIEW);
				gl.glLoadIdentity();

				gl.glEnable(GL2.GL_BLEND);
				gl.glBlendFuncSeparate(GL2.GL_ZERO, GL2.GL_SRC_COLOR, GL2.GL_ZERO, GL2.GL_ONE);
				gl.glBlendEquation(GL2.GL_FUNC_ADD);

				gl.glBegin(GL2.GL_POLYGON);
				for (int i = 0; i < entry.vertices.length; ++i) {
					Point3d vert = entry.vertices[i];
					gl.glVertex3f((float)vert.x, (float)vert.y, (float)vert.z);
				}
				gl.glEnd();
			}
		};

		double shadowDarkness = light.getShadowDarkness() / 100;

		Set<GLUniformData> uniforms = Util.newSet();
		uniforms.add(new GLUniformData("shadow", 0));
		uniforms.add(new GLUniformData("shadowSize", 2, FloatBuffer.wrap(new float[] { bounds.width, bounds.height })));
		uniforms.add(new GLUniformData("shadowDarkness", (float)shadowDarkness));

		int pushAttribs = GL2.GL_ENABLE_BIT | GL2.GL_COLOR_BUFFER_BIT;
		support.useShaderProgram(shadowMultiplyProgram, uniforms, operation, pushAttribs, buffer, shadow);
	}

	private void normalBlend3D(List<Triangle> triangles, List<IVideoBuffer> buffers, IVideoBuffer composeBuffer) {
		IArray<float[]> attribs = null;
		try {
			attribs = arrayPools.getFloatArray(triangles.size()*3*6);
			float[] array = attribs.getArray();

			int k = 0;
			for (Triangle tri : triangles) {
				for (int i = 0; i < 3; ++i) {
					array[k++] = (float)tri.vertices[i].x;
					array[k++] = (float)tri.vertices[i].y;
					array[k++] = (float)tri.vertices[i].z;
					array[k++] = (float)tri.entry.opacity;
					array[k++] = tri.sourceIndex + 0.5f;
					array[k++] = tri.matteIndex + 0.5f;
				}
			}

			final ByteBuffer directAttribs = ByteBuffer.allocateDirect(attribs.getLength()*4);	// この4は float のバイト数
			directAttribs.order(ByteOrder.nativeOrder());
			directAttribs.asFloatBuffer().put(attribs.getArray(), 0, attribs.getLength());

			Set<GLUniformData> uniforms = Util.newSet();
			float[] texOffsetAndSize = new float[buffers.size()*4];
			for (ListIterator<IVideoBuffer> it = buffers.listIterator(); it.hasNext(); ) {
				int i = it.nextIndex();
				uniforms.add(new GLUniformData("texture"+i, i));

				VideoBounds b = it.next().getBounds();
				texOffsetAndSize[i*4  ] = (float)(-b.x*ssScale);
				texOffsetAndSize[i*4+1] = (float)(-b.y*ssScale);
				texOffsetAndSize[i*4+2] = b.width*ssScale;
				texOffsetAndSize[i*4+3] = b.height*ssScale;
			}
			uniforms.add(new GLUniformData("texOffsetAndSize[0]", 4, FloatBuffer.wrap(texOffsetAndSize)));
			uniforms.add(new GLUniformData("dstSize", 2, FloatBuffer.wrap(new float[] { boundsX.width, boundsX.height })));

			Runnable operation = new Runnable() {
				public void run() {
					GL2 gl = context.getGL().getGL2();
					gl.glViewport(0, 0, boundsX.width, boundsX.height);
					gl.glMatrixMode(GL2.GL_PROJECTION);
					gl.glLoadMatrixd(camera.getProjection3D(), 0);
					gl.glMatrixMode(GL2.GL_MODELVIEW);
					gl.glLoadIdentity();

					gl.glEnable(GL2.GL_BLEND);
					gl.glBlendFunc(GL2.GL_ONE, GL2.GL_ONE_MINUS_SRC_ALPHA);
					gl.glBlendEquation(GL2.GL_FUNC_ADD);

					int attr1Loc = normalBlend3DProgram.getAttributeLocation("attr1");
					int attr2Loc = normalBlend3DProgram.getAttributeLocation("attr2");
					try {
						gl.glEnableVertexAttribArray(attr1Loc);
						gl.glEnableVertexAttribArray(attr2Loc);

						// この12は「attr1Locの」ひとつの頂点あたり配列要素3つ x float のバイト数 
						directAttribs.position(12);
						gl.glVertexAttribPointer(attr2Loc, 3, GL2.GL_FLOAT, false, 6*4, directAttribs.slice());
												// ひとつの頂点あたり配列要素3つ　　　　// ストライドの要素数 x floatのバイト数

						directAttribs.position(0);
						gl.glVertexAttribPointer(attr1Loc, 3, GL2.GL_FLOAT, false, 6*4, directAttribs);

						gl.glDrawArrays(GL2.GL_TRIANGLES, 0, directAttribs.capacity()/(6*4));

					} finally {
						gl.glDisableVertexAttribArray(attr1Loc);
						gl.glDisableVertexAttribArray(attr2Loc);
					}
				}
			};

			int pushAttribs = GL2.GL_ENABLE_BIT | GL2.GL_COLOR_BUFFER_BIT;

			support.useShaderProgram(
					normalBlend3DProgram, uniforms, operation, pushAttribs,
					composeBuffer, buffers.toArray(new IVideoBuffer[buffers.size()]));

		} finally {
			if (attribs != null) {
				attribs.release();
			}
		}
	}

	private void customBlend3D(final Polygon p, IVideoBuffer composeBuffer) {
		Runnable operation = new Runnable() {
			public void run() {
				GL2 gl = context.getGL().getGL2();
				gl.glViewport(0, 0, boundsX.width, boundsX.height);
				gl.glMatrixMode(GL2.GL_PROJECTION);
				gl.glLoadMatrixd(camera.getProjection3D(), 0);
				gl.glMatrixMode(GL2.GL_MODELVIEW);
				gl.glLoadIdentity();

				gl.glBegin(GL2.GL_POLYGON);
				for (int i = 0, n = p.vertices.size(); i < n; ++i) {
					Point3d vert = p.vertices.get(i);
					gl.glVertex3f((float)vert.x, (float)vert.y, (float)vert.z);
				}
				gl.glEnd();
			}
		};

		Set<GLUniformData> uniforms = Util.newSet();
		uniforms.add(new GLUniformData("texDst", 0));
		uniforms.add(new GLUniformData("texSrc", 1));
		uniforms.add(new GLUniformData("opacity", (float)p.entry.opacity));
		uniforms.add(new GLUniformData("dstSize", 2, FloatBuffer.wrap(new float[] { boundsX.width, boundsX.height })));

		boolean matte = (p.entry.matte != null);
		if (matte) {
			uniforms.add(new GLUniformData("texMatte", 2));
		}

		BlendMode blendMode = p.entry.layer.getBlendMode();
		switch (blendMode) {
			case DISSOLVE:
				uniforms.add(new GLUniformData("dissolveSeed",
						(float)((p.entry.layer.getId().hashCode() + p.entry.layer.getName().hashCode())*100.0/Integer.MAX_VALUE)));
				break;

			case DANCING_DISSOLVE:
				uniforms.add(new GLUniformData("dissolveSeed",
						(float)((p.entry.layer.getId().hashCode() + p.entry.layer.getName().hashCode())*100.0/Integer.MAX_VALUE
								+ context.getTime().toSecond())));
				blendMode = BlendMode.DISSOLVE;
				break;
		}

		VideoBounds b = p.entry.lighted.getBounds();
		uniforms.add(new GLUniformData("srcOffset", 2, FloatBuffer.wrap(new float[] { (float)(-b.x*ssScale), (float)(-b.y*ssScale) })));
		uniforms.add(new GLUniformData("srcSize", 2, FloatBuffer.wrap(new float[] { b.width*ssScale, b.height*ssScale })));

		IShaderProgram program = getCustomBlend3DProgram(blendMode, matte);

		IVideoBuffer copy = null;
		try {
			copy = support.createVideoBuffer(boundsX);
			support.copy(composeBuffer, copy);
			if (matte) {
				support.useShaderProgram(program, uniforms, operation, 0, composeBuffer, copy, p.entry.lighted, p.entry.matte);
			} else {
				support.useShaderProgram(program, uniforms, operation, 0, composeBuffer, copy, p.entry.lighted);
			}
		} finally {
			if (copy != null) copy.dispose();
		}
	}

	private void compose2D(ComposeGroup group, IVideoBuffer composeBuffer) {
		List<Entry2D> entries = Util.newList();
		try {
			int numTextures = 0;
			for (ListIterator<GeneralLayerRenderer> it = group.renderers.listIterator(); it.hasNext(); ) {
				GeneralLayerRenderer r = it.next();
				BlendMode blendMode = r.getLayer().getBlendMode();

				if (blendMode == BlendMode.NORMAL) {
					// 使用するテクスチャ数が2増える状況でmaxTexImageUnitsまでの残りが1の場合、
					// そのまま続行することができないので、イテレータをひとつ巻き戻した上でひとつ前までの分だけ処理する。
					int incTextures = (r.getMatteIds() != null) ? 2 : 1;
					if (incTextures == 2 && numTextures == maxTexImageUnits-1) {
						it.previous();
					} else {
						Entry2D entry = render2DLayer(r);
						if (entry != null) {
							entries.add(entry);
							numTextures += incTextures;
							if (numTextures < maxTexImageUnits && it.hasNext()) {
								continue;
							}
						} else if (it.hasNext()) {
							continue;
						}
					}
				}

				if (!entries.isEmpty()) {
					normalBlend2D(entries, composeBuffer);
					for (Entry2D entry : entries) entry.source.dispose();
					entries.clear();
					numTextures = 0;
				}

				if (blendMode != BlendMode.NORMAL) {
					Entry2D entry = render2DLayer(r);
					if (entry != null) {
						try {
							customBlend2D(entry, composeBuffer);
						} finally {
							entry.source.dispose();
						}
					}
				}
			}
		} finally {
			for (Entry2D entry : entries) entry.source.dispose();
		}
	}

	private Entry2D render2DLayer(final GeneralLayerRenderer r) {
		return context.saveAndExecute(new WrappedOperation<Entry2D>() {
			public Entry2D execute() {
				final MediaLayer layer = r.getLayer();
				MotionBlur mblur = composition.isMotionBlurEnabled()
						? LayerNature.getMotionBlur(layer) : MotionBlur.NONE;

				IVideoBuffer source = null;
				IVideoBuffer matte = null;

				try {
					Point3d[] vertices = new Point3d[4];
					Point2d[] texCoords = new Point2d[4];
					double opacity;

					if (r instanceof NormalLayerRenderer) {
						final NormalLayerRenderer nr = (NormalLayerRenderer) r;

						if (mblur == MotionBlur.NONE) {
							VideoBounds b = nr.calcBounds(true);
							if (b.isEmpty()) {
								return null;
							}

							opacity = nr.getOpacity();
							if (opacity <= 0) {
								return null;
							}

							double[] mvMatrix = new double[16];
							System.arraycopy(camera.getModelView2D(), 0, mvMatrix, 0, 16);
							nr.multModelViewMatrix(mvMatrix);

							if (!transform2DLayerBounds(b, mvMatrix, vertices, texCoords)) {
								return null;
							}

							source = nr.render(true, composition.isFrameBlendEnabled());
							if (!source.getBounds().equals(b)) {
								logger.warn("bounds differ");
							}

							setTextureFilter(source, layer);

						} else {
							MotionBlurSampler2 sampler = new MotionBlurSampler2() {
								private final double[] mvMatrix = new double[16];

								public VideoBounds calcBounds() {
									return nr.calcBounds(true);
								}

								public double getOpacity() {
									return nr.getOpacity();
								}

								public boolean transform(VideoBounds bounds, Point3d[] vertices, Point2d[] texCoords) {
									System.arraycopy(camera.getModelView2D(), 0, mvMatrix, 0, 16);
									nr.multModelViewMatrix(mvMatrix);
									return transform2DLayerBounds(bounds, mvMatrix, vertices, texCoords);
								}

								public IVideoBuffer sample() {
									IVideoBuffer buffer = nr.render(true, composition.isFrameBlendEnabled());
									setTextureFilter(buffer, layer);
									return buffer;
								}
							};

							source = motionBlur2(mblur, sampler);
							if (source == null) {
								return null;
							}

							VideoBounds b = source.getBounds();
							vertices[0] = new Point3d(b.x        , b.y         , 0);
							vertices[1] = new Point3d(b.x+b.width, b.y         , 0);
							vertices[2] = new Point3d(b.x+b.width, b.y+b.height, 0);
							vertices[3] = new Point3d(b.x        , b.y+b.height, 0);

							texCoords[0] = new Point2d(0, 0);
							texCoords[1] = new Point2d(1, 0);
							texCoords[2] = new Point2d(1, 1);
							texCoords[3] = new Point2d(0, 1);

							opacity = 1.0;
						}

					} else {
						final CRLayerRenderer crr = (CRLayerRenderer) r;

						if (mblur == MotionBlur.NONE) {
							opacity = crr.getOpacity();
							if (opacity <= 0) {
								return null;
							}

							double[] mvMatrix = new double[16];
							System.arraycopy(camera.getModelView2D(), 0, mvMatrix, 0, 16);
							crr.multModelViewMatrix(mvMatrix);
							source = crr.render(true, bounds, camera.getProjection2D(), mvMatrix);

						} else {
							MotionBlurSampler1 sampler = new MotionBlurSampler1() {
								private final double[] mvMatrix = new double[16];

								public IVideoBuffer sample() {
									System.arraycopy(camera.getModelView2D(), 0, mvMatrix, 0, 16);
									crr.multModelViewMatrix(mvMatrix);
									return crr.render(true, bounds, camera.getProjection2D(), mvMatrix);
								}

								public double getOpacity() {
									return crr.getOpacity();
								}
							};

							source = motionBlur1(null, sampler);
							if (source == null) {
								return null;
							}

							opacity = 1.0;
						}

						VideoBounds b = source.getBounds();
						vertices[0] = new Point3d(b.x        , b.y         , 0);
						vertices[1] = new Point3d(b.x+b.width, b.y         , 0);
						vertices[2] = new Point3d(b.x+b.width, b.y+b.height, 0);
						vertices[3] = new Point3d(b.x        , b.y+b.height, 0);

						texCoords[0] = new Point2d(0, 0);
						texCoords[1] = new Point2d(1, 0);
						texCoords[2] = new Point2d(1, 1);
						texCoords[3] = new Point2d(0, 1);
					}

					matte = renderMatte(r);

					Entry2D entry = new Entry2D(layer, source, matte, vertices, texCoords, opacity);
					source = null;
					matte = null;
					return entry;

				} finally {
					if (source != null) source.dispose();
					if (matte != null) matte.dispose();
				}
			}
		});
	}

	private boolean transform2DLayerBounds(VideoBounds b, double[] mvMatrix, Point3d[] vertices, Point2d[] texCoords) {
		double expand = 16;		// TODO 拡張するサイズはどの程度必要か？
		double ex = expand/b.width;
		double ey = expand/b.height;

		vertices[0] = new Point3d(b.x        -expand, b.y         -expand, 0);
		vertices[1] = new Point3d(b.x+b.width+expand, b.y         -expand, 0);
		vertices[2] = new Point3d(b.x+b.width+expand, b.y+b.height+expand, 0);
		vertices[3] = new Point3d(b.x        -expand, b.y+b.height+expand, 0);

		Matrix4d m = new Matrix4d(mvMatrix);
		m.transpose();
		m.transform(vertices[0]);
		m.transform(vertices[1]);
		m.transform(vertices[2]);
		m.transform(vertices[3]);

		texCoords[0] = new Point2d( -ex,  -ey);
		texCoords[1] = new Point2d(1+ex,  -ey);
		texCoords[2] = new Point2d(1+ex, 1+ey);
		texCoords[3] = new Point2d( -ex, 1+ey);


		// vertices を囲む矩形と bounds が交差するかテストし、
		// 交差している場合は vertices が作るポリゴンと bounds が交差するかテストする。

		double[] b2 = new double[] {
				Double.POSITIVE_INFINITY,	// left
				Double.POSITIVE_INFINITY,	// top
				Double.NEGATIVE_INFINITY,	// right
				Double.NEGATIVE_INFINITY	// bottom
		};
		for (Point3d v : vertices) {
			b2[0] = Math.min(b2[0], v.x);
			b2[1] = Math.min(b2[1], v.y);
			b2[2] = Math.max(b2[2], v.x);
			b2[3] = Math.max(b2[3], v.y);
		}
		if (!(b2[0] < bounds.x + bounds.width) && (b2[2] > bounds.x)
			&& (b2[1] < bounds.y + bounds.height) && (b2[3] > bounds.y)) {
			return false;
		}

		GLUtessellator tess = GLU.gluNewTess();
		try {
			final boolean[] intersects = new boolean[1];
			GLUtessellatorCallback callback = new GLUtessellatorCallbackAdapter() {
				public void end() { intersects[0] = true; }
				public void combine(double[] coords, Object[] data, float[] weight, Object[] outData) {
					outData[0] = new double[] { coords[0], coords[1], coords[2] };
				}
			};
			GLU.gluTessCallback(tess, GLU.GLU_TESS_BEGIN, callback);
			GLU.gluTessCallback(tess, GLU.GLU_TESS_END, callback);
			GLU.gluTessCallback(tess, GLU.GLU_TESS_VERTEX, callback);
			GLU.gluTessCallback(tess, GLU.GLU_TESS_COMBINE, callback);
			GLU.gluTessCallback(tess, GLU.GLU_TESS_ERROR, callback);

			GLU.gluTessProperty(tess, GLU.GLU_TESS_WINDING_RULE, GLU.GLU_TESS_WINDING_ABS_GEQ_TWO);
			GLU.gluTessProperty(tess, GLU.GLU_TESS_TOLERANCE, 0);

			@SuppressWarnings("unchecked")
			List<Point3d>[] contours = new List[] {
				Util.newList(Arrays.asList(vertices)),
				Arrays.asList(
						new Point3d(bounds.x, bounds.y, 0),
						new Point3d(bounds.x+bounds.width, bounds.y, 0),
						new Point3d(bounds.x+bounds.width, bounds.y+bounds.height, 0),
						new Point3d(bounds.x, bounds.y+bounds.height, 0))
			};
			for (int i = 0; i < 2; ++i) {
				GLU.gluTessBeginPolygon(tess, null);
				for (List<Point3d> contour : contours) {
					GLU.gluTessBeginContour(tess);
					for (Point3d v : contour) {
						double[] vertex = new double[] { v.x, v.y, 0 };
						GLU.gluTessVertex(tess, vertex, 0, vertex);
					}
					GLU.gluTessEndContour(tess);
				}
				GLU.gluTessEndPolygon(tess);

				if (intersects[0]) {
					return true;
				}
				if (i == 0) {
					Collections.reverse(contours[0]);
				}
			}
		} finally {
			GLU.gluDeleteTess(tess);
		}
		return false;
	}

	private void normalBlend2D(List<Entry2D> entries, IVideoBuffer composeBuffer) {
		IArray<float[]> attribs = null;
		try {
			attribs = arrayPools.getFloatArray(entries.size()*4*7);
			float[] array = attribs.getArray();

			List<IVideoBuffer> buffers = Util.newList();

			int k = 0;
			for (Entry2D entry : entries) {
				int sourceIndex = buffers.size();
				buffers.add(entry.source);

				int matteIndex = -1;
				if (entry.matte != null) {
					matteIndex = buffers.size();
					buffers.add(entry.matte);
				}

				for (int i = 0; i < 4; ++i) {
					array[k++] = (float)entry.vertices[i].x;
					array[k++] = (float)entry.vertices[i].y;
					array[k++] = (float)entry.opacity;
					array[k++] = (float)entry.texCoords[i].x;
					array[k++] = (float)entry.texCoords[i].y;
					array[k++] = sourceIndex + 0.5f;
					array[k++] = matteIndex + 0.5f;
				}
			}

			final ByteBuffer directAttribs = ByteBuffer.allocateDirect(attribs.getLength()*4);	// この4は float のバイト数
			directAttribs.order(ByteOrder.nativeOrder());
			directAttribs.asFloatBuffer().put(attribs.getArray(), 0, attribs.getLength());

			Set<GLUniformData> uniforms = Util.newSet();
			for (int i = 0, n = buffers.size(); i < n; ++i) {
				uniforms.add(new GLUniformData("texture"+i, i));
			}
			uniforms.add(new GLUniformData("dstSize", 2, FloatBuffer.wrap(new float[] { bounds.width, bounds.height })));

			Runnable operation = new Runnable() {
				public void run() {
					GL2 gl = context.getGL().getGL2();
					gl.glViewport(0, 0, bounds.width, bounds.height);
					gl.glMatrixMode(GL2.GL_PROJECTION);
					gl.glLoadMatrixd(camera.getProjection2D(), 0);
					gl.glMatrixMode(GL2.GL_MODELVIEW);
					gl.glLoadIdentity();

					gl.glEnable(GL2.GL_BLEND);
					gl.glBlendFunc(GL2.GL_ONE, GL2.GL_ONE_MINUS_SRC_ALPHA);
					gl.glBlendEquation(GL2.GL_FUNC_ADD);

					int attr1Loc = normalBlend2DProgram.getAttributeLocation("attr1");
					int attr2Loc = normalBlend2DProgram.getAttributeLocation("attr2");
					try {
						gl.glEnableVertexAttribArray(attr1Loc);
						gl.glEnableVertexAttribArray(attr2Loc);

						// この12は「attr1Locの」ひとつの頂点あたり配列要素3つ x float のバイト数 
						directAttribs.position(12);
						gl.glVertexAttribPointer(attr2Loc, 4, GL2.GL_FLOAT, false, 7*4, directAttribs.slice());
												// ひとつの頂点あたり配列要素4つ　　　　// ストライドの要素数 x floatのバイト数

						directAttribs.position(0);
						gl.glVertexAttribPointer(attr1Loc, 3, GL2.GL_FLOAT, false, 7*4, directAttribs);

						gl.glDrawArrays(GL2.GL_QUADS, 0, directAttribs.capacity()/(7*4));

					} finally {
						gl.glDisableVertexAttribArray(attr1Loc);
						gl.glDisableVertexAttribArray(attr2Loc);
					}
				}
			};

			int pushAttribs = GL2.GL_ENABLE_BIT | GL2.GL_COLOR_BUFFER_BIT;

			support.useShaderProgram(
					normalBlend2DProgram, uniforms, operation, pushAttribs,
					composeBuffer, buffers.toArray(new IVideoBuffer[buffers.size()]));

		} finally {
			if (attribs != null) attribs.release();
		}
	}

	private void customBlend2D(final Entry2D entry, IVideoBuffer composeBuffer) {
		Runnable operation = new Runnable() {
			public void run() {
				GL2 gl = context.getGL().getGL2();

				switch (entry.layer.getBlendMode()) {
					case STENCIL_ALPHA:
					case STENCIL_LUMA:
						gl.glClearColor(0, 0, 0, 0);
						gl.glClear(GL2.GL_COLOR_BUFFER_BIT);
						break;
				}

				gl.glViewport(0, 0, bounds.width, bounds.height);
				gl.glMatrixMode(GL2.GL_PROJECTION);
				gl.glLoadMatrixd(camera.getProjection2D(), 0);
				gl.glMatrixMode(GL2.GL_MODELVIEW);
				gl.glLoadIdentity();

				gl.glBegin(GL2.GL_QUADS);
				for (int i = 0; i < 4; ++i) {
					gl.glTexCoord2f((float)entry.texCoords[i].x, (float)entry.texCoords[i].y);
					gl.glVertex2f((float)entry.vertices[i].x, (float)entry.vertices[i].y);
				}
				gl.glEnd();
			}
		};

		Set<GLUniformData> uniforms = Util.newSet();
		uniforms.add(new GLUniformData("texDst", 0));
		uniforms.add(new GLUniformData("texSrc", 1));
		uniforms.add(new GLUniformData("opacity", (float)entry.opacity));
		uniforms.add(new GLUniformData("dstSize", 2, FloatBuffer.wrap(new float[] { bounds.width, bounds.height })));

		boolean matte = (entry.matte != null);
		if (matte) {
			uniforms.add(new GLUniformData("texMatte", 2));
		}

		BlendMode blendMode = entry.layer.getBlendMode();
		switch (blendMode) {
			case DISSOLVE:
				uniforms.add(new GLUniformData("dissolveSeed",
						(float)((entry.layer.getId().hashCode() + entry.layer.getName().hashCode())*100.0/Integer.MAX_VALUE)));
				break;

			case DANCING_DISSOLVE:
				uniforms.add(new GLUniformData("dissolveSeed",
						(float)((entry.layer.getId().hashCode() + entry.layer.getName().hashCode())*100.0/Integer.MAX_VALUE
								+ context.getTime().toSecond())));
				blendMode = BlendMode.DISSOLVE;
				break;
		}

		IShaderProgram program = getCustomBlend2DProgram(blendMode, matte);
		int pushAttribs = GL2.GL_COLOR_BUFFER_BIT;

		IVideoBuffer copy = null;
		try {
			copy = support.createVideoBuffer(bounds);
			support.copy(composeBuffer, copy);
			if (matte) {
				support.useShaderProgram(program, uniforms, operation, pushAttribs, composeBuffer, copy, entry.source, entry.matte);
			} else {
				support.useShaderProgram(program, uniforms, operation, pushAttribs, composeBuffer, copy, entry.source);
			}
		} finally {
			if (copy != null) copy.dispose();
		}
	}

	private void setTextureFilter(IVideoBuffer buffer, MediaLayer layer) {
		Quality quality = (resolution.scale < 1) ? Quality.DRAFT : LayerNature.getQuality(layer);
		switch (quality) {
			case BEST:
				buffer.setTextureFilter(TextureFilter.MIPMAP);
				break;

			case NORMAL:
				buffer.setTextureFilter(TextureFilter.LINEAR);
				break;

			case DRAFT:
				buffer.setTextureFilter(TextureFilter.NEAREST);
				break;

			default:
				throw new RuntimeException("unknown quality: " + quality);
		}
	}

	private <V> V getValue(AnimatableValue<V> avalue) {
		Time time = context.getTime();

		@SuppressWarnings("unchecked")
		Map<Time, V> map = (Map<Time, V>) valueCache.get(avalue);
		if (map != null) {
			V value = map.get(time);
			if (value != null) {
				return value;
			}
		} else {
			map = Util.newMap();
			valueCache.put(avalue, map);
		}

		V value = avalue.value(context);
		map.put(time, value);

		return value;
	}

	private IVideoBuffer renderMatte(GeneralLayerRenderer renderer) {
		List<String> matteIds = renderer.getMatteIds();
		if (matteIds == null) {
			return null;
		}

		StringBuilder cacheKeySb = new StringBuilder(matteIds.get(0)); 
		for (String matteId : matteIds.subList(1, matteIds.size())) {
			cacheKeySb.append(":").append(matteId);
		}
		String cacheKey = cacheKeySb.toString();

		IVideoBuffer buffer = matteBufferCache.get(cacheKey);
		if (buffer != null) {
			return buffer;
		}

		List<IVideoBuffer> buffers = Util.newList();
		List<TrackMatte> trackMattes = Util.newList();
		try {
			buffer = support.createVideoBuffer(bounds);
			buffer.clear(Color.WHITE);

			for (Iterator<String> it = matteIds.iterator(); it.hasNext(); ) {
				final MatteLayerRenderers mlr = matteLayerRenderers.get(it.next());

				buffers.add(context.saveAndExecute(new WrappedOperation<IVideoBuffer>() {
					public IVideoBuffer execute() {
						context.setTime(baseTime);
						VideoLayerComposer composer = new VideoLayerComposer(
								context, support, accumSupport, convolution,
								blurSupport, arrayPools, shaders, glGlobal);
						composer.addLayerRenderers(mlr.getRenderers());
						return composer.compose();
					}
				}));

				trackMattes.add(mlr.getLayer().getTrackMatte());

				if (buffers.size() == maxTexImageUnits || !it.hasNext()) {
					matteMultiply(buffer, buffers, trackMattes);
					for (IVideoBuffer vb : buffers) vb.dispose();
					buffers.clear();
					trackMattes.clear();
				}
			}

			matteBufferCache.put(cacheKey, buffer);
			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
			for (IVideoBuffer vb : buffers) vb.dispose();
		}
	}

	private void matteMultiply(IVideoBuffer dstBuffer,
			List<IVideoBuffer> buffers, List<TrackMatte> trackMattes) {

		Runnable operation = new Runnable() {
			public void run() {
				GL2 gl = context.getGL().getGL2();
				gl.glEnable(GL2.GL_BLEND);
				gl.glBlendFunc(GL2.GL_ZERO, GL2.GL_SRC_ALPHA);
				gl.glBlendEquation(GL2.GL_FUNC_ADD);

				support.ortho2D(bounds);
				support.quad2D(bounds);
			}
		};

		Set<GLUniformData> uniforms = Util.newSet();
		float[] trackMattesData = new float[buffers.size()];
		for (int i = 0, n = buffers.size(); i < n; ++i) {
			uniforms.add(new GLUniformData("texture"+i, i));
			trackMattesData[i] = trackMattes.get(i).ordinal() + 0.5f;
		}
		uniforms.add(new GLUniformData("trackMattes[0]", 1, FloatBuffer.wrap(trackMattesData)));
		uniforms.add(new GLUniformData("numMattes", trackMattesData.length));
		uniforms.add(new GLUniformData("size", 2, FloatBuffer.wrap(new float[] { bounds.width, bounds.height })));

		int pushAttribs = GL2.GL_ENABLE_BIT | GL2.GL_COLOR_BUFFER_BIT;

		support.useShaderProgram(matteMultiplyProgram, uniforms, operation, pushAttribs,
				dstBuffer, buffers.toArray(new IVideoBuffer[buffers.size()]));
	}

	private interface MotionBlurSampler1 {
		IVideoBuffer sample();
		double getOpacity();
	}

	private IVideoBuffer motionBlur1(IVideoBuffer dstBuffer, MotionBlurSampler1 sampler) {
		IVideoBuffer accumBuf = null;
		List<IVideoBuffer> buffers = Util.newList();
		try {
			ColorMode ctxColorMode = context.getColorMode(); 
			VideoBounds dstBounds = (dstBuffer != null) ? dstBuffer.getBounds() : bounds;

			double shutterAngle = composition.getMotionBlurShutterAngle();
			double shutterPhase = composition.getMotionBlurShutterPhase();
			int samples = composition.getMotionBlurSamples();
			Time baseTime = context.getTime();
			Time frameDuration = context.getVideoFrameDuration();
			List<Double> weights = Util.newList();
			int maxSources = accumSupport.getMaxSourcesAtATime();

			for (int i = 0; i < samples; ++i) {
				double angle = shutterPhase + shutterAngle*i/samples;
				Time time = baseTime.add(
						new Time((long)(frameDuration.timeValue*angle/360), frameDuration.timeScale));
				context.setTime(time);

				double opacity = sampler.getOpacity();
				if (opacity > 0) {
					buffers.add(sampler.sample());
					weights.add(opacity / samples);
				}

				int n = buffers.size();
				if (n == maxSources || (n > 0 && i == samples-1)) {
					if (accumBuf == null) {
						accumBuf = ctxColorMode.isFloat()
								? support.createVideoBuffer(dstBounds)
								: support.createVideoBuffer(dstBounds, ColorMode.RGBA16_FLOAT);
						accumBuf.clear();
					}
					accumSupport.accumulate(buffers, weights, accumBuf);
					for (IVideoBuffer vb : buffers) vb.dispose();
					buffers.clear();
					weights.clear();
				}
			}
			if (accumBuf == null) {
				return null;
			}

			if (accumBuf.getColorMode() == ctxColorMode) {
				// 元の dstBuffer を破棄するのは呼び出し側の責任
				//if (dstBuffer != null) dstBuffer.dispose();
				dstBuffer = accumBuf;
				accumBuf = null;

			} else if (dstBuffer != null) {
				support.copy(accumBuf, dstBuffer);

			} else {
				IVideoBuffer copy = null;
				try {
					copy = support.createVideoBuffer(dstBounds);
					support.copy(accumBuf, copy);
					dstBuffer = copy;
					copy = null;
				} finally {
					if (copy != null) copy.dispose();
				}
			}

			return dstBuffer;

		} finally {
			context.setTime(baseTime);
			for (IVideoBuffer vb : buffers) vb.dispose();
			if (accumBuf != null) accumBuf.dispose();
		}
	}

	private interface MotionBlurSampler2 {
		VideoBounds calcBounds();
		double getOpacity();
		boolean transform(VideoBounds bounds, Point3d[] vertices, Point2d[] texCoords);
		IVideoBuffer sample();
	}

	private IVideoBuffer motionBlur2(MotionBlur mblur, MotionBlurSampler2 sampler) {
		if (mblur == MotionBlur.NONE) throw new IllegalArgumentException();

		IVideoBuffer accumBuf = null;
		List<IVideoBuffer> buffers = Util.newList();
		try {
			ColorMode ctxColorMode = context.getColorMode(); 

			double shutterAngle = composition.getMotionBlurShutterAngle();
			double shutterPhase = composition.getMotionBlurShutterPhase();
			int samples = composition.getMotionBlurSamples();
			Time baseTime = context.getTime();
			Time frameDuration = context.getVideoFrameDuration();

			List<Point3d[]> verticesList = Util.newList();
			List<Point2d[]> texCoordsList = Util.newList();
			List<Double> weightList = Util.newList();

			switch (mblur) {
				case TRANSFORM: {
					VideoBounds b = sampler.calcBounds();
					if (b.isEmpty()) {
						return null;
					}

					for (int i = 0; i < samples; ++i) {
						double angle = shutterPhase + shutterAngle*i/samples;
						Time time = baseTime.add(
								new Time((long)(frameDuration.timeValue*angle/360), frameDuration.timeScale));
						context.setTime(time);

						double opacity = sampler.getOpacity();
						if (opacity > 0) {
							Point3d[] vertices = new Point3d[4];
							Point2d[] texCoords = new Point2d[4];
							if (sampler.transform(b, vertices, texCoords)) {
								verticesList.add(vertices);
								texCoordsList.add(texCoords);
								weightList.add(opacity / samples);
							}
						}
					}

					if (verticesList.isEmpty()) {
						return null;
					}

					context.setTime(baseTime);

					IVideoBuffer buffer = sampler.sample();
					buffers.add(buffer);

					if (!buffer.getBounds().equals(b)) {
						logger.warn("bounds differ");
					}

					accumBuf = ctxColorMode.isFloat()
							? support.createVideoBuffer(bounds)
							: support.createVideoBuffer(bounds, ColorMode.RGBA16_FLOAT);
					accumBuf.clear();

					motionBlur2Accumulate1(accumBuf, buffer, verticesList, texCoordsList, weightList);
					break;
				}

				case FULL:
					for (int i = 0; i < samples; ++i) {
						double angle = shutterPhase + shutterAngle*i/samples;
						Time time = baseTime.add(
								new Time((long)(frameDuration.timeValue*angle/360), frameDuration.timeScale));
						context.setTime(time);

						VideoBounds b;
						double opacity;
						if (!(b = sampler.calcBounds()).isEmpty()
								&& (opacity = sampler.getOpacity()) > 0) {

							Point3d[] vertices = new Point3d[4];
							Point2d[] texCoords = new Point2d[4];
							if (sampler.transform(b, vertices, texCoords)) {
								verticesList.add(vertices);
								texCoordsList.add(texCoords);
								weightList.add(opacity / samples);

								IVideoBuffer buffer = sampler.sample();
								buffers.add(buffer);

								if (!buffer.getBounds().equals(b)) {
									logger.warn("bounds differ");
								}
							}
						}

						int n = buffers.size();
						if (n == maxTexImageUnits || (n > 0 && i == samples-1)) {
							if (accumBuf == null) {
								accumBuf = ctxColorMode.isFloat()
										? support.createVideoBuffer(bounds)
										: support.createVideoBuffer(bounds, ColorMode.RGBA16_FLOAT);
								accumBuf.clear();
							}
							motionBlur2Accumulate2(accumBuf, buffers, verticesList, texCoordsList, weightList);
							for (IVideoBuffer vb : buffers) vb.dispose();
							buffers.clear();
							verticesList.clear();
							texCoordsList.clear();
							weightList.clear();
						}
					}
					if (accumBuf == null) {
						return null;
					}
					break;
			}

			IVideoBuffer result;
			if (accumBuf.getColorMode() == ctxColorMode) {
				result = accumBuf;
				accumBuf = null;
			} else {
				IVideoBuffer copy = null;
				try {
					copy = support.createVideoBuffer(bounds);
					support.copy(accumBuf, copy);
					result = copy;
					copy = null;
				} finally {
					if (copy != null) copy.dispose();
				}
			}
			return result;

		} finally {
			context.setTime(baseTime);
			for (IVideoBuffer vb : buffers) vb.dispose();
			if (accumBuf != null) accumBuf.dispose();
		}
	}

	private void motionBlur2Accumulate1(
			IVideoBuffer accumBuffer, IVideoBuffer srcBuffer,
			final List<Point3d[]> verticesList, final List<Point2d[]> texCoordsList, final List<Double> weightList) {

		Runnable operation = new Runnable() {
			public void run() {
				GL2 gl = context.getGL().getGL2();
				gl.glViewport(0, 0, bounds.width, bounds.height);
				gl.glMatrixMode(GL2.GL_PROJECTION);
				gl.glLoadMatrixd(camera.getProjection2D(), 0);
				gl.glMatrixMode(GL2.GL_MODELVIEW);
				gl.glLoadIdentity();

				gl.glEnable(GL2.GL_BLEND);
				gl.glBlendFunc(GL2.GL_ONE, GL2.GL_ONE);
				gl.glBlendEquation(GL2.GL_FUNC_ADD);

				gl.glBegin(GL2.GL_QUADS);
				for (int i = 0, n = verticesList.size(); i < n; ++i) {
					Point3d[] vertices = verticesList.get(i);
					Point2d[] texCoords = texCoordsList.get(i);
					float weight = weightList.get(i).floatValue();
					for (int j = 0; j < 4; ++j) {
						gl.glColor4f(weight, weight, weight, weight);
						gl.glTexCoord2f((float)texCoords[j].x, (float)texCoords[j].y);
						gl.glVertex2f((float)vertices[j].x, (float)vertices[j].y);
					}
				}
				gl.glEnd();
			}
		};

		int pushAttribs = GL2.GL_ENABLE_BIT | GL2.GL_COLOR_BUFFER_BIT | GL2.GL_CURRENT_BIT;
		support.useFramebuffer(operation, pushAttribs, accumBuffer, srcBuffer);
	}

	private void motionBlur2Accumulate2(
			IVideoBuffer accumBuffer, List<IVideoBuffer> srcBuffers,
			List<Point3d[]> verticesList, List<Point2d[]> texCoordsList, List<Double> weightList) {

		IArray<float[]> attribs = null;
		try {
			attribs = arrayPools.getFloatArray(srcBuffers.size()*4*6);
			float[] array = attribs.getArray();

			List<IVideoBuffer> buffers = Util.newList();

			for (int i = 0, k = 0, n = srcBuffers.size(); i < n; ++i) {
				Point3d[] vertices = verticesList.get(i);
				Point2d[] texCoords = texCoordsList.get(i);
				float weight = weightList.get(i).floatValue();
				for (int j = 0; j < 4; ++j) {
					array[k++] = (float)vertices[j].x;
					array[k++] = (float)vertices[j].y;
					array[k++] = weight;
					array[k++] = (float)texCoords[j].x;
					array[k++] = (float)texCoords[j].y;
					array[k++] = i + 0.5f;
				}
				buffers.add(srcBuffers.get(i));
			}

			final ByteBuffer directAttribs = ByteBuffer.allocateDirect(attribs.getLength()*4);	// この4は float のバイト数
			directAttribs.order(ByteOrder.nativeOrder());
			directAttribs.asFloatBuffer().put(attribs.getArray(), 0, attribs.getLength());

			Set<GLUniformData> uniforms = Util.newSet();
			for (int i = 0, n = buffers.size(); i < n; ++i) {
				uniforms.add(new GLUniformData("texture"+i, i));
			}

			Runnable operation = new Runnable() {
				public void run() {
					GL2 gl = context.getGL().getGL2();
					gl.glViewport(0, 0, bounds.width, bounds.height);
					gl.glMatrixMode(GL2.GL_PROJECTION);
					gl.glLoadMatrixd(camera.getProjection2D(), 0);
					gl.glMatrixMode(GL2.GL_MODELVIEW);
					gl.glLoadIdentity();

					gl.glEnable(GL2.GL_BLEND);
					gl.glBlendFunc(GL2.GL_ONE, GL2.GL_ONE);
					gl.glBlendEquation(GL2.GL_FUNC_ADD);

					int attr1Loc = mblur2Accum2Program.getAttributeLocation("attr1");
					int attr2Loc = mblur2Accum2Program.getAttributeLocation("attr2");
					try {
						gl.glEnableVertexAttribArray(attr1Loc);
						gl.glEnableVertexAttribArray(attr2Loc);

						// この12は「attr1Locの」ひとつの頂点あたり配列要素3つ x float のバイト数 
						directAttribs.position(12);
						gl.glVertexAttribPointer(attr2Loc, 3, GL2.GL_FLOAT, false, 6*4, directAttribs.slice());
												// ひとつの頂点あたり配列要素3つ　　　　// ストライドの要素数 x floatのバイト数

						directAttribs.position(0);
						gl.glVertexAttribPointer(attr1Loc, 3, GL2.GL_FLOAT, false, 6*4, directAttribs);

						gl.glDrawArrays(GL2.GL_QUADS, 0, directAttribs.capacity()/(6*4));

					} finally {
						gl.glDisableVertexAttribArray(attr1Loc);
						gl.glDisableVertexAttribArray(attr2Loc);
					}
				}
			};

			int pushAttribs = GL2.GL_ENABLE_BIT | GL2.GL_COLOR_BUFFER_BIT;

			support.useShaderProgram(
					mblur2Accum2Program, uniforms, operation, pushAttribs,
					accumBuffer, buffers.toArray(new IVideoBuffer[buffers.size()]));

		} finally {
			if (attribs != null) attribs.release();
		}
	}


	private static int[][] frustumPlaneTable = new int[][] {
		// 最初の3つがクリップ面を構成する頂点のインデックス。
		// 残りひとつがそのクリップ面と向き合う面の頂点のひとつ。
		{0,2,6,1}, {1,3,7,0}, {0,1,2,6}, {6,7,4,0}, {0,1,6,2}, {2,3,4,0}
	};

	private boolean intersectsWithFrustum(Point3d[] vertices, double[] mvMatrix) {
		double[] prjMatrix = camera.getProjection3D();
		int w = boundsX.width;
		int h = boundsX.height;
		int[] viewport = new int[] { 0, 0, w, h };
		double[][] unprj = new double[8][3];

		GLU glu = context.getGLU();
		glu.gluUnProject(0, 0, 0, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[0], 0);
		glu.gluUnProject(0, 0, 1, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[1], 0);
		glu.gluUnProject(w, 0, 0, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[2], 0);
		glu.gluUnProject(w, 0, 1, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[3], 0);
		glu.gluUnProject(w, h, 0, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[4], 0);
		glu.gluUnProject(w, h, 1, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[5], 0);
		glu.gluUnProject(0, h, 0, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[6], 0);
		glu.gluUnProject(0, h, 1, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[7], 0);

		Vector4d[] frustumPlanes = new Vector4d[6];
		for (int i = 0; i < 6; ++i) {
			Point3d p0 = new Point3d(unprj[frustumPlaneTable[i][0]]);
			Point3d p1 = new Point3d(unprj[frustumPlaneTable[i][1]]);
			Point3d p2 = new Point3d(unprj[frustumPlaneTable[i][2]]);

			Vector3d v1 = new Vector3d();
			v1.sub(p1, p0);
			Vector3d v2 = new Vector3d();
			v2.sub(p2, p0);

			Vector3d n = new Vector3d();
			n.cross(v1, v2);
			n.normalize();

			double d = -n.dot(new Vector3d(p0));
			frustumPlanes[i] = new Vector4d(n.x, n.y, n.z, d);
		}

		List<Point3d> list = Util.newList(Arrays.asList(vertices));
		for (int i = 0; i < 6; ++i) {
			Vector4d plane = frustumPlanes[i];

			double[] opposite = unprj[frustumPlaneTable[i][3]];
			double signum = Math.signum(plane.dot(
					new Vector4d(opposite[0], opposite[1], opposite[2], 1)));

			List<Point3d>[] partitioned = partitionPolygon(plane, signum, list);
			if (partitioned == null) {
				// クリップ面のひとつに(ほぼ)張り付いている場合。
				continue;
			}
			if (partitioned[1] == null) {
				continue;
			} else if (partitioned[0] == null) {
				// 全ての頂点がクリップ面の外側にある場合。
				list.clear();
				break;
			} else {
				list = partitioned[0];
			}
		}
		return !list.isEmpty();
	}

	private static final int[][] frustumEdgeTable = {
		{ 0, 1 },		// 0
		{ 0, 2 },		// 1
		{ 0, 6 },		// 2
		{ 1, 3 },		// 3
		{ 1, 7 },		// 4
		{ 2, 3 },		// 5
		{ 2, 4 },		// 6
		{ 3, 5 },		// 7
		{ 4, 5 },		// 8
		{ 4, 6 },		// 9
		{ 5, 7 },		// 10
		{ 6, 7 }		// 11
	};

	private static final int[][] frustumClipSearchTable = {
		{ 1, 3, 5, 2, 4, 11 },		// 0
		{ 0, 3, 5, 2, 6, 9 },		// 1
		{ 1, 6, 9, 0, 4, 11 },		// 2
		{ 0, 1, 5, 4, 7, 10 },		// 3
		{ 0, 2, 11, 3, 7, 10 },		// 4
		{ 0, 1, 3, 6, 7, 8 },		// 5
		{ 1, 2, 9, 5, 7, 8 },		// 6
		{ 3, 4, 10, 5, 6, 8 },		// 7
		{ 5, 6, 7, 9, 10, 11 },		// 8
		{ 1, 2, 6, 8, 10, 11 },		// 9
		{ 3, 4, 7, 8, 9, 11 },		// 10
		{ 0, 2, 4, 8, 9, 10 }		// 11
	};

	private Point3d[] unProjectViewport(double[] mvMatrix) {
		double[] prjMatrix = camera.getProjection3D();
		int w = boundsX.width;
		int h = boundsX.height;
		int[] viewport = new int[] { 0, 0, w, h };
		double[][] unprj = new double[8][3];

		GLU glu = context.getGLU();
		glu.gluUnProject(0, 0, 0, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[0], 0);
		glu.gluUnProject(0, 0, 1, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[1], 0);
		glu.gluUnProject(w, 0, 0, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[2], 0);
		glu.gluUnProject(w, 0, 1, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[3], 0);
		glu.gluUnProject(w, h, 0, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[4], 0);
		glu.gluUnProject(w, h, 1, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[5], 0);
		glu.gluUnProject(0, h, 0, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[6], 0);
		glu.gluUnProject(0, h, 1, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[7], 0);

		Point3d[] frustumVertices = new Point3d[8];
		for (int i = 0; i < 8; ++i) {
			frustumVertices[i] = new Point3d(unprj[i]);
		}

		Point3d[][] frustumEdges = new Point3d[12][2];
		for (int i = 0; i < 12; ++i) {
			frustumEdges[i][0] = frustumVertices[frustumEdgeTable[i][0]];
			frustumEdges[i][1] = frustumVertices[frustumEdgeTable[i][1]];
		}

		List<Point3d> list = Util.newList();
		Vector3d v = new Vector3d();

		for (int i = 0; i < 12; ++i) {
			Point3d p0 = frustumEdges[i][0];
			Point3d p1 = frustumEdges[i][1];

			// FIXME p0.z==0 または p1.z==0 の場合も特別扱いしなくて良い？
			if ((p0.z <= 0 && p1.z >= 0) || (p0.z >= 0 && p1.z <= 0)) {
				v.sub(p1, p0);
				Point3d p = new Point3d(
						p0.x - v.x/v.z*p0.z,
						p0.y - v.y/v.z*p0.z, 0);

				list.add(p);

				FRUSTUM_CLIP_SEARCH:
				while (true) {
					for (int j = 0; j < 6; ++j) {
						int k = frustumClipSearchTable[i][j];
						p0 = frustumEdges[k][0];
						p1 = frustumEdges[k][1];

						// FIXME p0.z==0 または p1.z==0 の場合も特別扱いしなくて良い？
						if ((p0.z <= 0 && p1.z >= 0) || (p0.z >= 0 && p1.z <= 0)) {
							v.sub(p1, p0);
							p = new Point3d(
									p0.x - v.x/v.z*p0.z,
									p0.y - v.y/v.z*p0.z, 0);
	
							if (!list.contains(p)) {
								list.add(p);
								i = k;
								continue FRUSTUM_CLIP_SEARCH;
							}
						}
					}
					break;
				}
				break;
			}
		}

		return (list.size() >= 3) ? list.toArray(new Point3d[list.size()]) : null;
	}

	private List<Polygon> partitionPolygons(List<Polygon> polygons) {
		if (polygons.isEmpty()) {
			return polygons;
		}

		Iterator<Polygon> it = polygons.iterator();
		Polygon base = it.next();
		List<Polygon> near = Util.newList();
		List<Polygon> far = Util.newList();

		while (it.hasNext()) {
			Polygon target = it.next();
			partitionPolygon(base, target, near, far, null);
		}

		near = partitionPolygons(near);
		far = partitionPolygons(far);

		far.add(base);
		far.addAll(near);
		return far;
	}

	private void partitionPolygon(Polygon base, Polygon target, List<Polygon> near, List<Polygon> far, Vec3d lightPos) {
		double signum;
		if (lightPos != null) {
			signum = Math.signum(base.entry.plane.x * lightPos.x
								+ base.entry.plane.y * lightPos.y
								+ base.entry.plane.z * lightPos.z
								+ base.entry.plane.w);
		} else {
			signum = Math.signum(base.entry.plane.w);
		}

		if (!partitionPolygon(base.entry.plane, signum, target, near, far)) {
			Matrix4d mat = new Matrix4d(base.entry.mvMatrix);
			Point3d p0 = new Point3d(0, 0, -1000000);
			mat.transform(p0);
			double signum0 = Math.signum(base.entry.plane.dot(new Vector4d(p0.x, p0.y, p0.z, 1)));

			if ((signum == signum0) ^ (base.entry.indexInGroup < target.entry.indexInGroup)) {
				far.add(target);
			} else {
				near.add(target);
			}
		}
	}

	private boolean partitionPolygon(Vector4d plane, double signum, Polygon target, List<Polygon> near, List<Polygon> far) {
		List<Point3d>[] vertices = partitionPolygon(plane, signum, target.vertices);
		if (vertices == null) {
			return false;
		}
		if (vertices[1] == null) {
			near.add(target);
		} else if (vertices[0] == null) {
			far.add(target);
		} else {
			near.add(new Polygon(target.entry, vertices[0]));
			far.add(new Polygon(target.entry, vertices[1]));
		}
		return true;
	}

	private List<Point3d>[] partitionPolygon(Vector4d plane, double signum, List<Point3d> vertices) {
		double d = 0;
		for (Point3d pt : vertices) {
			double dd = plane.dot(new Vector4d(pt.x, pt.y, pt.z, 1.0));
			if (Math.abs(dd) > Math.abs(d)) {
				d = dd;
			}
		}
		if (Math.abs(d) < 1e-3) {	// TODO 1e-3が妥当な値かどうかわからない。1e-4ではダメなケースがあることは確認した。
			return null;
		}
//		if (Math.abs(d) < 0.01) {
//			System.out.printf("%f: %f%n", context.getTime().toSecond(), d);
//		}

		vertices = Util.newList(vertices);
		int vertIndex1 = -1;
		int vertIndex2 = -1;

		for (int i1 = 0; i1 < vertices.size(); ++i1) {
			int i2 = (i1+1) % vertices.size();
			Point3d vert1 = vertices.get(i1);
			Point3d vert2 = vertices.get(i2);

			double d1 = plane.dot(new Vector4d(vert1.x, vert1.y, vert1.z, 1.0));
			double d2 = plane.dot(new Vector4d(vert2.x, vert2.y, vert2.z, 1.0));

			if (Math.signum(d1) != Math.signum(d2)) {
				if (/*d1 == 0*/ Math.abs(d1) < 1e-3) {
					if (vertIndex1 == -1) {
						vertIndex1 = i1;
					} else {
						vertIndex2 = i1;
						break;
					}
				} else if (/*d2 == 0*/ Math.abs(d2) < 1e-3) {
					if (vertIndex1 == -1) {
						vertIndex1 = ++i1;
					} else {
						vertIndex2 = i1+1;
						break;
					}
				} else {
					Vector3d v = new Vector3d();
					v.sub(vert2, vert1);

					double t = -d1 / plane.dot(new Vector4d(v));
					v.scale(t);

					Point3d vertNew = new Point3d();
					vertNew.add(vert1, v);
					vertices.add(i1+1, vertNew);

					if (vertIndex1 == -1) {
						vertIndex1 = ++i1;
					} else {
						vertIndex2 = i1+1;
						break;
					}
				}
			}
		}

		if (vertIndex2 == -1 || vertIndex2-vertIndex1 <= 1 || vertIndex2-vertIndex1 >= vertices.size()-1) {
			d = 0;
			for (Point3d pt : vertices) {
				double dd = plane.dot(new Vector4d(pt.x, pt.y, pt.z, 1.0));
				if (Math.abs(dd) > Math.abs(d)) {
					d = dd;
				}
			}

			@SuppressWarnings("unchecked")
			List<Point3d>[] result = (Math.signum(d) == signum)
								   ? new List[] { vertices, null }
								   : new List[] { null, vertices };
			return result;
		}

		vertices.add(vertices.get(0));
		List<Point3d> vertices1 = Util.newList(vertices.subList(0, vertIndex1+1));
		List<Point3d> vertices2 = Util.newList(vertices.subList(vertIndex1, vertIndex2+1));
		vertices1.addAll(vertices.subList(vertIndex2, vertices.size()-1));

		d = 0;
		for (Point3d pt : vertices1) {
			double dd = plane.dot(new Vector4d(pt.x, pt.y, pt.z, 1.0));
			if (Math.abs(dd) > Math.abs(d)) {
				d = dd;
			}
		}

		@SuppressWarnings("unchecked")
		List<Point3d>[] result = (Math.signum(d) == signum)
							   ? new List[] { vertices1, vertices2 }
							   : new List[] { vertices2, vertices1 };
		return result;
	}

	private IVideoBuffer magnifyForSuperSampling(IVideoBuffer input) {
		IVideoBuffer output = null;
		try {
			Runnable operation = new Runnable() {
				public void run() {
					GL2 gl = context.getGL().getGL2();
					gl.glColor4f(1, 1, 1, 1);
					support.ortho2D(boundsX);
					support.quad2D(boundsX, new double[][] { {0, 0}, {1, 0}, {1, 1}, {0, 1} });
				}
			};

			// inputのフィルタはNEARESTになっているはず。
			//input.setTextureFilter(TextureFilter.NEAREST);

			output = support.createVideoBuffer(boundsX);
			support.useFramebuffer(operation, GL2.GL_CURRENT_BIT, output, input);

			IVideoBuffer result = output;
			output = null;
			return result;

		} finally {
			if (output != null) output.dispose();
		}
	}

	private void downSample(IVideoBuffer input, IVideoBuffer output) {
		VideoBounds bin = input.getBounds();
		final VideoBounds bout = output.getBounds();
		if (bin.width != bout.width*ssScale || bin.height != bout.height*ssScale) {
			throw new IllegalArgumentException();
		}

		float[] kernel = new float[ssScale*ssScale];
		float[] offset = new float[kernel.length*2];

		for (int j = 0; j < ssScale; ++j) {
			for (int i = 0; i < ssScale; ++i) {
				int k = j*ssScale+i;
				kernel[k] = 1f/kernel.length;
				offset[k*2  ] = (float)(i-ssScale*0.5+0.5)/bin.width;
				offset[k*2+1] = (float)(j-ssScale*0.5+0.5)/bin.height;
			}
		}

		Runnable operation = new Runnable() {
			public void run() {
				support.ortho2D(bout);
				support.quad2D(bout, new double[][] { {0, 0}, {1, 0}, {1, 1}, {0, 1} });
			}
		};

		convolution.convolve(input, output, kernel, offset, operation, 0);
	}


	private static class ComposeGroup {

		private final boolean threeD;

		private final List<GeneralLayerRenderer> renderers = Util.newList();

		private ComposeGroup(boolean threeD) {
			this.threeD = threeD;
		}

		private void add(GeneralLayerRenderer r) {
			renderers.add(r);
		}
	}

	private static class Entry3D {

		private final int indexInGroup;

		private final MediaLayer layer;

		private final CastsShadows castsShadows;

		private final IVideoBuffer source;

		private final IVideoBuffer sourceCR;

		private IVideoBuffer diffusion;

		private IVideoBuffer lighted;

		private final IVideoBuffer matte;

		private final double[] mvMatrix;

		private final Point3d[] vertices;
		
		private final Point2d[] texCoords;

		private final double opacity;

		private final Vector4d plane;

		private Entry3D(int indexInGroup, MediaLayer layer, CastsShadows castsShadows,
				IVideoBuffer source, IVideoBuffer sourceCR,
				IVideoBuffer diffusion, IVideoBuffer matte, double[] mvMatrix,
				Point3d[] vertices, Point2d[] texCoords, double opacity) {

			this.indexInGroup = indexInGroup;
			this.layer = layer;
			this.castsShadows = castsShadows;
			this.source = source;
			this.sourceCR = sourceCR;
			this.diffusion = diffusion;
			this.matte = matte;
			this.mvMatrix = mvMatrix;
			this.vertices = vertices;
			this.texCoords = texCoords;
			this.opacity = opacity;

			if (vertices != null) {
				Vector3d v1 = new Vector3d();
				v1.sub(vertices[1], vertices[0]);

				Vector3d v2 = new Vector3d();
				v2.sub(vertices[vertices.length-1], vertices[0]);

				Vector3d n = new Vector3d();
				n.cross(v1, v2);
				n.normalize();
				double d = -n.dot(new Vector3d(vertices[0]));
				plane = new Vector4d(n.x, n.y, n.z, d);
			} else {
				plane = null;
			}
		}

		private void dispose() {
			if (source != null) source.dispose();
			if (sourceCR != null) sourceCR.dispose();
			if (diffusion != null) diffusion.dispose();
			if (lighted != null) lighted.dispose();
		}

		private void disposeLighted() {
			if (lighted != null && lighted != sourceCR) {
				lighted.dispose();
			}
			lighted = null;
		}

		private static final Comparator<Entry3D> partitioningOrderComparator = new Comparator<Entry3D>() {

			public int compare(Entry3D o1, Entry3D o2) {
				// BlendModeがNORMALでないレイヤーは、分割後のポリゴンひとつずつの処理となるので、できるだけ分割されない方がよい。
				// 分割前のリスト内で先頭に移動しておけば、分割されずに(または、少ない分割数で)済む。

				BlendMode blendMode1 = o1.layer.getBlendMode();
				BlendMode blendMode2 = o2.layer.getBlendMode();
				if (blendMode1 == BlendMode.NORMAL && blendMode2 != BlendMode.NORMAL) {
					return 1;
				}
				if (blendMode1 != BlendMode.NORMAL && blendMode2 == BlendMode.NORMAL) {
					return -1;
				}

				// (レイヤーサイズではなく)3D空間内に配置した状態での差し渡しの長さが長いレイヤーは
				// 他のレイヤー平面を跨ぐ可能性が高いので、分割前のリスト内で前方に移動しておけば
				// 少ない分割数で済むはず。

				double span1 = spanSquared(o1);
				double span2 = spanSquared(o2);
				return (int)Math.signum(span2 - span1);
			}

			private double spanSquared(Entry3D entry) {
				double spanSquared = 0;
				Vector3d vec = new Vector3d();
				for (int i = 0, n = entry.vertices.length; i < n; ++i) {
					for (int j = i+1; j < n; ++j) {
						vec.sub(entry.vertices[i], entry.vertices[j]);
						spanSquared = Math.max(spanSquared, vec.lengthSquared());
					}
				}
				return spanSquared;
			}
		};
	}

	private static class Caster {

		private final Entry3D entry;

		private final List<Point3d> vertices;

		private final Vector3d partitionLine;

		private final boolean partitionExpand;

		private Caster(Entry3D entry, List<Point3d> vertices,
				Vector3d partitionLine, boolean partitionExpand) {
			this.entry = entry;
			this.vertices = vertices;
			this.partitionLine = partitionLine;
			this.partitionExpand = partitionExpand;
		}
	}

	private static class Polygon {

		private final Entry3D entry;

		private final List<Point3d> vertices;

		private Polygon(Entry3D entry) {
			this(entry, Arrays.asList(entry.vertices));
		}

		private Polygon(Entry3D entry, List<Point3d> vertices) {
			this.entry = entry;
			this.vertices = vertices;
		}

		private List<Triangle> toTriangles(int sourceIndex, int matteIndex) {
			List<Triangle> triangles = Util.newList();
			Point3d vert0 = vertices.get(0);
			Point3d vert1 = vertices.get(1);
			Point3d vert2;

			for (int i = 2, n = vertices.size(); i < n; ++i, vert1 = vert2) {
				vert2 = vertices.get(i);
				triangles.add(new Triangle(entry, sourceIndex, matteIndex, vert0, vert1, vert2));
			}

			return triangles;
		}
	}

	private static class Triangle {

		private final Entry3D entry;

		private final int sourceIndex;

		private final int matteIndex;

		private final Point3d[] vertices;

		private Triangle(Entry3D entry, int sourceIndex, int matteIndex,
				Point3d vert0, Point3d vert1, Point3d vert2) {
			this.entry = entry;
			this.sourceIndex = sourceIndex;
			this.matteIndex = matteIndex;
			vertices = new Point3d[] { vert0, vert1, vert2 };
		}
	}

	private static class Entry2D {

		private final MediaLayer layer;

		private final IVideoBuffer source;

		private final IVideoBuffer matte;

		private final Point3d[] vertices;
		
		private final Point2d[] texCoords;

		private final double opacity;

		private Entry2D(MediaLayer layer, IVideoBuffer source, IVideoBuffer matte,
				Point3d[] vertices, Point2d[] texCoords, double opacity) {

			this.layer = layer;
			this.source = source;
			this.matte = matte;
			this.vertices = vertices;
			this.texCoords = texCoords;
			this.opacity = opacity;
		}
	}


	private void createReadTextureFunction() {
		String name = VideoLayerComposer.class.getName() + ".read_texture";
		if (!shaders.isShaderRegistered(name)) {
			List<String> list = Util.newList();

			for (int i = 0; i < maxTexImageUnits; ++i) {
				list.add(String.format(	"uniform sampler2D texture%d;", i));
			}

			list.addAll(Arrays.asList(new String[] {
										"vec4 readTexture(int i, vec2 coord) {",
										"	return"
			}));

			for (int i = 0; i < maxTexImageUnits; ++i) {
				list.add(String.format(	"		(i==%1$d) ? texture2D(texture%1$d, coord) :", i));
			}

			list.addAll(Arrays.asList(new String[] {
										"		vec4(0.0);",
										"}"
			}));

			list.addAll(Arrays.asList(new String[] {
										"vec4 mixLod(sampler2D sampler, vec2 coord, float lod) {",
										"	float lod1 = floor(lod);",
										"	return mix(texture2DLod(sampler, coord, lod1),",
										"			texture2DLod(sampler, coord, lod1+1.0), lod-lod1);",
										"}"
			}));

			list.addAll(Arrays.asList(new String[] {
										"vec4 readTexture(int i, vec2 coord, float lod) {",
										"	return"
			}));

			for (int i = 0; i < maxTexImageUnits; ++i) {
				list.add(String.format(	"		(i==%1$d) ? mixLod(texture%1$d, coord, lod) :", i));
			}

			list.addAll(Arrays.asList(new String[] {
										"		vec4(0.0);",
										"}"
			}));

			shaders.registerShader(name, ShaderType.FRAGMENT_SHADER, list.toArray(new String[list.size()]));
		}
	}

	@ShaderSource(type=ShaderType.VERTEX_SHADER, program=false)
	public static final String[] normal_blend_3d_vert = {
		"attribute vec3 attr1;",
		"attribute vec3 attr2;",
		"",
		"varying float opacity;",
		"varying float source;",
		"varying float matte;",
		"",
		"void main(void)",
		"{",
//		"	gl_Position = gl_ModelViewProjectionMatrix * vec4(attr1, 1.0);",	// モデルビュー行列は単位行列なので。
		"	gl_Position = gl_ProjectionMatrix * vec4(attr1, 1.0);",
		"	opacity = attr2.x;",
		"	source = attr2.y;",
		"	matte = attr2.z;",
		"}"
	};

	@ShaderSource(attach={ "normal_blend_3d_vert", "read_texture" })
	public static final String[] NORMAL_BLEND_3D = {
		"varying float opacity;",
		"varying float source;",
		"varying float matte;",
		"",
//		"uniform vec2 texOffset[gl_MaxTextureImageUnits];",
//		"uniform vec2 texSize[gl_MaxTextureImageUnits];",
		"uniform vec4 texOffsetAndSize[gl_MaxTextureImageUnits];",
		"uniform vec2 dstSize;",
		"",
		"vec4 readTexture(int i, vec2 coord);",
		"",
		"void main(void) {",
		"	int srcIndex = int(source);",
		"	vec4 offAndSize = texOffsetAndSize[srcIndex];",
		"	vec2 coord = (gl_FragCoord.xy + offAndSize.xy) / offAndSize.zw;",
		"	vec4 color = readTexture(srcIndex, coord) * opacity;",
		"	if (matte > 0.0) {",
		"		color *= readTexture(int(matte), gl_FragCoord.xy / dstSize).a;",
		"	}",
		"	gl_FragColor = color;",
		"}"
	};

	@ShaderSource(type=ShaderType.VERTEX_SHADER, program=false)
	public static final String[] normal_blend_2d_vert = {
		"attribute vec3 attr1;",
		"attribute vec4 attr2;",
		"",
		"varying float opacity;",
		"varying float source;",
		"varying float matte;",
		"",
		"void main(void)",
		"{",
//		"	gl_Position = gl_ModelViewProjectionMatrix * vec4(attr1.xy, 0.0, 1.0);",
		"	gl_Position = gl_ProjectionMatrix * vec4(attr1.xy, 0.0, 1.0);",
		"	gl_TexCoord[0] = vec4(attr2.xy, 0.0, 0.0);",
		"	opacity = attr1.z;",
		"	source = attr2.z;",
		"	matte = attr2.w;",
		"}"
	};

	@ShaderSource(attach={ "normal_blend_2d_vert", "read_texture" })
	public static final String[] NORMAL_BLEND_2D = {
		"varying float opacity;",
		"varying float source;",
		"varying float matte;",
		"",
		"uniform vec2 dstSize;",
		"",
		"vec4 readTexture(int i, vec2 coord);",
		"",
		"void main(void) {",
		"	vec4 color = readTexture(int(source), gl_TexCoord[0].st) * opacity;",
		"	if (matte > 0.0) {",
		"		color *= readTexture(int(matte), gl_FragCoord.xy / dstSize).a;",
		"	}",
		"	gl_FragColor = color;",
		"}"
	};

	private String[] createCustomBlend2DSource(String name, boolean dissolve, boolean matte) {
		return new String[] {
						"uniform sampler2D texDst;",
						"uniform sampler2D texSrc;",
				matte ? "uniform sampler2D texMatte;" : "",
						"uniform vec2 dstSize;",
						"uniform float opacity;",
			 dissolve ?	"uniform float dissolveSeed;" : "",
						"",
		  String.format("vec4 blend_%s(vec4 pDst, vec4 pSrc, float opacity%s);", name, dissolve ? ", float dissolveSeed" : ""),
						"",
						"void main(void)",
						"{",
						"	vec2 dstCoord = gl_FragCoord.xy / dstSize;",
						"	vec4 dst = texture2D(texDst, dstCoord);",
						"	vec4 src = texture2D(texSrc, gl_TexCoord[0].st);",
		  String.format("	gl_FragColor = blend_%s(dst, src, opacity%s%s);", name,
												matte ? "*texture2D(texMatte, dstCoord).a" : "",
												dissolve ? ", dissolveSeed" : ""),
						"}"
		};
	}

	private String[] createCustomBlend3DSource(String name, boolean dissolve, boolean matte) {
		return new String[] {
						"uniform sampler2D texDst;",
						"uniform sampler2D texSrc;",
				matte ? "uniform sampler2D texMatte;" : "",
						"uniform vec2 dstSize;",
					 	"uniform vec2 srcOffset;",
					 	"uniform vec2 srcSize;",
						"uniform float opacity;",
			 dissolve ?	"uniform float dissolveSeed;" : "",
						"",
		  String.format("vec4 blend_%s(vec4 pDst, vec4 pSrc, float opacity%s);", name, dissolve ? ", float dissolveSeed" : ""),
						"",
						"void main(void)",
						"{",
						"	vec2 dstCoord = gl_FragCoord.xy / dstSize;",
						"	vec4 dst = texture2D(texDst, dstCoord);",
						"	vec4 src = texture2D(texSrc, (gl_FragCoord.xy + srcOffset) / srcSize);",
		  String.format("	gl_FragColor = blend_%s(dst, src, opacity%s%s);", name,
												matte ? "*texture2D(texMatte, dstCoord).a" : "",
												dissolve ? ", dissolveSeed" : ""),
						"}"
		};
	}

	private String[] createCustomBlendSource(BlendMode blendMode, boolean matte, boolean threeD) {
		boolean dissolve;
		switch (blendMode) {
			case DANCING_DISSOLVE:
				blendMode = BlendMode.DISSOLVE;
				// fall through
			case DISSOLVE:
				dissolve = true;
				break;
			default:
				dissolve = false;
				break;
		}

		String name = blendMode.name().toLowerCase();

		if (threeD) {
			return createCustomBlend3DSource(name, dissolve, matte);
		} else {
			return createCustomBlend2DSource(name, dissolve, matte);
		}
	}

	private IShaderProgram getCustomBlendProgram(BlendMode blendMode, boolean matte, boolean threeD) {
		String programName = VideoLayerComposer.class.getName() + "." + blendMode.name()
										+ (matte ? "_MATTE" : "") + (threeD ? "_3D" : "_2D");
		IShaderProgram program = shaders.getProgram(programName);
		if (program == null) {
			String[] attach = { "ch.kuramo.javie.core.shaders.BlendModeShaders.blend_functions" };
			String[] source = createCustomBlendSource(blendMode, matte, threeD);
			program = shaders.registerProgram(programName, ShaderType.FRAGMENT_SHADER, attach, source);
		}
		return program;
	}

	private IShaderProgram getCustomBlend3DProgram(BlendMode blendMode, boolean matte) {
		return getCustomBlendProgram(blendMode, matte, true);
	}

	private IShaderProgram getCustomBlend2DProgram(BlendMode blendMode, boolean matte) {
		return getCustomBlendProgram(blendMode, matte, false);
	}

	@ShaderSource(type=ShaderType.VERTEX_SHADER, program=false)
	public static final String[] mblur2_accum2_vert = {
		"attribute vec3 attr1;",
		"attribute vec3 attr2;",
		"",
		"varying float weight;",
		"varying float source;",
		"",
		"void main(void)",
		"{",
//		"	gl_Position = gl_ModelViewProjectionMatrix * vec4(attr1.xy, 0.0, 1.0);",
		"	gl_Position = gl_ProjectionMatrix * vec4(attr1.xy, 0.0, 1.0);",
		"	gl_TexCoord[0] = vec4(attr2.xy, 0.0, 0.0);",
		"	weight = attr1.z;",
		"	source = attr2.z;",
		"}"
	};

	@ShaderSource(attach={ "mblur2_accum2_vert", "read_texture" })
	public static final String[] MBLUR2_ACCUM2 = {
		"varying float weight;",
		"varying float source;",
		"",
		"vec4 readTexture(int i, vec2 coord);",
		"",
		"void main(void) {",
		"	gl_FragColor = readTexture(int(source), gl_TexCoord[0].st) * weight;",
		"}"
	};

	@ShaderSource(attach="read_texture")
	public static final String[] MATTE_MULTIPLY = {
		"uniform float trackMattes[gl_MaxTextureImageUnits];",
		"uniform int numMattes;",
		"uniform vec2 size;",
		"",
		"const vec3 lumaVec = vec3(0.299, 0.587, 0.114);",
		"",
		"vec4 readTexture(int i, vec2 coord);",
		"",
		"void main(void) {",
		"	vec2 coord = gl_FragCoord.xy / size;",
		"	float a = 1.0;",
		"	for (int i = 0; i < numMattes; ++i) {",
		"		vec4 color = readTexture(i, coord);",
		"",
		"		float t = trackMattes[i];",
		"		if (t > 4.0) {",
		"			a *= 1.0 - dot(color.rgb, lumaVec);",
		"		} else if (t > 3.0) {",
		"			a *= dot(color.rgb, lumaVec);",
		"		} else if (t > 2.0) {",
		"			a *= 1.0 - color.a;",
		"		} else if (t > 1.0) {",
		"			a *= color.a;",
		"		} else {",
		"			a *= 0.0;",
		"		}",
		"",
		"		if (a == 0.0) {",
		"			break;",
		"		}",
		"	}",
		"	gl_FragColor = vec4(a);",
		"}"
	};

	@ShaderSource
	public static final String[] DIFFUSION_CONVOLUTION = {
		"uniform sampler2D texture;",
		"uniform int ksize;",
		"uniform float kernel[17];",
		"uniform vec2 offset[17];",
		"uniform float lod;",
		"",
		"void main(void)",
		"{",
		"	vec2 texCoord = gl_TexCoord[0].st;",
		"	vec4 sum = vec4(0.0);",
		"	for (int i = 0; i < ksize; ++i) {",
		"		sum += kernel[i] * texture2DLod(texture, texCoord + offset[i], lod);",
		"	}",
		"	gl_FragColor = sum;",
		"}"
	};

	@ShaderSource(type=ShaderType.VERTEX_SHADER, program=false)
	public static final String[] shadow_vert = {
		"attribute vec4 attr;",
		"varying vec4 coord;",
		"varying float caster;",
		"",
		"void main(void)",
		"{",
//		"	coord = gl_ModelViewMatrix * vec4(attr.xyz, 1.0);",		// モデルビュー行列は単位行列なのでこの計算は不要。
		"	coord = vec4(attr.xyz, 1.0);",
		"	caster = attr.w;",
//		"	gl_Position = gl_ModelViewProjectionMatrix * coord;",	// モデルビュー行列は単位行列なので。
		"	gl_Position = gl_ProjectionMatrix * coord;",
		"}",
	};

	private static final String[] createShadowOfParallelLightSource(boolean partitioned) {
		return new String[] {
					"varying vec4 coord;",
					"varying float caster;",
					"",
					"uniform vec4 casterPlanes[gl_MaxTextureImageUnits];",
					"uniform mat4 casterMvTxInvs[gl_MaxTextureImageUnits];",
					"uniform float casterOpacs[gl_MaxTextureImageUnits];",
					"uniform float casterTrans[gl_MaxTextureImageUnits];",
					"uniform vec4 lightDir;",
					"uniform vec2 viewport;",
	  partitioned ? "uniform vec3 partitionLines[gl_MaxTextureImageUnits];" : "",
	  partitioned ? "uniform float partitionExpands[gl_MaxTextureImageUnits];" : "",
					"",
					"vec4 readTexture(int i, vec2 coord);",
					"",
					"void main(void)",
					"{",
					"	int i = int(caster);",
					"	vec4 casterPlane = casterPlanes[i];",
					"	float t = -dot(casterPlane, coord) / dot(casterPlane, lightDir);",
	  partitioned ? "	float t2 = abs(dot(partitionLines[i], vec3(gl_FragCoord.xy, 1.0)))*sign(-t);" : "",
	  partitioned ?	"	if (t2 > -1.0) {" : "",

					"		vec4 p = coord + t*lightDir;",
					"		vec4 casterColor = readTexture(i, (casterMvTxInvs[i]*p).xy)*casterOpacs[i];",
					"",
	 !partitioned ? "		gl_FragColor = vec4(1.0-casterColor.a) + casterColor*casterTrans[i];"
				  : "		gl_FragColor = mix(vec4(1.0), vec4(1.0-casterColor.a) + casterColor*casterTrans[i],",
	  partitioned ? "							clamp(t2+partitionExpands[i], 0.0, 1.0));" : "",
					"",
	  partitioned ? "	} else {" : "",
	  partitioned ? "		discard;" : "",
	  partitioned ? "	}" : "",
					"}"
		};
	}

	private IShaderProgram getShadowOfParallelLightProgram(boolean partitioned) {
		String programName = VideoLayerComposer.class.getName() + ".SHADOW_OF_PARALLEL_LIGHT"
															+ (partitioned ? "_PARTITIONED" : "");
		IShaderProgram program = shaders.getProgram(programName);
		if (program == null) {
			String[] attach = {
					VideoLayerComposer.class.getName() + ".shadow_vert",
					VideoLayerComposer.class.getName() + ".read_texture"
			};
			String[] source = createShadowOfParallelLightSource(partitioned);
			program = shaders.registerProgram(programName, ShaderType.FRAGMENT_SHADER, attach, source);
		}
		return program;
	}

	private static final String[] createShadowOfPointLightSource(boolean diffusion, boolean partitioned) {
		return new String[] {
					"varying vec4 coord;",
					"varying float caster;",
					"",
					"uniform vec4 casterPlanes[gl_MaxTextureImageUnits];",
					"uniform mat4 casterMvTxInvs[gl_MaxTextureImageUnits];",
					"uniform float casterOpacs[gl_MaxTextureImageUnits];",
					"uniform float casterTrans[gl_MaxTextureImageUnits];",
					"uniform vec4 lightPos;",
					"uniform vec2 viewport;",
		diffusion ? "uniform float diffusion;" : "",
	  partitioned ? "uniform vec3 partitionLines[gl_MaxTextureImageUnits];" : "",
	  partitioned ? "uniform float partitionExpands[gl_MaxTextureImageUnits];" : "",
					"",
		diffusion ? "vec4 readTexture(int i, vec2 coord, float lod);"
				  : "vec4 readTexture(int i, vec2 coord);",
					"",
					"void main(void)",
					"{",
					"	int i = int(caster);",
					"	vec4 casterPlane = casterPlanes[i];",
					"	vec4 v = lightPos - coord;",
					"	float t = -dot(casterPlane, coord) / dot(casterPlane, v);",
	  partitioned ? "	float t2 = abs(dot(partitionLines[i], vec3(gl_FragCoord.xy, 1.0)))*sign(t);" : "",
	  partitioned ?	"	if (t2 > -1.0) {" : "",
					"",
					"		vec4 p = coord + v*t;",

		diffusion ?	"		float blur = diffusion*t/(1.0-t);" : "",
		diffusion ?	"		float lod = (blur >= 32.0) ? log2(0.25*blur) - 1.0" : "",
		diffusion ?	"				  : (blur >= 2.0) ? 0.5*(log2(blur) - 1.0) : 0.0;" : "",
		diffusion ?	"		vec4 casterColor = readTexture(i, (casterMvTxInvs[i]*p).xy, lod)*casterOpacs[i];"
				  : "		vec4 casterColor = readTexture(i, (casterMvTxInvs[i]*p).xy)*casterOpacs[i];",

	 !partitioned ? "		gl_FragColor = vec4(1.0-casterColor.a) + casterColor*casterTrans[i];"
				  : "		gl_FragColor = mix(vec4(1.0), vec4(1.0-casterColor.a) + casterColor*casterTrans[i],",
	  partitioned ? "							clamp(t2+partitionExpands[i], 0.0, 1.0));" : "",
					"",
	  partitioned ? "	} else {" : "",
	  partitioned ? "		discard;" : "",
	  partitioned ? "	}" : "",
					"}"
		};
	}

	private IShaderProgram getShadowOfPointLightProgram(boolean diffusion, boolean partitioned) {
		String programName = VideoLayerComposer.class.getName() + ".SHADOW_OF_POINT_LIGHT"
										+ (diffusion ? "_WITH_DIFFUSION" : "") + (partitioned ? "_PARTITIONED" : "");
		IShaderProgram program = shaders.getProgram(programName);
		if (program == null) {
			String[] attach = {
					VideoLayerComposer.class.getName() + ".shadow_vert",
					VideoLayerComposer.class.getName() + ".read_texture"
			};
			String[] source = createShadowOfPointLightSource(diffusion, partitioned);
			program = shaders.registerProgram(programName, ShaderType.FRAGMENT_SHADER, attach, source);
		}
		return program;
	}

	@ShaderSource(type=ShaderType.VERTEX_SHADER, program=false)
	public static final String[] light_vert = {
		"varying vec3 coord;",
		"",
		"void main(void)",
		"{",
//		"	coord = (gl_ModelViewMatrix * gl_Vertex).xyz;",
		"	coord = gl_Vertex.xyz;",
//		"	gl_Position = gl_ModelViewProjectionMatrix * gl_Vertex;",
		"	gl_Position = gl_ProjectionMatrix * gl_Vertex;",
//		"	gl_TexCoord[0] = gl_TextureMatrix[0] * gl_MultiTexCoord0;",
		"	gl_TexCoord[0] = gl_MultiTexCoord0;",
		"}",
	};

	private String[] createParallelOrPointLightSource(LightType type, boolean shadow, boolean back, boolean cr) {
		boolean parallel = (type == LightType.PARALLEL);
		boolean spot = (type == LightType.SPOT);

		return new String[] {
						"varying vec3 coord;",
						"",
						"uniform sampler2D source;",
			   shadow ? "uniform sampler2D shadow;" : "",
			   shadow ? "uniform vec2 shadowSize;" : "",
				   cr ? "uniform vec2 sourceOffset;" : "",
				   cr ? "uniform vec2 sourceSize;" : "",
						"",
						"uniform vec3 normal;",
						"uniform vec4 material;",	// x=diffuse, y=specular, z=shininess, w=metal
				 back ? "uniform float lightTransmission;" : "", 
						"",
			 parallel ? "uniform vec3 lightVec;"
					  : "uniform vec3 lightPos;",
						"uniform float lightIntensity;",
			!parallel ? "uniform vec3 lightAttenuation;" : "",
						"uniform vec3 lightColor;",
				 spot ? "uniform vec3 spotVec;" : "",
				 spot ? "uniform float spotCutoff;" : "",
				 spot ? "uniform float spotSlope;" : "",
			   shadow ? "uniform float shadowDarkness;" : "",
						"",
						"void main(void)",
						"{",
				   cr ?	"	vec4 srcColor = texture2D(source, (gl_FragCoord.xy + sourceOffset)/sourceSize);"
					  : "	vec4 srcColor = texture2D(source, gl_TexCoord[0].st);",
			   shadow ? "	vec4 sdwColor = vec4(1.0)-(vec4(1.0)-texture2D(shadow, gl_FragCoord.xy/shadowSize))*shadowDarkness;" : "",
						"",
			!parallel ? "	vec3 lightVec = lightPos - coord;" : "",
			!parallel ? "	float lightDis = length(lightVec);" : "",
			!parallel ? "	lightVec /= lightDis;" : "",
						"",
			!parallel ? "	float attenuation = 1.0 / dot(lightAttenuation, vec3(1.0, lightDis, lightDis*lightDis));" : "",
						"",
				 spot ?	"	float spot = smoothstep(0.0, 1.0, (dot(lightVec, spotVec)-spotCutoff)*spotSlope);" : "",
						"",
						"	float diffuse = abs(dot(lightVec, normal));",
						"",
						"	vec3 viewVec = normalize(-coord);",
				!back ? "	vec3 halfVec = normalize(lightVec + viewVec);" : "",
				!back ? "	float specular = pow(abs(dot(normal, halfVec)), material.z);"
					  : "	float specular = pow(max(-dot(lightVec, viewVec), 0.0), material.z);",
						"",
						"	gl_FragColor = vec4(lightIntensity*lightColor",
			!parallel ? "					*attenuation" : "",
				 spot ? "					*spot" : "",
			   shadow ? "					*sdwColor.rgb" : "",
			     back ? "					*lightTransmission" : "",
						"					*(material.x*diffuse*srcColor.rgb",
						"					+ material.y*specular*mix(vec3(srcColor.a), srcColor.rgb, material.w)), srcColor.a);",
						"",
						"}",
		};
	}

	private IShaderProgram getParallelOrPointLightProgram(LightType type, boolean shadow, boolean back, boolean cr) {
		if (type == LightType.AMBIENT) {
			throw new IllegalArgumentException();
		}

		String programName = VideoLayerComposer.class.getName() + "." + type.name() + "_LIGHT"
										+ (shadow ? "_WITH_SHADOW" : "") + (back ? "_BACK" : "") + (cr ? "_CR" : "");
		IShaderProgram program = shaders.getProgram(programName);
		if (program == null) {
			String[] attach = { VideoLayerComposer.class.getName() + ".light_vert" };
			String[] source = createParallelOrPointLightSource(type, shadow, back, cr);
			program = shaders.registerProgram(programName, ShaderType.FRAGMENT_SHADER, attach, source);
		}
		return program;
	}

	private String[] createAmbientLightsSource(boolean cr) {
		return new String[] {
						"uniform sampler2D source;",
				   cr ? "uniform vec2 sourceOffset;" : "",
				   cr ? "uniform vec2 sourceSize;" : "",
						"",
						"uniform vec3 ambient;",
						"",
						"void main(void)",
						"{",
				   cr ?	"	vec4 sourceColor = texture2D(source, (gl_FragCoord.xy + sourceOffset)/sourceSize);"
					  : "	vec4 sourceColor = texture2D(source, gl_TexCoord[0].st);",
						"",
						"	gl_FragColor = vec4(ambient*sourceColor.rgb, sourceColor.a);",
						"}",
		};
	}

	private IShaderProgram getAmbientLightsProgram(boolean cr) {
		String programName = VideoLayerComposer.class.getName() + ".AMBIENT_LIGHTS" + (cr ? "_CR" : "");
		IShaderProgram program = shaders.getProgram(programName);
		if (program == null) {
			String[] source = createAmbientLightsSource(cr);
			program = shaders.registerProgram(programName, ShaderType.FRAGMENT_SHADER, null, source);
		}
		return program;
	}

	@ShaderSource
	public static final String[] SHADOW_MULTIPLY = {
		"uniform sampler2D shadow;",
		"uniform vec2 shadowSize;",
		"uniform float shadowDarkness;",
		"",
		"void main(void)",
		"{",
		"	gl_FragColor = vec4(1.0)-(vec4(1.0)-texture2D(shadow, gl_FragCoord.xy/shadowSize))*shadowDarkness;",
		"}"
	};

}
