/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.actions;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.ui.IWorkbenchWindow;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.project.DisableTimeRemapOperation;
import ch.kuramo.javie.app.project.ModifyLayerKeyframesOperation;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.views.layercomp.LayerAnimatableValueElement;
import ch.kuramo.javie.app.views.layercomp.LayerElement;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaLayer;

public class TimeRemapAction extends Action {

	private final IWorkbenchWindow window;

	private final TreeViewer viewer;


	public TimeRemapAction(IWorkbenchWindow window, TreeViewer viewer) {
		super("タイムリマップ");
		this.window = window;
		this.viewer = viewer;

		//setId(CommandIds.TIME_REMAP);
		//setActionDefinitionId(CommandIds.TIME_REMAP);
		//setImageDescriptor(Activator.getImageDescriptor("/icons/time_remap.png"));

		setEnabled(false);
		setChecked(false);

		viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				boolean enabled = false;
				boolean checked = false;

				TreeSelection selection = (TreeSelection) event.getSelection();
				if (selection.size() == 1) {
					Object element = selection.getFirstElement();
					if (element instanceof LayerElement) {
						Layer layer = ((LayerElement) element).layer;
						enabled = LayerNature.isTimeRemapNature(layer);
						checked = LayerNature.isTimeRemapEnabled(layer);
					} else if (element instanceof LayerAnimatableValueElement) {
						LayerAnimatableValueElement avalueElem = (LayerAnimatableValueElement) element;
						enabled = checked = avalueElem.getProperty().equals("timeRemap");
					}
				}

				setEnabled(enabled);
				setChecked(checked);
			}
		});
	}

	public void run() {
		TreeSelection selection = (TreeSelection) viewer.getSelection();
		Object element = selection.getFirstElement();

		Layer layer = null;
		if (element instanceof LayerElement) {
			layer = ((LayerElement) element).layer;
		} else if (element instanceof LayerAnimatableValueElement) {
			LayerAnimatableValueElement avalueElem = (LayerAnimatableValueElement) element;
			if (avalueElem.getProperty().equals("timeRemap")) {
				layer = ((LayerElement) avalueElem.parent).layer;
			}
		}

		if (layer == null || !LayerNature.isTimeRemapNature(layer)) {
			return;
		}

		ProjectManager pm = ProjectManager.forWorkbenchWindow(window);

		if (LayerNature.isTimeRemapEnabled(layer)) {
			pm.postOperation(new DisableTimeRemapOperation(pm, (MediaLayer) layer));

		} else {
			double rate = layer.getRate();
			Time duration = ((MediaLayer) layer).getMediaInput().getDuration();
			Time startTime = layer.getStartTime();
			Time endTime = startTime.add(new Time((long)(duration.timeValue/Math.abs(rate)), duration.timeScale));

			Time[] times = (rate == 0.0)
								? new Time[] { startTime }
								: new Time[] { startTime, endTime };

			Double[] values = (rate == 0.0)
								? new Double[] { 0.0 }
							: (rate > 0.0)
								? new Double[] { 0.0, duration.toSecond() }
								: new Double[] { duration.toSecond(), 0.0 };

			pm.postOperation(new ModifyLayerKeyframesOperation<Double>(
					pm, layer, "timeRemap", times, values, "タイムリマップの使用"));
		}
	}

}
