/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javassist.CannotCompileException;
import javassist.ClassPool;
import javassist.CtClass;
import javassist.CtConstructor;
import javassist.CtField;
import javassist.CtMethod;
import javassist.LoaderClassPath;
import javassist.Modifier;
import javassist.NotFoundException;
import javassist.bytecode.BadBytecode;
import javassist.bytecode.ClassFile;
import javassist.bytecode.MethodInfo;
import javassist.bytecode.SignatureAttribute;
import javassist.bytecode.SignatureAttribute.ClassType;
import javassist.bytecode.SignatureAttribute.NestedClassType;

import org.scannotation.AnnotationDB;

import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectConverter;
import ch.kuramo.javie.core.EffectDescriptor;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.EffectRegistry;

import com.google.inject.Inject;
import com.google.inject.Injector;

public class EffectRegistryImpl implements EffectRegistry {

	private final Map<String, EffectDescriptor> _registry = Util.newLinkedHashMap();

	private final Map<String, EffectConverter> _converters = Util.newMap();

	@Inject
	private Injector _injector;


	public void searchClasses(AnnotationDB db, ClassLoader cl) {
		Set<String> classNames = db.getAnnotationIndex().get(
				ch.kuramo.javie.api.annotations.Effect.class.getName());
		if (classNames == null) {
			return;
		}

		List<EffectDescriptor> descriptors = Util.newList();

		ClassPool pool = new ClassPool();
		pool.appendClassPath(new LoaderClassPath(cl));

		ClassLoader cl2 = Effect.class.getClassLoader();
		if (cl2 != cl) {
			pool.appendClassPath(new LoaderClassPath(cl2));
		}

		for (String className : classNames) {
			CtClass cc;
			try {
				cc = pool.get(className);
			} catch (NotFoundException e) {
				// 引数の AnnotationDB と ClassLoader の対応が正しければ必ず見つかるはず。
				throw new IllegalArgumentException(e);
			}

			for (Object anno : cc.getAvailableAnnotations()) {
				if (anno instanceof ch.kuramo.javie.api.annotations.Effect) {
					ch.kuramo.javie.api.annotations.Effect effectAnno
							= (ch.kuramo.javie.api.annotations.Effect) anno;

					String type = effectAnno.id();
					if (type.equals("")) {
						type = className;
					} else if (!type.contains(".")) {
						type = cc.getPackageName() + "." + type;
					}

					String convertTo = effectAnno.convertTo();
					if (convertTo.equals("")) {
						descriptors.add(new EffectClassProcessor(cl, type, cc).process());
						
					} else {
						if (!convertTo.contains(".")) {
							int lastDot = type.lastIndexOf('.');
							if (lastDot != -1) {
								convertTo = type.substring(0, lastDot+1) + convertTo;
							}
						}
						_converters.put(type, new ConverterClassProcessor(cl, convertTo, cc).process());
					}

					break;
				}
			}
		}

		// type順にソートして、常に同じ順序で登録されるようにする。
		Collections.sort(descriptors, new Comparator<EffectDescriptor>() {
			public int compare(EffectDescriptor o1, EffectDescriptor o2) {
				return o1.getType().compareTo(o2.getType());
			}
		});
		for (EffectDescriptor ed : descriptors) {
			_registry.put(ed.getType(), ed);
		}
	}

	public Collection<EffectDescriptor> getEffectDescriptors() {
		return Collections.unmodifiableCollection(_registry.values());
	}

	public EffectDescriptor getEffectDescriptor(String type) {
		return _registry.get(type);
	}

	public EffectConverter getEffectConverter(String type) {
		return _converters.get(type);
	}

	public Effect newEffect(String type) {
		EffectDescriptor desc = _registry.get(type);
		if (desc == null) {
			// TODO 例外を投げる？
			return null;
		}

		return _injector.getInstance(desc.getEffectClass());
	}

}

class EffectClassProcessor {

	private final ClassLoader cl;

	private final String type;

	private final ClassPool pool;

	private final CtClass ccInterior;

	private final CtClass ccExterior;

	private final CtClass ccExprInterface;

	private int countAEnum;


	EffectClassProcessor(ClassLoader cl, String type, CtClass ccInterior) {
		this.cl = cl;
		this.type = type;
		this.pool = ccInterior.getClassPool();
		this.ccInterior = ccInterior;

		try {
			ccExterior = pool.getAndRename(
					"ch.kuramo.javie.core.internal.templates.EffectTemplate",
					ccInterior.getName() + "$Exterior");

			ccExterior.setModifiers(Modifier.setPublic(ccExterior.getModifiers()));

			ccExprInterface = pool.makeInterface(ccInterior.getName() + "$ExpressionElement");

		} catch (NotFoundException e) {
			throw new JavieRuntimeException(e);
		}
	}

	EffectDescriptor process() {
		try {
			processProperties();

			setInteriorClass();
			setType();
			implementsVideoEffect();
			implementsAudioEffect();

			Class<?> interiorClass = ccInterior.toClass(cl, null);

			@SuppressWarnings("unchecked")
			Class<Effect> exteriorClass = ccExterior.toClass(cl, null);

			Class<?> exprInterface = ccExprInterface.toClass(cl, null);

			return new EffectDescriptorImpl(type, interiorClass, exteriorClass, exprInterface);

		} catch (ClassNotFoundException e) {
			throw new JavieRuntimeException(e);
		} catch (NotFoundException e) {
			throw new JavieRuntimeException(e);
		} catch (CannotCompileException e) {
			throw new JavieRuntimeException(e);
		} catch (BadBytecode e) {
			throw new JavieRuntimeException(e);
		}
	}

	private void processProperties() throws ClassNotFoundException, NotFoundException, CannotCompileException, BadBytecode {
//		CtClass enumType = pool.get("java.lang.Enum");

		for (CtField cf : ccInterior.getDeclaredFields()) {
			Property anno = (Property) cf.getAnnotation(Property.class);
			if (anno != null) {
				int mod = cf.getModifiers();
				if (Modifier.isPrivate(mod)) {
					cf.setModifiers(Modifier.setPackage(mod));
				}

				CtClass fieldType = cf.getType();
				String fieldTypeName = fieldType.getName();

				if (fieldTypeName.equals("ch.kuramo.javie.api.IAnimatableEnum")) {
					processAnimatableEnumProperty(cf);
				} else if (fieldTypeName.startsWith("ch.kuramo.javie.api.IAnimatable")) {
					processAnimatableValueProperty(cf);
//				} else if (fieldTypeName.equals("java.lang.String")
//							|| fieldType.subtypeOf(enumType) || fieldType.isPrimitive()) {
//					makeAccessor(fieldTypeName, cf.getName());
				} else {
					// 非サポートの型
					// TODO ログに警告を出す
				}
			}
		}
	}

	private void processAnimatableEnumProperty(CtField cf) throws NotFoundException, CannotCompileException, BadBytecode {
		SignatureAttribute fieldSig = (SignatureAttribute) cf.getFieldInfo().getAttribute(SignatureAttribute.tag);
		ClassType fieldType = (ClassType) SignatureAttribute.toFieldSignature(fieldSig.getSignature());
		ClassType enumType = (ClassType) fieldType.getTypeArguments()[0].getType();
		String enumName = enumType.getName();

		if (enumType instanceof NestedClassType) {
			enumName = ((NestedClassType) enumType).getDeclaringClass().getName() + "$" + enumName;
		}

		CtClass ccAEnum = makeAnimatableEnum(enumName);
		makeAccessor(ccAEnum.getName(), cf.getName());
	}

	private void processAnimatableValueProperty(CtField cf) throws NotFoundException, CannotCompileException {
		String fieldTypeName = cf.getType().getName();
		String accessorTypeName = fieldTypeName.replaceFirst(
				"^ch\\.kuramo\\.javie\\.api\\.IAnimatable",
				"ch.kuramo.javie.core.Animatable");

		makeAccessor(accessorTypeName, cf.getName());
	}

	private CtClass makeAnimatableEnum(String enumName) throws NotFoundException, CannotCompileException {
		String enumNameSlash = enumName.replace('.', '/');

		CtClass ccAEnum = pool.getAndRename("ch.kuramo.javie.core.internal.templates.AnimatableEnumTemplate",
									ccInterior.getName() + "$AnimatableEnum" + (++countAEnum));
		ccAEnum.setModifiers(Modifier.setPublic(ccAEnum.getModifiers()));

		CtClass ccEnum = pool.get(enumName);
		CtClass ccCollection = pool.get("java.util.Collection");
		CtClass ccString = pool.get("java.lang.String");

		CtConstructor cnstr1 = new CtConstructor(new CtClass[] { ccEnum, ccCollection, ccString }, ccAEnum);
		cnstr1.setBody(String.format("super(%s.class, $1, $2, $3);", enumName));
		ccAEnum.addConstructor(cnstr1);

		CtConstructor cnstr2 = new CtConstructor(new CtClass[] { ccEnum }, ccAEnum);
		cnstr2.setBody(String.format("super(%s.class, $1);", enumName));
		ccAEnum.addConstructor(cnstr2);

		ClassFile classFile = ccAEnum.getClassFile();
		SignatureAttribute classSig = new SignatureAttribute(classFile.getConstPool(),
				String.format("Lch/kuramo/javie/core/AbstractAnimatableEnum<L%1$s;>;Lch/kuramo/javie/api/IAnimatableEnum<L%1$s;>;", enumNameSlash));
		classFile.addAttribute(classSig);

		MethodInfo cnstr1Info = cnstr1.getMethodInfo();
		SignatureAttribute cstr1Sig = new SignatureAttribute(cnstr1Info.getConstPool(),
				String.format("(L%1$s;Ljava/util/Collection<Lch/kuramo/javie/core/Keyframe<L%1$s;>;>;Ljava/lang/String;)V", enumNameSlash));
		cnstr1Info.addAttribute(cstr1Sig);

		ccAEnum.toClass(cl, null);

		return ccAEnum;
	}

	private void makeAccessor(String typeName, String fieldName) throws CannotCompileException {
		String upperCamelCase = Character.toUpperCase(fieldName.charAt(0)) + fieldName.substring(1);
		String interiorName = ccInterior.getName();

		String getter = String.format("public %1$s get%2$s() { return (%1$s) ((%3$s) _interior).%4$s; }",
								typeName, upperCamelCase, interiorName, fieldName);
		ccExterior.addMethod(CtMethod.make(getter, ccExterior));

		String setter = String.format("public void set%2$s(%1$s value) {"
									+ "		if (((%3$s) _interior).%4$s != null) {"
									+ "			value.copyConfigurationFrom((%1$s) ((%3$s) _interior).%4$s);"
									+ "		}"
									+ "		((%3$s) _interior).%4$s = value;"
									+ "}",
								typeName, upperCamelCase, interiorName, fieldName);
		ccExterior.addMethod(CtMethod.make(setter, ccExterior));

		String exprGetter = String.format("public abstract Object get%s();", upperCamelCase);
		ccExprInterface.addMethod(CtMethod.make(exprGetter, ccExprInterface));
	}

	private void setInteriorClass() {
		try {
			CtMethod method = ccExterior.getDeclaredMethod("getInteriorClass");
			method.setBody(String.format("return %s.class;", ccInterior.getName()));
		} catch (NotFoundException e) {
			throw new JavieRuntimeException(e);
		} catch (CannotCompileException e) {
			throw new JavieRuntimeException(e);
		}
	}

	private void setType() {
		try {
			CtMethod method = ccExterior.getDeclaredMethod("getType");
			method.setBody(String.format("return \"%s\";", type.replaceAll("\"", "\\\\\"")));
		} catch (NotFoundException e) {
			throw new JavieRuntimeException(e);
		} catch (CannotCompileException e) {
			throw new JavieRuntimeException(e);
		}
	}

	private void implementsVideoEffect() {
		CtMethod inEffectMethod;
		try {
			inEffectMethod = ccInterior.getDeclaredMethod("doVideoEffect");
		} catch (NotFoundException e) {
			// this is ok.
			return;
		}

		CtMethod inBoundsMethod;
		try {
			inBoundsMethod = ccInterior.getDeclaredMethod("getVideoBounds");
		} catch (NotFoundException e) {
			// this is ok.
			inBoundsMethod = null;
		}

		try {
			int mod = inEffectMethod.getModifiers();
			if (Modifier.isPrivate(mod)) {
				inEffectMethod.setModifiers(Modifier.setPackage(mod));
			}

			CtMethod exEffectMethod = ccExterior.getDeclaredMethod("doVideoEffect");
			exEffectMethod.setBody(String.format("return ((%s) _interior).doVideoEffect();", ccInterior.getName()));

			if (inBoundsMethod != null) {
				mod = inBoundsMethod.getModifiers();
				if (Modifier.isPrivate(mod)) {
					inBoundsMethod.setModifiers(Modifier.setPackage(mod));
				}

				CtMethod exBoundsMethod = ccExterior.getDeclaredMethod("getVideoBounds");
				exBoundsMethod.setBody(String.format("return ((%s) _interior).getVideoBounds();", ccInterior.getName()));
			}

			ccExterior.addInterface(pool.get("ch.kuramo.javie.core.VideoEffect"));

		} catch (NotFoundException e) {
			throw new JavieRuntimeException(e);
		} catch (CannotCompileException e) {
			throw new JavieRuntimeException(e);
		}
	}

	private void implementsAudioEffect() {
		CtMethod inMethod;
		try {
			inMethod = ccInterior.getDeclaredMethod("doAudioEffect");
		} catch (NotFoundException e) {
			// this is ok.
			return;
		}

		try {
			int mod = inMethod.getModifiers();
			if (Modifier.isPrivate(mod)) {
				inMethod.setModifiers(Modifier.setPackage(mod));
			}

			CtMethod exMethod = ccExterior.getDeclaredMethod("doAudioEffect");
			exMethod.setBody(String.format("return ((%s) _interior).doAudioEffect();", ccInterior.getName()));

			ccExterior.addInterface(pool.get("ch.kuramo.javie.core.AudioEffect"));

		} catch (NotFoundException e) {
			throw new JavieRuntimeException(e);
		} catch (CannotCompileException e) {
			throw new JavieRuntimeException(e);
		}
	}

}

class ConverterClassProcessor {
	
	private final ClassLoader cl;

	private final String convertTo;

	private final CtClass cc;


	ConverterClassProcessor(ClassLoader cl, String convertTo, CtClass cc) {
		this.cl = cl;
		this.convertTo = convertTo;
		this.cc = cc;
	}

	EffectConverter process() {
		try {
			ClassPool pool = cc.getClassPool();

			CtClass[] methodParams = new CtClass[] { pool.get("java.util.Map") };
			try {
				cc.getDeclaredMethod("convert", methodParams);
			} catch (NotFoundException e) {
				// idの変更だけの場合は、convertメソッドを実装する必要はない。
				cc.addMethod(CtMethod.make("public void convert(java.util.Map map) { }", cc));
			}

			String getNewTypeMethod = String.format(
					"public String getNewType() { return \"%s\"; }",
					convertTo.replaceAll("\"", "\\\\\""));
			cc.addMethod(CtMethod.make(getNewTypeMethod, cc));

			cc.addInterface(pool.get("ch.kuramo.javie.core.EffectConverter"));

			@SuppressWarnings("unchecked")
			Class<EffectConverter> clazz = cc.toClass(cl, null);
			return clazz.newInstance();

		} catch (NotFoundException e) {
			throw new JavieRuntimeException(e);
		} catch (CannotCompileException e) {
			throw new JavieRuntimeException(e);
		} catch (InstantiationException e) {
			throw new JavieRuntimeException(e);
		} catch (IllegalAccessException e) {
			throw new JavieRuntimeException(e);
		}
	}

}
