/*
 * Copyright (c) 2010,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.blurSharpen;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableInteger;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.IVideoBuffer.TextureWrapMode;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.SpinBlur", category=Categories.BLUR_AND_SHARPEN)
public class SpinBlur {

	public enum Direction { CLOCKWISE, COUNTERCLOCKWISE, BOTH } 


	@Property(value="0", min="0", max="360")
	private IAnimatableDouble amount;

	@Property
	private IAnimatableVec2d center;

	@Property("BOTH")
	private IAnimatableEnum<Direction> direction;

	@Property(value="100", min="0")
	private IAnimatableInteger maxSamples;

	@Property("true")
	private IAnimatableBoolean fast;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram clockwiseProgram;

	private final IShaderProgram countercwProgram;

	private final IShaderProgram bothProgram;

	@Inject
	public SpinBlur(IVideoEffectContext context, IVideoRenderSupport support, IShaderRegistry shaders) {
		this.context = context;
		this.support = support;
		clockwiseProgram = shaders.getProgram(SpinBlur.class, "CLOCKWISE");
		countercwProgram = shaders.getProgram(SpinBlur.class, "COUNTERCLOCKWISE");
		bothProgram = shaders.getProgram(SpinBlur.class, "BOTH");
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer buffer = context.doPreviousEffect();
		VideoBounds bounds = buffer.getBounds();
		if (bounds.isEmpty()) {
			return buffer;
		}

		double amount = context.value(this.amount);
		int maxSamples = context.value(this.maxSamples);
		if (amount == 0 || maxSamples == 0) {
			return buffer;
		}

		Resolution resolution = context.getVideoResolution();
		Vec2d center = resolution.scale(context.value(this.center));
		Direction direction = context.value(this.direction);
		boolean fast = context.value(this.fast);

		Set<IVideoBuffer> tmpBuffers = new HashSet<IVideoBuffer>();
		tmpBuffers.add(buffer);

		try {
			IShaderProgram program;
			switch (direction) {
				case CLOCKWISE:
					program = clockwiseProgram;
					++maxSamples;
					break;

				case COUNTERCLOCKWISE:
					program = countercwProgram;
					++maxSamples;
					break;

				case BOTH:
					program = bothProgram;
					amount *= 0.5;
					break;

				default:
					throw new RuntimeException("unknown spin direction: " + direction);
			}

			for (int ms : multipassMaxSamples(maxSamples, fast)) {
				Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
				uniforms.add(new GLUniformData("texture", 0));
				uniforms.add(new GLUniformData("amount", (float)Math.toRadians(amount)));
				uniforms.add(new GLUniformData("center", 2, toFloatBuffer(center.x-bounds.x, center.y-bounds.y)));
				uniforms.add(new GLUniformData("size", 2, toFloatBuffer(bounds.width, bounds.height)));
				uniforms.add(new GLUniformData("maxSamples", (float)ms));

				buffer.setTextureFilter(TextureFilter.LINEAR);
				buffer.setTextureWrapMode(TextureWrapMode.CLAMP_TO_EDGE);

				buffer = support.useShaderProgram(program, uniforms, null, buffer);
				tmpBuffers.add(buffer);

				amount /= ms;
			}

			tmpBuffers.remove(buffer);
			return buffer;

		} finally {
			for (IVideoBuffer vb : tmpBuffers) {
				vb.dispose();
			}
		}
	}

	private int[] multipassMaxSamples(int maxSamples, boolean fast) {
		if (!fast || maxSamples <= 10) {
			return new int[] { maxSamples };

		} else if (maxSamples <= 100) {
			int ms1 = (int)Math.ceil(Math.sqrt(maxSamples));
			int ms2 = (int)Math.ceil((double)maxSamples / ms1);
			return new int[] { ms1, ms2 };

		} else {
			int ms1 = (int)Math.ceil(Math.cbrt(maxSamples));
			int ms2 = (int)Math.ceil((double)maxSamples / ms1 / ms1);
			return new int[] { ms1, ms1, ms2 };
		}
	}

	private FloatBuffer toFloatBuffer(double... values) {
		float[] array = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			array[i] = (float)values[i];
		}
		return FloatBuffer.wrap(array);
	}

	private static String[] createShaderSource(Direction direction) {
		boolean cw = (direction != Direction.COUNTERCLOCKWISE);
		boolean ccw = (direction != Direction.CLOCKWISE);

		return new String[] {
						"uniform sampler2D texture;",
						"uniform float amount;",
						"uniform vec2 center;",
						"uniform vec2 size;",
						"uniform float maxSamples;",
						"",
						"void main(void)",
						"{",
						"	vec2 coord = gl_TexCoord[0].st;",
						"	vec4 color = texture2D(texture, coord);",
						"",
						"	vec2 v = coord * size - center;",
						"	float r = length(v);",
						"",
						"	if (r > 0.0) {",
						"		float arc = r*amount;",
						"		int   n = int(min(arc+1.0, maxSamples));",
						"		for (int i = 1; i < n; ++i) {",
						"			float t = amount*float(i)/float(n);",
			cw ?		"			color += texture2D(texture, (v * mat2(cos(t), sin(t), -sin(t), cos(t)) + center) / size);" : "",
			ccw ?		"			color += texture2D(texture, (v * mat2(cos(t), -sin(t), sin(t), cos(t)) + center) / size);" : "",
						"		}",
			cw && ccw ?	"		color += texture2D(texture, (v * mat2(cos(amount), sin(amount), -sin(amount), cos(amount)) + center) / size) * 0.5;" : "",
			cw && ccw ?	"		color += texture2D(texture, (v * mat2(cos(amount), -sin(amount), sin(amount), cos(amount)) + center) / size) * 0.5;" : "",
			cw && ccw ?	"		color /= float(2*n);"
					  : "		color /= float(n);",
						"	}",
						"",
						"	gl_FragColor = color;",
						"}"
		};
	}

	@ShaderSource
	public static final String[] CLOCKWISE = createShaderSource(Direction.CLOCKWISE);

	@ShaderSource
	public static final String[] COUNTERCLOCKWISE = createShaderSource(Direction.COUNTERCLOCKWISE);

	@ShaderSource
	public static final String[] BOTH = createShaderSource(Direction.BOTH);



	// SpinBlur を最初にリリースしたとき、クラス名とIDを誤って SpinZoom にしていた。
	// 紛らわしいのでクラス名を訂正。そして、このコンバータでIDも変換する。
	@Effect(id="ch.kuramo.javie.SpinZoom", convertTo="SpinBlur")
	public static class ConvertFromSpinZoom { }

}
