/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.utility;

import ch.kuramo.javie.api.IAnimatableInteger;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.ChangeBounds", category=Categories.UTILITY)
public class ChangeBounds {

	@Property(max="1000")				// TODO 最大値はどのくらいまであればばよいだろうか。
	private IAnimatableInteger	left;	//      あまり大きくするとテクスチャの確保に失敗すると思われる。以下同様。

	@Property(max="1000")
	private IAnimatableInteger	top;

	@Property(max="1000")
	private IAnimatableInteger	right;

	@Property(max="1000")
	private IAnimatableInteger	bottom;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	@Inject
	public ChangeBounds(IVideoEffectContext context, IVideoRenderSupport support) {
		this.context = context;
		this.support = support;
	}

	private VideoBounds calcNewBounds(VideoBounds oldBounds) {
		Resolution resolution = context.getVideoResolution();
		int left = (int) resolution.scale(context.value(this.left));
		int top = (int) resolution.scale(context.value(this.top));
		int right = (int) resolution.scale(context.value(this.right));
		int bottom = (int) resolution.scale(context.value(this.bottom));

		VideoBounds newBounds = new VideoBounds(
				oldBounds.x - left,
				oldBounds.y - top,
				Math.max(0, oldBounds.width + left + right),
				Math.max(0, oldBounds.height + top + bottom));

		return newBounds;
	}

	public VideoBounds getVideoBounds() {
		return calcNewBounds(context.getPreviousBounds());
	}

	public IVideoBuffer doVideoEffect() {
		final IVideoBuffer input = context.doPreviousEffect();

		VideoBounds oldBounds = input.getBounds();
		VideoBounds newBounds = calcNewBounds(oldBounds);

		if (newBounds.equals(oldBounds)) {
			return input;
		}

		if (newBounds.isEmpty()) {
			input.dispose();
			return context.createVideoBuffer(newBounds);
		}

		IVideoBuffer buffer = null;
		try {
			final IVideoBuffer output = buffer = context.createVideoBuffer(newBounds);

			Runnable operation = new Runnable() {
				public void run() {
					support.ortho2D(output);
					support.quad2D(output, input);
				}
			};

			support.useFramebuffer(operation, 0, output, input);

			buffer = null;
			return output;

		} finally {
			input.dispose();
			if (buffer != null) buffer.dispose();
		}
	}

}
