/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaItemLayer;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.PropertyDescriptor;
import ch.kuramo.javie.core.Util;

public class LayerSlipOperation extends ProjectOperation {

	private static class Entry<V> {
		final String layerId;
		final int effectIndex;
		final String property;
		final Set<Keyframe<V>> keyframesToShift;
		final Set<Keyframe<V>> restOfKeyframes;

		Entry(Layer layer, Object[] data) {
			layerId = layer.getId();
			effectIndex = (Integer) data[1];
			property = (String) data[2];

			@SuppressWarnings("unchecked")
			Set<Keyframe<V>> tmp = (Set<Keyframe<V>>) data[3];
			keyframesToShift = tmp;

			AnimatableValue<V> avalue;
			if (effectIndex == -1) {
				avalue = PropertyUtil.getProperty(layer, property);
			} else {
				Effect effect = ((EffectableLayer) layer).getEffects().get(effectIndex);
				PropertyDescriptor pd = effect.getEffectDescriptor().getPropertyDescriptor(property);
				@SuppressWarnings("unchecked")
				AnimatableValue<V> tmp2 = (AnimatableValue<V>) pd.get(effect);
				avalue = tmp2;
			}
			restOfKeyframes = Util.<Keyframe<V>>newSet(avalue.getKeyframeMap().values());
			restOfKeyframes.removeAll(keyframesToShift);
		}
	}


	private final String _compId;

	private final Map<String, Time> _startTimes;

	private final List<Entry<?>> _entries;

	private volatile Time _deltaTime;


	public LayerSlipOperation(
			ProjectManager projectManager, LayerComposition comp,
			Object[][] layersAndStartTimes, Object[][] keyframeData,
			Time deltaTime, String relation) {

		super(projectManager, "レイヤー時間の変更", relation);

		projectManager.checkComposition(comp);
		_compId = comp.getId();
		_deltaTime = clampDeltaTime(deltaTime, layersAndStartTimes);

		Map<String, Time> startTimes = null;
		List<Entry<?>> entries = null;

		if (keyframeData != null) {
			if (_deltaTime.timeValue != 0) {
				startTimes = Util.newMap();
				for (Object[] o : layersAndStartTimes) {
					Layer layer = (Layer) o[0];
					startTimes.put(layer.getId(), (Time) o[1]);
				}

				entries = Util.newList();
				for (Object[] data : keyframeData) {
					Layer layer = (Layer) data[0];
					if (projectManager.checkLayer(layer) != comp) {
						throw new IllegalArgumentException();
					}
					@SuppressWarnings("unchecked")
					Entry<?> entry = new Entry(layer, data);
					entries.add(entry);
				}
			} else {
				_noEffect = true;
			}
		}

		_startTimes = startTimes;
		_entries = entries;
	}

	private Time clampDeltaTime(Time deltaTime, Object[][] layersAndStartTimes) {
		for (Object[] o : layersAndStartTimes) {
			Layer layer = (Layer) o[0];
			if (layer instanceof MediaItemLayer) {
				MediaInput input = ((MediaItemLayer) layer).getMediaInput();
				if (input != null) {
					Time duration = input.getDuration();
					if (duration != null) {
						Time inPoint = layer.getInPoint();
						Time outPoint = layer.getOutPoint();
						Time startTime = (Time) o[1];
						Time endTime = startTime.add(new Time((long)(duration.timeValue/Math.abs(layer.getRate())), duration.timeScale));

						Time t = inPoint.subtract(startTime);
						if (t.before(deltaTime)) {
							deltaTime = t;
						}

						t = outPoint.subtract(endTime);
						if (t.after(deltaTime)) {
							deltaTime = t;
						}
					}
				}
			}
		}
		return deltaTime;
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		slip(true, project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		slip(false, project, pm);
		return Status.OK_STATUS;
	}

	private void slip(boolean shift, Project project, ProjectManager pm) throws ExecutionException {
		if (_startTimes == null || _entries == null) {
			throw new IllegalStateException();
		}

		LayerComposition comp = project.getComposition(_compId);

		List<Layer> data1 = shiftStartTimeOrRestore(shift, pm != null, comp);
		Object[][] data2 = shiftKeyframesOrRestore(shift, pm != null, comp);

		if (pm != null) {
			pm.fireLayerSlipEdit(comp, new Object[] { data1, data2 });
		}
	}

	private List<Layer> shiftStartTimeOrRestore(boolean shift, boolean createData, LayerComposition comp) {
		List<Layer> layers = createData ? Util.<Layer>newList() : null;

		for (Map.Entry<String, Time> e : _startTimes.entrySet()) {
			Layer layer = comp.getLayer(e.getKey());
			Time startTime = e.getValue();
			layer.setStartTime(shift ? _deltaTime.add(startTime) : startTime);

			if (createData) {
				layers.add(layer);
			}
		}

		return layers;
	}

	private Object[][] shiftKeyframesOrRestore(boolean shift, boolean createData, LayerComposition comp) throws ExecutionException {
		List<Object[]> data = createData ? Util.<Object[]>newList() : null;

		for (Entry<?> e : _entries) {
			Layer layer = comp.getLayer(e.layerId);

			AnimatableValue<Object> avalue;
			if (e.effectIndex == -1) {
				avalue = PropertyUtil.getProperty(layer, e.property);
			} else {
				Effect effect = ((EffectableLayer) layer).getEffects().get(e.effectIndex);
				PropertyDescriptor pd = effect.getEffectDescriptor().getPropertyDescriptor(e.property);
				@SuppressWarnings("unchecked")
				AnimatableValue<Object> tmp = (AnimatableValue<Object>) pd.get(effect);
				avalue = tmp;
			}

			@SuppressWarnings("unchecked")
			Entry<Object> eo = (Entry<Object>) e;
			List<Time> times = shiftKeyframesOrRestore(shift, avalue, eo, createData);

			if (createData) {
				data.add(new Object[] { layer, e.effectIndex, e.property, times });
			}
		}

		return createData ? data.toArray(new Object[data.size()][]) : null;
	}

	private <V> List<Time> shiftKeyframesOrRestore(boolean shift, AnimatableValue<V> avalue, Entry<V> entry, boolean returnTimes) {
		avalue.clearKeyframes(avalue.getStaticValue());
		for (Keyframe<V> kf : entry.restOfKeyframes) {
			avalue.putKeyframe(kf);
		}

		List<Time> times = returnTimes ? new ArrayList<Time>(entry.keyframesToShift.size()) : null;
		for (Keyframe<V> kf : entry.keyframesToShift) {
			Time time;
			if (shift) {
				time = _deltaTime.add(kf.time);
				avalue.putKeyframe(new Keyframe<V>(time, kf.value, kf.interpolation));
			} else {
				time = kf.time;
				avalue.putKeyframe(kf);
			}

			if (returnTimes) {
				times.add(time);
			}
		}
		return times;
	}

	@Override
	protected boolean merge(ProjectOperation operation, IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) {

		if (pm == null) {
			try {
				executeOrRedo(monitor, info, project, null);
			} catch (ExecutionException e) {
				throw new JavieRuntimeException(e);
			}
			return true;
		}

		LayerSlipOperation newOp = (LayerSlipOperation) operation;

		if (newOp._startTimes != null || newOp._entries != null) {
			throw new IllegalStateException();
		}

		if (!newOp._deltaTime.equals(_deltaTime)) {
			_deltaTime = newOp._deltaTime;

			try {
				executeOrRedo(monitor, info, project, pm);
			} catch (ExecutionException e) {
				throw new JavieRuntimeException(e);
			}

			// 結合の結果、このオペレーションの直前の値に戻ったかどうか。
			_noEffect = (_deltaTime.timeValue == 0);
		}

		return true;
	}

}
