/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views;

import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.ListenerList;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.util.LocalSelectionTransfer;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.EditingSupport;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ITreeSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeColumnViewerLabelProvider;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DragSourceEvent;
import org.eclipse.swt.dnd.DragSourceListener;
import org.eclipse.swt.dnd.DropTargetAdapter;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.DropTargetListener;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.dnd.TreeDragSourceEffect;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Scale;
import org.eclipse.swt.widgets.Slider;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.ViewPart;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.ColorUtil;
import ch.kuramo.javie.app.CommandIds;
import ch.kuramo.javie.app.ImageUtil;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.IntervalTask;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.app.UIUtil;
import ch.kuramo.javie.app.actions.AVIOutputAction;
import ch.kuramo.javie.app.actions.AddEffectAction;
import ch.kuramo.javie.app.actions.AddExpressionsAction;
import ch.kuramo.javie.app.actions.KeyframeInterpolationAction;
import ch.kuramo.javie.app.actions.PrecomposeAction;
import ch.kuramo.javie.app.actions.QTMovieOutputAction;
import ch.kuramo.javie.app.actions.RemoveKeyframesAction;
import ch.kuramo.javie.app.actions.RenameLayerEtcAction;
import ch.kuramo.javie.app.actions.ReplaceLayerItemAction;
import ch.kuramo.javie.app.actions.ResetPropertiesAction;
import ch.kuramo.javie.app.actions.ReverseLayerTimeAction;
import ch.kuramo.javie.app.actions.SequenceOutputAction;
import ch.kuramo.javie.app.actions.TimeRemapAction;
import ch.kuramo.javie.app.actions.TimeStretchLayerAction;
import ch.kuramo.javie.app.actions.WaveOutputAction;
import ch.kuramo.javie.app.player.MediaPlayer;
import ch.kuramo.javie.app.player.PlayerLinkEvent;
import ch.kuramo.javie.app.player.PlayerLinkListener;
import ch.kuramo.javie.app.project.DuplicateEffectsOperation;
import ch.kuramo.javie.app.project.DuplicateLayersOperation;
import ch.kuramo.javie.app.project.DuplicateTASelectorsOperation;
import ch.kuramo.javie.app.project.DuplicateTextAnimatorsOperation;
import ch.kuramo.javie.app.project.MoveEffectsOperation;
import ch.kuramo.javie.app.project.MoveTASelectorsOperation;
import ch.kuramo.javie.app.project.MoveTextAnimatorsOperation;
import ch.kuramo.javie.app.project.NewCameraLayerOperation;
import ch.kuramo.javie.app.project.NewLayerFromItemOperation;
import ch.kuramo.javie.app.project.NewLightLayerOperation;
import ch.kuramo.javie.app.project.NewNullLayerOperation;
import ch.kuramo.javie.app.project.NewTextLayerOperation;
import ch.kuramo.javie.app.project.ProjectEvent;
import ch.kuramo.javie.app.project.ProjectEventHub;
import ch.kuramo.javie.app.project.ProjectListener;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.project.RemoveEffectsOperation;
import ch.kuramo.javie.app.project.RemoveLayersOperation;
import ch.kuramo.javie.app.project.RemoveTAPropertiesOperation;
import ch.kuramo.javie.app.project.RemoveTASelectorsOperation;
import ch.kuramo.javie.app.project.RemoveTextAnimatorsOperation;
import ch.kuramo.javie.app.project.ReorderLayersOperation;
import ch.kuramo.javie.app.project.ProjectEvent.Type;
import ch.kuramo.javie.app.views.layercomp.AnimatableValueElement;
import ch.kuramo.javie.app.views.layercomp.AnimatableValueElementDelegate;
import ch.kuramo.javie.app.views.layercomp.EffectAnimatableValueElement;
import ch.kuramo.javie.app.views.layercomp.EffectElement;
import ch.kuramo.javie.app.views.layercomp.EffectsElement;
import ch.kuramo.javie.app.views.layercomp.Element;
import ch.kuramo.javie.app.views.layercomp.LayerAnimatableValueElement;
import ch.kuramo.javie.app.views.layercomp.LayerCompFrameBlendAction;
import ch.kuramo.javie.app.views.layercomp.LayerCompMotionBlurAction;
import ch.kuramo.javie.app.views.layercomp.LayerCompSettingsAction;
import ch.kuramo.javie.app.views.layercomp.LayerCompShyAction;
import ch.kuramo.javie.app.views.layercomp.LayerCompSuperSamplingAction;
import ch.kuramo.javie.app.views.layercomp.LayerElement;
import ch.kuramo.javie.app.views.layercomp.LayerPropertyElement;
import ch.kuramo.javie.app.views.layercomp.TASelectorElement;
import ch.kuramo.javie.app.views.layercomp.TextAnimatorElement;
import ch.kuramo.javie.app.views.layercomp.TextAnimatorsElement;
import ch.kuramo.javie.app.views.layercomp.TextElement;
import ch.kuramo.javie.app.views.layercomp.TimelineManager;
import ch.kuramo.javie.app.widgets.FontUtil;
import ch.kuramo.javie.core.AnimatableLayerReference;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectDescriptor;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.Folder;
import ch.kuramo.javie.core.Interpolation;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.ItemLayer;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LightLayer;
import ch.kuramo.javie.core.MediaItem;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.TASelector;
import ch.kuramo.javie.core.TextAnimator;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.TimeCode;
import ch.kuramo.javie.core.TransformableLayer;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.output.MacOSXQTMovieOutput;
import ch.kuramo.javie.core.output.WindowsDirectShowOutput;
import ch.kuramo.javie.core.services.AnimatableValueCollector;
import ch.kuramo.javie.core.services.EffectRegistry;
import ch.kuramo.javie2mmd.Javie2MMDAction;

import com.google.inject.Inject;
import com.ibm.icu.text.Normalizer;
import com.ibm.icu.text.Transliterator;

public class LayerCompositionView extends ViewPart
		implements ProjectListener, PlayerLinkListener {

	public static final String ID = "ch.kuramo.javie.app.views.layerCompositionView";

	public static final int NAME_COL		= 0;
	public static final int VALUE_COL		= 1;
	public static final int MODE_COL		= 2;
	public static final int TRACKMATTE_COL	= 3;
	public static final int PARENT_COL		= 4;
	public static final int SHOWHIDE_COL	= 5;
	public static final int LABEL_COL		= 6;
	public static final int TIMELINE_COL	= 7;

	public static final String PROJECT_MANAGER = "PROJECT_MANAGER";
	public static final String LAYER_COMPOSITION = "LAYER_COMPOSITION";
	public static final String ANIMATABLE_VALUES = "ANIMATABLE_VALUES";
	public static final String TIMELINE_MANAGER = "TIMELINE_MANAGER";
	public static final String LAYER_NAME_MODE = "LAYER_NAME_MODE";
	public static final String EDIT_ELEMENT_NAME = "EDIT_ELEMENT_NAME";


	private static final Logger _logger = LoggerFactory.getLogger(LayerCompositionView.class);

	private static final boolean COCOA = SWT.getPlatform().equals("cocoa");

	private static final boolean WIN32 = SWT.getPlatform().equals("win32");

	private static final int ROW_PADDING = 4;


	private Text _searchText;

	private Composite _timeCodeComposite;

	private Label _fpsLabel;

	private Composite _topSpacer;

	private Composite _bottomSpacer;

	private Composite _timelineMeter;

	private Scale _timelineScale;

	private Slider _timelineSlider;

	private TreeViewer _treeViewer;

	private TreeViewerColumn _nameColumn;

	private Point _mouseDownPoint;

	private volatile Time _time;

	private IntervalTask _timelineUpdateTask;

	private IntervalTask _collectTask;

	private ProjectManager _projectManager;

	private CompositionItem _compItem;

	private LayerComposition _composition;

	private TimelineManager _timelineManager;

	private Action _deleteAction;

	private Action _splitLayersAction;

	@Inject
	private EffectRegistry _effectRegistry;

	@Inject
	private AnimatableValueCollector _collector;


	public LayerCompositionView() {
		super();
		InjectorHolder.getInjector().injectMembers(this);

		_collector.activate();
	}

	public void createPartControl(final Composite parent) {
		ProjectManager pm = ProjectManager.forWorkbenchWindow(getSite().getWorkbenchWindow());
		if (pm == null) {
			throw new IllegalStateException("No ProjectManager exists for this window.");
		}

		parent.setLayout(new FormLayout());

		_topSpacer = new Composite(parent, SWT.NONE);
		_bottomSpacer = new Composite(parent, SWT.NONE);

		createTreeViewer(parent);
		createTimelineControls(parent);
		createSearchText(parent);
		createTimeCodeControls(parent);
		setLayoutData();

		setupGlobalActionsAndContextMenu();
		setupDragAndDrop();

		_timelineUpdateTask = new IntervalTask(parent.getDisplay(), 50) {
			public void run() {
				if (!parent.isDisposed()) {
					if (_timelineManager != null) {
						_timelineManager.update(_time, true);
					}
					_timeCodeComposite.redraw();
					_splitLayersAction.setEnabled(canSplitSelectedLayers());
				}
			}
		};

		_collectTask = new IntervalTask(parent.getDisplay(), 300) {
			public void run() {
				if (!parent.isDisposed()) {
					collectAnimatableValues(_time);
				}
			}
		};

		setProjectManager(pm);
		setupActionBars();

		ProjectEventHub.getHub().addProjectListener(this);
	}

	public void dispose() {
		ProjectEventHub.getHub().removeProjectListener(this);

		_collector.deactivate();

		super.dispose();
	}

	public void setFocus() {
		_treeViewer.getControl().setFocus();
	}

	private String getItemId() {
		return getViewSite().getSecondaryId();
	}

	private void setProjectManager(ProjectManager pm) {
		_compItem = pm.getProject().getItem(getItemId());
		if (_compItem == null) {
			throw new IllegalStateException("no such CompositionItem found: " + getItemId());
		}

		_projectManager = pm;
		_composition = (LayerComposition) _compItem.getComposition();


		setPartName(_compItem.getName());

		_treeViewer.setData(PROJECT_MANAGER, pm);
		_treeViewer.setData(LAYER_COMPOSITION, _composition);
		_treeViewer.setInput(_composition);

		Time frameDuration = _composition.getFrameDuration();
		_fpsLabel.setText(String.format("(%.2f fps)", (double)frameDuration.timeScale / frameDuration.timeValue));

		_time = Time.fromFrameNumber(0, frameDuration);
		_timelineManager = new TimelineManager(
				pm, _composition, this, _timelineMeter, _timelineScale, _timelineSlider, _treeViewer.getTree());
		_treeViewer.setData(TIMELINE_MANAGER, _timelineManager);

		collectAnimatableValues();
		connectToPlayer();
	}

	public void handleProjectEvent(ProjectEvent event) {
		if (event.getProjectManager() != _projectManager) {
			return;
		}

		switch (event.type) {
			case LAYER_PROPERTY_CHANGE:
			case LAYER_ITEM_CHANGE:
			case LAYERS_REMOVE:
			case EFFECT_PROPERTY_CHANGE:
			case EFFECTS_REMOVE:
				if (event.composition == _composition) {
					_timelineManager.clearKeyframeSelection();
					_timelineManager.redraw();
				}
				break;
		}

		switch (event.type) {
			case ITEM_NAME_CHANGE:
				handleItemUpdate(event, true);
				break;

			case ITEM_UPDATE:
				handleItemUpdate(event, false);
				break;

			case COMPOSITION_SETTINGS_CHANGE:
				handleCompositionSettingsChange(event);
				break;

			case COMPOSITION_PROPERTY_CHANGE:
				handleCompositionPropertyChange(event);
				break;

			case LAYER_PROPERTY_CHANGE:
				handleLayerPropertyChange(event);
				break;

			case LAYER_EXPRESSION_CHANGE:
				handleLayerExpressionChange(event);
				break;

			case LAYER_ITEM_CHANGE:
				handleLayerItemChange(event);
				break;

			case LAYERS_ADD:
			case LAYERS_REMOVE:
			case LAYERS_REORDER:
				handleLayersAddRemoveReorder(event);
				break;

			case EXPRESSIONS_ADD:
			case EXPRESSIONS_REMOVE:
				handleExpressionsAddRemove(event);
				break;

			case TEXT_ANIMATORS_ADD:
			case TEXT_ANIMATORS_REMOVE:
				handleTextAnimatorsAddRemove(event);
				break;

			case TA_SELECTORS_ADD:
			case TA_SELECTORS_REMOVE:
				handleTASelectorsAddRemove(event);
				break;

			case TA_PROPERTIES_ADD:
			case TA_PROPERTIES_REMOVE:
				handleTAPropertiesAddRemove(event);
				break;

			case EFFECTS_ADD:
			case EFFECTS_REMOVE:
				handleEffectsAddRemove(event);
				break;

			case EFFECT_PROPERTY_CHANGE:
				handleEffectPropertyChange(event);
				break;

			case EFFECT_EXPRESSION_CHANGE:
				handleEffectExpressionChange(event);
				break;

			case LAYER_TIMES_CHANGE:
				handleLayerTimesChange(event);
				break;

			case KEYFRAMES_CHANGE:
				handleKeyframesChange(event);
				break;

			case LAYER_SLIP_EDIT:
				handleLayerSlipEdit(event);
				break;
		}
	}

	private void createTimelineControls(Composite parent) {
		_timelineMeter = new Composite(parent, WIN32 ? SWT.DOUBLE_BUFFERED : SWT.NONE);
		_timelineScale = new Scale(parent, SWT.HORIZONTAL);
		_timelineSlider = new Slider(parent, SWT.HORIZONTAL);

		FontUtil.setCompatibleFont(_timelineMeter);

		SelectionListener selectionListener = new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				_timelineManager.update(_time, false);
			}
		};

		_timelineScale.addSelectionListener(selectionListener);
		_timelineSlider.addSelectionListener(selectionListener);
	}

	private void createTreeViewer(Composite parent) {
		class DragSourceEffect extends TreeDragSourceEffect {
			private boolean creatingImage;

			public DragSourceEffect(Tree tree) {
				super(tree);
			}

			public void dragStart(DragSourceEvent event) {
				creatingImage = true;
				try {
					super.dragStart(event);
				} finally {
					creatingImage = false;
				}
			}
		}

		_treeViewer = new TreeViewer(parent, SWT.MULTI | SWT.NO_SCROLL | SWT.V_SCROLL);
		_treeViewer.setUseHashlookup(true);
		_treeViewer.setContentProvider(new LayerCompositionContentProvider());

		Tree tree = _treeViewer.getTree();
		tree.setHeaderVisible(true);
		tree.setLinesVisible(!COCOA);

		FontUtil.setCompatibleFont(tree);

		final DragSourceEffect dragEffect = new DragSourceEffect(tree);
		tree.setData("DEFAULT_DRAG_SOURCE_EFFECT", dragEffect);

		_nameColumn =	createViewerColumn(NAME_COL,      200, null, "ソース名");
						createViewerColumn(VALUE_COL,     150, ImageUtil.getSwitchesHeaderIcon(), "");
						createViewerColumn(MODE_COL,       50, null, "モード");
						createViewerColumn(TRACKMATTE_COL, 50, null, "ﾄﾗｯｸﾏｯﾄ");
						createViewerColumn(PARENT_COL,    100, null, "親");
						createViewerColumn(SHOWHIDE_COL,   60, ImageUtil.getShowhideHeaderIcon(), "");
						createViewerColumn(LABEL_COL,      40, null, "");
						createViewerColumn(TIMELINE_COL,  300, null, "");

		setLayerNameMode(false);


		ControlAdapter columnWidthAdjuster = new ControlAdapter() {
			private boolean adjusting = false;
			public void controlResized(ControlEvent e) {
				if (!adjusting) {
					adjusting = true;
					try {
						adjustColumnWidth();

						if (_timelineManager != null) {
							_timelineManager.update(_time, false);
						}
					} finally {
						adjusting = false;
					}
				}
			}
		};

		tree.addControlListener(columnWidthAdjuster);

		for (TreeColumn column : tree.getColumns()) {
			column.addControlListener(columnWidthAdjuster);
		}


		tree.setBackground(ColorUtil.tableBackground());

		if (COCOA) {
			final Color layerColor = ColorUtil.layerRowBackground();
			final Color lineColor = ColorUtil.tableRowLine();

			tree.addListener(SWT.EraseItem, new Listener() {
				public void handleEvent(Event event) {
					if (dragEffect.creatingImage) {
						return;
					}

					GC gc = event.gc;

					TreeItem treeItem = (TreeItem) event.item;
					Object element = treeItem.getData();
					if (element instanceof LayerElement) {
						gc.setBackground(layerColor);
						gc.fillRectangle(event.x, event.y, event.width, event.height - 1);
					}

					int x = event.x;
					int y = event.y + event.height - 1;
					gc.setForeground(lineColor);
					gc.drawLine(x, y, x + event.width, y);
				}
			});

		} else {
			final Color layerColor = ColorUtil.layerRowBackground();

			// SWT.SetData イベントで TreeItem#setBackground して背景色を設定しようとすると、
			// レイヤーの順序を入れ替えたときに背景色が正しく設定されないことがあるようなので、
			// SWT.EraseItem イベント時に塗りつぶす。
			tree.addListener(SWT.EraseItem, new Listener() {
				public void handleEvent(Event event) {
					if (dragEffect.creatingImage) {
						// WIN32ではここですぐに抜けても特に変化は無いようではあるが、とりあえず抜けておく。
						return;
					}

					TreeItem treeItem = (TreeItem) event.item;
					Object element = treeItem.getData();
					if (element instanceof LayerElement) {
						GC gc = event.gc;
						gc.setBackground(layerColor);
						gc.fillRectangle(event.x, event.y, event.width, event.height);

						// TreeItemの背景色を設定しておかないと、
						// 選択時のみカラムテキストの背景がツリーの背景色になってしまう。
						treeItem.setBackground(layerColor);
					}
				}
			});
		}

		tree.addListener(SWT.PaintItem, new Listener() {
			public void handleEvent(Event event) {
				Object element = event.item.getData();
				if (element instanceof Element) {
					((Element) element).paintColumn(event);
				}
			}
		});

		tree.addListener(SWT.MeasureItem, new Listener() {
			public void handleEvent(Event e) {
				e.height = Math.max(e.height, e.gc.getFontMetrics().getHeight() + ROW_PADDING*2);
			}
		});

		tree.addMouseMoveListener(new MouseMoveListener() {
			public void mouseMove(MouseEvent e) {
				updateCursor(e);
			}
		});

		tree.addMouseListener(new MouseAdapter() {
			public void mouseDown(MouseEvent e) {
				if (COCOA) {
					_mouseDownPoint = new Point(e.x, e.y);
				}
				handleMouseDown(e);
			}
		});

		tree.addKeyListener(new KeyAdapter() {
			public void keyPressed(KeyEvent e) {
				handleKeyPressed(e);
			}
		});

		tree.addPaintListener(new PaintListener() {
			public void paintControl(PaintEvent e) {
				if (_timelineManager != null) {
					_timelineManager.drawTimeIndicatorLine(e);
					if (COCOA) {
						_timelineManager.drawColumnLeftLine(e);
					}
				}
			}
		});
	}

	private TreeViewerColumn createViewerColumn(final int columnIndex, int width, Image image, String name) {
		TreeViewerColumn viewerColumn = new TreeViewerColumn(_treeViewer, SWT.NONE);
		TreeColumn column = viewerColumn.getColumn();
		column.setWidth(width);
		column.setImage(image);
		column.setText(name);
		column.setMoveable(false);

		column.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				treeColumnSelected(columnIndex);
			}
		});

		viewerColumn.setLabelProvider(new TreeColumnViewerLabelProvider(new LayerCompositionLabelProvider()));
		viewerColumn.setEditingSupport(createEditingSupport(columnIndex));

		return viewerColumn;
	}

	private EditingSupport createEditingSupport(final int columnIndex) {
		return new EditingSupport(_treeViewer) {

			protected boolean canEdit(Object element) {
				return (element instanceof Element) ? ((Element) element).canEdit(columnIndex) : false;
			}

			protected CellEditor getCellEditor(Object element) {
				return (element instanceof Element) ? ((Element) element).getCellEditor(columnIndex) : null;
			}
			
			protected Object getValue(Object element) {
				return (element instanceof Element) ? ((Element) element).getCellEditorValue(columnIndex) : null;
			}

			protected void setValue(Object element, Object value) {
				if (element instanceof Element) {
					((Element) element).setCellEditorValue(columnIndex, value);
				}
			}
		};
	}

	private void createSearchText(Composite parent) {
		final TreePath[][] treePaths = new TreePath[1][];

		final Color colorGray = parent.getDisplay().getSystemColor(SWT.COLOR_GRAY);
		final Color colorBlack = parent.getDisplay().getSystemColor(SWT.COLOR_BLACK);

		_searchText = new Text(parent, SWT.SEARCH | SWT.ICON_SEARCH);
		_searchText.setForeground(colorGray);
		_searchText.setText("検索");

		FontUtil.setCompatibleFont(_searchText);

		_searchText.addKeyListener(new KeyAdapter() {
			public void keyPressed(KeyEvent e) {
				switch (e.character) {
					case SWT.CR:
						searchTreeItem();
						break;
					case SWT.ESC:
						if (treePaths[0] != null) {
							_treeViewer.setExpandedTreePaths(treePaths[0]);
						}
						_searchText.setText("");
						break;
				}
			}
		});

		_searchText.addFocusListener(new FocusListener() {
			private String savedText = "";

			public void focusGained(FocusEvent e) {
				treePaths[0] = _treeViewer.getExpandedTreePaths();
				_searchText.setText(savedText);
				_searchText.setForeground(colorBlack);
			}

			public void focusLost(FocusEvent e) {
				treePaths[0] = null;
				savedText = _searchText.getText();
				_searchText.setForeground(colorGray);
				_searchText.setText("検索");
			}
		});
	}

	private void createTimeCodeControls(Composite parent) {
		Display display = parent.getDisplay();

		_timeCodeComposite = new Composite(parent, WIN32 ? SWT.DOUBLE_BUFFERED : SWT.NONE);
		_fpsLabel = new Label(parent, SWT.NONE);

		FontUtil.setCompatibleFont(_timeCodeComposite);
		FontUtil.setCompatibleFont(_fpsLabel);

		final Font font = new Font(display,
				_timeCodeComposite.getFont().getFontData()[0].getName(), COCOA ? 16 : 14, SWT.BOLD);

		_timeCodeComposite.addDisposeListener(new DisposeListener() {
			public void widgetDisposed(DisposeEvent e) {
				font.dispose();
			}
		});

		_timeCodeComposite.setFont(font);
		_timeCodeComposite.setForeground(display.getSystemColor(SWT.COLOR_DARK_BLUE));
		_timeCodeComposite.setLayout(new FillLayout());

		final Text editor = new Text(_timeCodeComposite, SWT.RIGHT);
		editor.setVisible(false);
		editor.setFont(font);
		editor.setBackground(_timeCodeComposite.getBackground());

		final boolean[] editorCanceled = new boolean[1];

		editor.addFocusListener(new FocusAdapter() {
			public void focusLost(FocusEvent e) {
				editor.setVisible(false);
				if (!editorCanceled[0]) {
					Time frameDuration = _composition.getFrameDuration();
					Time time = Time.fromFrameNumber(TimeCode.parseTimeCode(editor.getText(), frameDuration), frameDuration);
					if (time.timeValue < 0) {
						time = new Time(0, time.timeScale);
					} else if (!time.before(_composition.getDuration())) {
						time = _composition.getDuration().subtract(frameDuration);
					}
					time = Time.fromFrameNumber(time.toFrameNumber(frameDuration), frameDuration);
					update(time, null);
				}
			}
		});

		editor.addKeyListener(new KeyAdapter() {
			public void keyPressed(KeyEvent e) {
				switch (e.character) {
					case SWT.ESC:
						editorCanceled[0] = true;
						// fall through
					case SWT.CR:
						// ここで直接 editor.setVisible(false) をすると、ESCを押したときになぜか警告音が鳴る。
						e.display.asyncExec(new Runnable() {
							public void run() {
								editor.setVisible(false);
							}
						});
						break;
				}
			}
		});


		final Rectangle timeCodeArea = new Rectangle(0, 0, 0, 0);
		final Time[] newTime = new Time[1];

		_timeCodeComposite.addPaintListener(new PaintListener() {
			private final int[] underlineDash = new int[] { 2, 2 };

			public void paintControl(PaintEvent e) {
				if (_composition == null) {
					return;
				}

				GC gc = e.gc;

				Time time;
				if (newTime[0] != null) {
					gc.setForeground(e.display.getSystemColor(SWT.COLOR_DARK_RED));
					time = newTime[0];
				} else {
					time = _time;
				}

				String timeCode = TimeCode.toTimeCode(time, _composition.getFrameDuration());
				Point extent = gc.textExtent(timeCode, SWT.DRAW_TRANSPARENT);
				Point size = _timeCodeComposite.getSize();
				int x = size.x - extent.x + (COCOA ? -3 : WIN32 ? -1 : 0);
				int y = (size.y - extent.y) / 2;
				gc.drawString(timeCode, x, y, true);

				timeCodeArea.x = x;
				timeCodeArea.y = y;
				timeCodeArea.width = extent.x;
				timeCodeArea.height = extent.y;

				y += extent.y;
				gc.setLineDash(underlineDash);
				if (COCOA) {
					gc.drawLine(x, y - 1, x + extent.x + 1, y - 1);
				} else if (WIN32) {
					gc.drawLine(x, y, x + extent.x - 3, y);
				} else {
					gc.drawLine(x, y, x + extent.x, y);
				}
			}
		});

		final Cursor handCursor = display.getSystemCursor(SWT.CURSOR_HAND);

		_timeCodeComposite.addMouseMoveListener(new MouseMoveListener() {
			public void mouseMove(MouseEvent e) {
				_timeCodeComposite.setCursor(
						timeCodeArea.contains(e.x, e.y) ? handCursor : null);
			}
		});

		_timeCodeComposite.addMouseListener(new MouseAdapter() {
			private Listener filter;

			public void mouseDown(MouseEvent event) {
				if (!timeCodeArea.contains(event.x, event.y) || filter != null) {
					return;
				}

				final Display display = event.display;
				final long downTime = System.currentTimeMillis();
				final Point downPoint = _timeCodeComposite.toDisplay(event.x, event.y);

				final Time frameDuration = _composition.getFrameDuration();
				final long originalValue = _time.toFrameNumber(frameDuration);

				filter = new Listener() {
					private boolean dragDetected;
					private Point detectPoint;
					private Point prevPoint;
					private double[] currentPoint;

					public void handleEvent(Event e) {
						switch (e.type) {
							case SWT.MouseMove:
								if (!_timeCodeComposite.isDisposed()) {
									if (COCOA && (getCocoaCurrentButtonState() & 1) == 0) {
										break;
									}

									Point pt = display.getCursorLocation();

									if (!dragDetected) {
										dragDetected = (System.currentTimeMillis() - downTime > 100)
												&& (Math.abs(pt.x - downPoint.x) > 3 || Math.abs(pt.y - downPoint.y) > 3);
										if (dragDetected) {
											currentPoint = new double[] { pt.x, pt.y };
											prevPoint = detectPoint = pt;
										}
									}

									if (dragDetected) {
										double deltaScale = (e.stateMask & SWT.MOD2) != 0 ? 10 :
															(e.stateMask & SWT.MOD1) != 0 ? 0.1 : 1;
										currentPoint[0] += (pt.x - prevPoint.x) * deltaScale;
										currentPoint[1] += (pt.y - prevPoint.y) * deltaScale;

										prevPoint = pt;

										long newValue = (long)(originalValue + currentPoint[0] - detectPoint.x);

										Time time = Time.fromFrameNumber(newValue, frameDuration);
										if (time.timeValue < 0) {
											time = new Time(0, time.timeScale);
										} else if (!time.before(_composition.getDuration())) {
											time = _composition.getDuration().subtract(frameDuration);
										}
										newTime[0] = Time.fromFrameNumber(time.toFrameNumber(frameDuration), frameDuration);

										update(newTime[0], Boolean.TRUE);
										_timeCodeComposite.redraw();
									}

									break;
								}
								// fall through

							case SWT.MouseUp:
							case SWT.Deactivate:
								if (newTime[0] != null /* && !_timeCodeComposite.isDisposed() */) {		// TODO disposeされてるかどうかのチェックは必要？
									update(newTime[0], Boolean.FALSE);									//      必要だとしても、スクラブ終了のイベントが送出されなくなるのはまずい。
								}

								filter = null;

								display.removeFilter(SWT.MouseMove, this);
								display.removeFilter(SWT.MouseUp, this);
								display.removeFilter(SWT.Deactivate, this);
								_timelineManager.setWrapMode(true);
								newTime[0] = null;

								if (!_timeCodeComposite.isDisposed()) {
									if (e.type == SWT.MouseUp && !dragDetected) {
										editorCanceled[0] = false;
										editor.setText(TimeCode.toTimeCode(_time, _composition.getFrameDuration()));
										editor.selectAll();
										editor.setVisible(true);
										if (!editor.setFocus()) {
											editor.setVisible(false);
										}
									} else {
										_timeCodeComposite.redraw();
									}
								}
								break;
						}
					}
				};

				_timelineManager.setWrapMode(false);
				display.addFilter(SWT.MouseMove, filter);
				display.addFilter(SWT.MouseUp, filter);
				display.addFilter(SWT.Deactivate, filter);
			}
		});
	}

	// TODO getCocoaCurrentButtonState は AnimatableValueElementDelegate などにも同じものがある。

	private static Method cocoaGetCurrentButtonStateMethod;

	private static int getCocoaCurrentButtonState() {
		try {
			if (cocoaGetCurrentButtonStateMethod == null) {
				Class<?> clazz = Class.forName("org.eclipse.swt.internal.cocoa.OS");
				cocoaGetCurrentButtonStateMethod = clazz.getMethod("GetCurrentButtonState");
			}
			return (Integer) cocoaGetCurrentButtonStateMethod.invoke(null);
		} catch (RuntimeException e) {
			throw e;
		} catch (Exception e) {
			throw new JavieRuntimeException(e);
		}
	}

	public void updateDeleteActionState() {
		if (_deleteAction != null) {
			_deleteAction.setEnabled(_timelineManager.hasKeyframeSelection()
									|| canRemoveTreeSelection());
		}
	}

	private void setupGlobalActionsAndContextMenu() {
		_deleteAction = new Action("削除") {
			public void run() {
				if (_timelineManager.hasKeyframeSelection()) {
					_timelineManager.removeSelectedKeyframes();
				} else {
					removeTreeSelection();
				}
			}
		};
		_deleteAction.setEnabled(false);
		_treeViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				updateDeleteActionState();
			}
		});

		Action duplicateAction = new Action("複製") {
			{
				setEnabled(false);
				_treeViewer.addSelectionChangedListener(new ISelectionChangedListener() {
					public void selectionChanged(SelectionChangedEvent event) {
						setEnabled(canDuplicateTreeSelection());
					}
				});
			}
			public void run() {
				duplicateTreeSelection();
			}
		};

		Action newLightLayerAction = new Action("ライト") {
			public void run() {
				handleNewLightLayerAction();
			}
		};

		Action newCameraLayerAction = new Action("カメラ") {
			public void run() {
				handleNewCameraLayerAction();
			}
		};

		Action newNullLayerAction = new Action("ヌルオブジェクト") {
			public void run() {
				handleNewNullLayerAction();
			}
		};

		Action newTextLayerAction = new Action("テキスト") {
			public void run() {
				handleNewTextLayerAction();
			}
		};

		_splitLayersAction = new Action("レイヤーの分割") {
			{
				setEnabled(false);
				_treeViewer.addSelectionChangedListener(new ISelectionChangedListener() {
					public void selectionChanged(SelectionChangedEvent event) {
						setEnabled(canSplitSelectedLayers());
					}
				});
			}
			public void run() {
				splitSelectedLayers();
			}
		};


		// GlobalActions

		IViewSite site = getViewSite();
		IActionBars actionBars = site.getActionBars();
		IWorkbenchWindow window = site.getWorkbenchWindow();

		actionBars.setGlobalActionHandler("delete", _deleteAction);
		actionBars.setGlobalActionHandler(CommandIds.DUPLICATE, duplicateAction);
		actionBars.setGlobalActionHandler(CommandIds.SPLIT_LAYERS, _splitLayersAction);

		actionBars.setGlobalActionHandler(CommandIds.NEW_LIGHT_LAYER, newLightLayerAction);
		actionBars.setGlobalActionHandler(CommandIds.NEW_CAMERA_LAYER, newCameraLayerAction);
		actionBars.setGlobalActionHandler(CommandIds.NEW_NULL_LAYER, newNullLayerAction);
		actionBars.setGlobalActionHandler(CommandIds.NEW_TEXT_LAYER, newTextLayerAction);
		if (WindowsDirectShowOutput.AVAILABLE) {
			actionBars.setGlobalActionHandler(CommandIds.AVI_OUTPUT, new AVIOutputAction(this));
		}
		if (MacOSXQTMovieOutput.AVAILABLE) {
			actionBars.setGlobalActionHandler(CommandIds.QTMOVIE_OUTPUT, new QTMovieOutputAction(this));
		}
		actionBars.setGlobalActionHandler(CommandIds.SEQUENCE_OUTPUT, new SequenceOutputAction(this));
		actionBars.setGlobalActionHandler(CommandIds.WAVE_OUTPUT, new WaveOutputAction(this));
		actionBars.setGlobalActionHandler(CommandIds.JAVIE2MMD, new Javie2MMDAction(this));

		List<AddEffectAction> effectActions = Util.newList();
		for (EffectDescriptor ed : _effectRegistry.getEffectDescriptors()) {
			AddEffectAction action = new AddEffectAction(window, _treeViewer, ed);
			actionBars.setGlobalActionHandler("EFFECT." + ed.getType(), action);
			effectActions.add(action);
		}


		// ContextMenu

		MenuManager menuMgr = new MenuManager();
		MenuManager newMenu = new MenuManager("新規");
		MenuManager timeMenu = new MenuManager("時間");
		MenuManager effectMenu = new MenuManager("エフェクト");
		MenuManager animeMenu = new MenuManager("アニメーション");
		MenuManager interpolationMenu = new MenuManager("キーフレーム補間法");

		menuMgr.add(newMenu);
		menuMgr.add(timeMenu);
		menuMgr.add(effectMenu);
		menuMgr.add(animeMenu);
		menuMgr.add(new Separator());
		menuMgr.add(new RenameLayerEtcAction(this, _treeViewer));
		menuMgr.add(_deleteAction);
		menuMgr.add(duplicateAction);
		menuMgr.add(_splitLayersAction);
		menuMgr.add(new ReplaceLayerItemAction(this, _treeViewer));
		menuMgr.add(new PrecomposeAction(this, _treeViewer));
		menuMgr.add(new ResetPropertiesAction(window, _treeViewer));
		menuMgr.add(new GroupMarker(IWorkbenchActionConstants.MB_ADDITIONS));

		newMenu.add(newLightLayerAction);
		newMenu.add(newCameraLayerAction);
		newMenu.add(newNullLayerAction);
		newMenu.add(newTextLayerAction);
		timeMenu.add(new TimeRemapAction(window, _treeViewer));
		timeMenu.add(new ReverseLayerTimeAction(_treeViewer));
		timeMenu.add(new TimeStretchLayerAction(_treeViewer));
		setupEffectContextMenu(effectMenu, effectActions);
		animeMenu.add(interpolationMenu);
		animeMenu.add(new RemoveKeyframesAction(_treeViewer));
		animeMenu.add(new AddExpressionsAction(window, _treeViewer));
		interpolationMenu.add(new KeyframeInterpolationAction(_treeViewer, Interpolation.HOLD));
		interpolationMenu.add(new KeyframeInterpolationAction(_treeViewer, Interpolation.LINEAR));
		interpolationMenu.add(new KeyframeInterpolationAction(_treeViewer, Interpolation.CATMULL_ROM));

		site.registerContextMenu(menuMgr, _treeViewer);

		Control treeViewControl = _treeViewer.getControl();
		Menu treeViewMenu = menuMgr.createContextMenu(treeViewControl);
		treeViewControl.setMenu(treeViewMenu);
	}

	private void setupEffectContextMenu(MenuManager effectMenu, List<AddEffectAction> effectActions) {
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IExtensionPoint point = registry.getExtensionPoint("ch.kuramo.javie.api.effectCategory");
		if (point == null) {
			return;
		}

		Map<String, MenuManager> subMenuMap = Util.newMap();
		List<MenuManager> subMenuList = Util.newList();

		for (IExtension ext : point.getExtensions()) {
			for (IConfigurationElement cfgElem : ext.getConfigurationElements()) {
				if ("effect-category".equals(cfgElem.getName())) {
					String category = cfgElem.getAttribute("category");
					String name = cfgElem.getAttribute("name").replaceAll("&", "&&");

					MenuManager subMenu = new MenuManager(name);
					subMenuMap.put(category, subMenu);
					subMenuList.add(subMenu);
				}
			}
		}

		Collections.sort(subMenuList, new Comparator<MenuManager>() {
			public int compare(MenuManager o1, MenuManager o2) {
				return o1.getMenuText().compareTo(o2.getMenuText());
			}
		});

		for (MenuManager subMenu : subMenuList) {
			effectMenu.add(subMenu);
		}

		List<Action> actions = Util.newList();

		for (AddEffectAction action : effectActions) {
			MenuManager subMenu = subMenuMap.get(action.getEffectDescriptor().getCategory());
			if (subMenu != null) {
				subMenu.add(action);
			} else {
				actions.add(action);
			}
		}

		Collections.sort(actions, new Comparator<Action>() {
			public int compare(Action o1, Action o2) {
				return o1.getText().compareTo(o2.getText());
			}
		});

		for (Action action : actions) {
			effectMenu.add(action);
		}
	}

	private void setupDragAndDrop() {
		final LocalSelectionTransfer selectionTransfer = LocalSelectionTransfer.getTransfer();

		DragSourceListener sourceListener = new DragSourceListener() {
			public void dragStart(DragSourceEvent event) {
				if (!canDrag(event)) {
					event.doit = false;
					return;
				}

				ISelection selection = new LocalSelectionWrapper(_treeViewer);
				if (selection.isEmpty()) {
					event.doit = false;
				} else {
					selectionTransfer.setSelection(selection);
					selectionTransfer.setSelectionSetTime(event.time & 0xFFFFFFFFL);
					event.doit = true;
				}
			}

			public void dragSetData(DragSourceEvent event) {
				event.data = selectionTransfer.getSelection();
			}

			public void dragFinished(DragSourceEvent event) {
				selectionTransfer.setSelection(null);
				selectionTransfer.setSelectionSetTime(0);
			}
		};

		DropTargetListener targetListener = new DropTargetAdapter() {

			private <T extends Element> T getTargetElement(DropTargetEvent event, Class<T> elementClass) {
				TreeItem treeItem = (TreeItem) event.item;
				if (treeItem != null) {
					Object data = treeItem.getData();
					if (elementClass.isInstance(data)) {
						@SuppressWarnings("unchecked")
						T element = (T) data;
						return element;
					}
				}
				return null;
			}

			private int getTargetFeedback(DropTargetEvent event, Class<? extends Element> elementClass) {
				if (event.item != null && getTargetElement(event, elementClass) != null) {
					TreeItem treeItem = (TreeItem) event.item;
					Point pt = treeItem.getParent().toControl(event.x, event.y);
					Rectangle b = treeItem.getBounds();
					return (pt.y <= b.y + b.height/2) ? DND.FEEDBACK_INSERT_BEFORE : DND.FEEDBACK_INSERT_AFTER;
				}
				return DND.FEEDBACK_NONE;
			}

			private List<LayerElement> getSourceLayerElements(DropTargetEvent event) {
				List<LayerElement> list = Util.newList();
				ITreeSelection selection = LocalSelectionWrapper.getSelection(selectionTransfer, _treeViewer);
				if (selection != null) {
					for (Object o : selection.toList()) {
						if (o instanceof LayerElement) {
							list.add((LayerElement) o);
						} else {
							return null;
						}
					}
				}
				return list.isEmpty() ? null : list;
			}

			private List<Item> getSourceItems(DropTargetEvent event) {
				List<Item> list = Util.newList();
				ITreeSelection selection = LocalSelectionWrapper.getSelection(selectionTransfer);
				if (selection != null) {
					for (Object o : selection.toList()) {
						if (o instanceof Folder) {
							// TODO Folderを再帰的に調べる。
						}
						if (o instanceof MediaItem) {
							list.add((MediaItem) o);
						} else {
							return null;
						}
					}
				}
				return list.isEmpty() ? null : list;
			}

			private List<EffectElement> getSourceEffectElements(DropTargetEvent event) {
				List<EffectElement> list = Util.newList();
				ITreeSelection selection = LocalSelectionWrapper.getSelection(selectionTransfer, _treeViewer);
				if (selection != null) {
					for (Object o : selection.toList()) {
						if (o instanceof EffectElement) {
							list.add((EffectElement) o);
						} else {
							return null;
						}
					}
				}
				return list.isEmpty() ? null : list;
			}

			private List<TextAnimatorElement> getSourceTextAnimatorElements(DropTargetEvent event) {
				List<TextAnimatorElement> list = Util.newList();
				ITreeSelection selection = LocalSelectionWrapper.getSelection(selectionTransfer, _treeViewer);
				if (selection != null) {
					for (Object o : selection.toList()) {
						if (o instanceof TextAnimatorElement) {
							list.add((TextAnimatorElement) o);
						} else {
							return null;
						}
					}
				}
				return list.isEmpty() ? null : list;
			}

			private List<TASelectorElement<?>> getSourceTASelectorElements(DropTargetEvent event) {
				List<TASelectorElement<?>> list = Util.newList();
				ITreeSelection selection = LocalSelectionWrapper.getSelection(selectionTransfer, _treeViewer);
				if (selection != null) {
					for (Object o : selection.toList()) {
						if (o instanceof TASelectorElement<?>) {
							list.add((TASelectorElement<?>) o);
						} else {
							return null;
						}
					}
				}
				return list.isEmpty() ? null : list;
			}

			public void dragOver(DropTargetEvent event) {
				int operations = event.operations;
				if (WIN32) {
					try {
						Class<?> os = Class.forName("org.eclipse.swt.internal.win32.OS");
						Method m = os.getMethod("GetKeyState", int.class);
						short keyState = (Short)m.invoke(null, 0x11);	// 0x11=OS.VK_CONTROL
						operations &= ((keyState < 0) ? DND.DROP_COPY : DND.DROP_MOVE) | DND.DROP_LINK;
					} catch (Exception e) {
						_logger.warn("error getting key state", e);
					}
				}

				if (!selectionTransfer.isSupportedType(event.currentDataType)) {
					event.detail = DND.DROP_NONE;
					event.feedback = DND.FEEDBACK_NONE;
					return;
				}

				if (getSourceLayerElements(event) != null) {
					if (event.item != null && getTargetElement(event, LayerElement.class) == null) {
						event.detail = DND.DROP_NONE;
						event.feedback = DND.FEEDBACK_NONE;
					} else {
						event.detail = ((operations & DND.DROP_MOVE) == DND.DROP_MOVE)
								? DND.DROP_MOVE : (operations & DND.DROP_COPY);
						event.feedback = (event.feedback | getTargetFeedback(event, LayerElement.class))
									& ~DND.FEEDBACK_SELECT & ~DND.FEEDBACK_EXPAND;
					}
					return;
				}

				if (getSourceItems(event) != null) {
					if (event.item != null && getTargetElement(event, LayerElement.class) == null) {
						event.detail = DND.DROP_NONE;
						event.feedback = DND.FEEDBACK_NONE;
					} else {
						event.detail = operations & DND.DROP_LINK;
						event.feedback = (event.feedback | getTargetFeedback(event, LayerElement.class))
									& ~DND.FEEDBACK_SELECT & ~DND.FEEDBACK_EXPAND;
					}
					return;
				}

				if (getSourceEffectElements(event) != null && event.item != null) {
					if (getTargetElement(event, EffectsElement.class) != null) {
						event.detail = ((operations & DND.DROP_MOVE) == DND.DROP_MOVE)
								? DND.DROP_MOVE : (operations & DND.DROP_COPY);
						event.feedback = (event.feedback | DND.FEEDBACK_SELECT)
								& ~DND.FEEDBACK_EXPAND;

					} else if (getTargetElement(event, EffectElement.class) != null) {
						event.detail = ((operations & DND.DROP_MOVE) == DND.DROP_MOVE)
								? DND.DROP_MOVE : (operations & DND.DROP_COPY);
						event.feedback = (event.feedback | getTargetFeedback(event, EffectElement.class))
								& ~DND.FEEDBACK_SELECT & ~DND.FEEDBACK_EXPAND;

					} else {
						event.detail = DND.DROP_NONE;
						event.feedback = DND.FEEDBACK_NONE;
					}
					return;
				}

				if (getSourceTextAnimatorElements(event) != null && event.item != null) {
					if (getTargetElement(event, TextAnimatorsElement.class) != null) {
						event.detail = ((operations & DND.DROP_MOVE) == DND.DROP_MOVE)
								? DND.DROP_MOVE : (operations & DND.DROP_COPY);
						event.feedback = (event.feedback | DND.FEEDBACK_SELECT)
								& ~DND.FEEDBACK_EXPAND;

					} else if (getTargetElement(event, TextAnimatorElement.class) != null) {
						event.detail = ((operations & DND.DROP_MOVE) == DND.DROP_MOVE)
								? DND.DROP_MOVE : (operations & DND.DROP_COPY);
						event.feedback = (event.feedback | getTargetFeedback(event, TextAnimatorElement.class))
								& ~DND.FEEDBACK_SELECT & ~DND.FEEDBACK_EXPAND;
						
					} else {
						event.detail = DND.DROP_NONE;
						event.feedback = DND.FEEDBACK_NONE;
					}
					return;
				}

				if (getSourceTASelectorElements(event) != null && event.item != null) {
					if (getTargetElement(event, TextAnimatorElement.class) != null) {
						event.detail = ((operations & DND.DROP_MOVE) == DND.DROP_MOVE)
								? DND.DROP_MOVE : (operations & DND.DROP_COPY);
						event.feedback = (event.feedback | DND.FEEDBACK_SELECT)
								& ~DND.FEEDBACK_EXPAND;

					} else if (getTargetElement(event, TASelectorElement.class) != null) {
						event.detail = ((operations & DND.DROP_MOVE) == DND.DROP_MOVE)
								? DND.DROP_MOVE : (operations & DND.DROP_COPY);
						event.feedback = (event.feedback | getTargetFeedback(event, TASelectorElement.class))
								& ~DND.FEEDBACK_SELECT & ~DND.FEEDBACK_EXPAND;

					} else {
						event.detail = DND.DROP_NONE;
						event.feedback = DND.FEEDBACK_NONE;
					}
					return;
				}

				event.detail = DND.DROP_NONE;
				event.feedback = DND.FEEDBACK_NONE;
			}

			public void dropAccept(DropTargetEvent event) {
				dragOver(event);
			}

			public void drop(DropTargetEvent event) {
				List<LayerElement> sourceLayerElements = getSourceLayerElements(event);
				if (sourceLayerElements != null) {
					if (event.detail == DND.DROP_COPY) {
						duplicateLayers(sourceLayerElements, getTargetElement(event, LayerElement.class),
										(getTargetFeedback(event, LayerElement.class) == DND.FEEDBACK_INSERT_AFTER));
					} else {
						moveLayers(sourceLayerElements, getTargetElement(event, LayerElement.class),
										(getTargetFeedback(event, LayerElement.class) == DND.FEEDBACK_INSERT_AFTER));
					}
					return;
				}

				List<Item> sourceItems = getSourceItems(event);
				if (sourceItems != null) {
					newLayers(sourceItems, getTargetElement(event, LayerElement.class),
								(getTargetFeedback(event, LayerElement.class) == DND.FEEDBACK_INSERT_AFTER));
					return;
				}

				List<EffectElement> sourceEffectElements = getSourceEffectElements(event);
				if (sourceEffectElements != null) {
					int targetIndex = -1;
					EffectsElement effectsElement = getTargetElement(event, EffectsElement.class);
					if (effectsElement == null) {
						EffectElement ee = getTargetElement(event, EffectElement.class);
						if (ee != null) {
							effectsElement = (EffectsElement) ee.parent;
							LayerElement le = (LayerElement) effectsElement.parent;
							targetIndex = ((EffectableLayer) le.layer).getEffects().indexOf(ee.effect);
							if (getTargetFeedback(event, EffectElement.class) == DND.FEEDBACK_INSERT_AFTER) {
								++targetIndex;
							}
						}
					}
					if (effectsElement != null) {
						EffectableLayer dstLayer = (EffectableLayer) ((LayerElement) effectsElement.parent).layer;
						if (event.detail == DND.DROP_COPY) {
							duplicateEffects(sourceEffectElements, dstLayer, targetIndex);
						} else {
							moveEffects(sourceEffectElements, dstLayer, targetIndex);
						}
						return;
					}
				}

				List<TextAnimatorElement> sourceTextAnimatorElements = getSourceTextAnimatorElements(event);
				if (sourceTextAnimatorElements != null) {
					int targetIndex = -1;
					TextAnimatorsElement textAnimatorsElement = getTargetElement(event, TextAnimatorsElement.class);
					if (textAnimatorsElement == null) {
						TextAnimatorElement tae = getTargetElement(event, TextAnimatorElement.class);
						if (tae != null) {
							textAnimatorsElement = (TextAnimatorsElement) tae.parent;
							LayerElement le = (LayerElement) textAnimatorsElement.parent.parent;
							targetIndex = ((TextLayer) le.layer).getTextAnimators().indexOf(tae.animator);
							if (getTargetFeedback(event, TextAnimatorElement.class) == DND.FEEDBACK_INSERT_AFTER) {
								++targetIndex;
							}
						}
					}
					if (textAnimatorsElement != null) {
						TextLayer dstLayer = (TextLayer) ((LayerElement) textAnimatorsElement.parent.parent).layer;
						if (event.detail == DND.DROP_COPY) {
							duplicateTextAnimators(sourceTextAnimatorElements, dstLayer, targetIndex);
						} else {
							moveTextAnimators(sourceTextAnimatorElements, dstLayer, targetIndex);
						}
						return;
					}
				}

				List<TASelectorElement<?>> sourceTASelectorElements = getSourceTASelectorElements(event);
				if (sourceTASelectorElements != null) {
					int targetIndex = -1;
					TextAnimatorElement textAnimatorElement = getTargetElement(event, TextAnimatorElement.class);
					if (textAnimatorElement == null) {
						TASelectorElement<?> tase = getTargetElement(event, TASelectorElement.class);
						if (tase != null) {
							textAnimatorElement = (TextAnimatorElement) tase.parent;
							targetIndex = textAnimatorElement.animator.getSelectors().indexOf(tase.selector);
							if (getTargetFeedback(event, TASelectorElement.class) == DND.FEEDBACK_INSERT_AFTER) {
								++targetIndex;
							}
						}
					}
					if (textAnimatorElement != null) {
						TextLayer dstLayer = (TextLayer) ((LayerElement) textAnimatorElement.parent.parent.parent).layer;
						if (event.detail == DND.DROP_COPY) {
							duplicateTASelectors(sourceTASelectorElements, dstLayer, textAnimatorElement.animator, targetIndex);
						} else {
							moveTASelectors(sourceTASelectorElements, dstLayer, textAnimatorElement.animator, targetIndex);
						}
						return;
					}
				}

				event.detail = DND.DROP_NONE;
			}
		};

		_treeViewer.addDragSupport(DND.DROP_COPY | DND.DROP_MOVE, new Transfer[] { selectionTransfer }, sourceListener);
		_treeViewer.addDropSupport(DND.DROP_COPY | DND.DROP_MOVE | DND.DROP_LINK, new Transfer[] { selectionTransfer }, targetListener);
	}

	private void setLayoutData() {
		FormData data;

		data = new FormData();
		data.left = new FormAttachment(0, COCOA ? 4 : 6);
		data.right = new FormAttachment(0, 200);
		data.top = new FormAttachment(0, COCOA ? 4 : 6);
		data.bottom = new FormAttachment(0, COCOA ? 26 : 24);
		_searchText.setLayoutData(data);

		data = new FormData();
		data.left = new FormAttachment(_searchText, 10, SWT.RIGHT);
		data.right = new FormAttachment(_searchText, 150, SWT.RIGHT);
		data.top = new FormAttachment(0, 5);
		data.bottom = new FormAttachment(0, 25);
		_timeCodeComposite.setLayoutData(data);

		data = new FormData();
		data.left = new FormAttachment(_timeCodeComposite, 10, SWT.RIGHT);
		data.right = new FormAttachment(_timeCodeComposite, 90, SWT.RIGHT);
		data.top = new FormAttachment(0, COCOA ? 7 : 9);
		data.bottom = new FormAttachment(0, COCOA ? 23 : 21);
		_fpsLabel.setLayoutData(data);

		data = new FormData();
		data.left = new FormAttachment(_fpsLabel, 0, SWT.RIGHT);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(0, 12);
		data.bottom = new FormAttachment(0, 30);
		_topSpacer.setLayoutData(data);

		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(100, COCOA ? -15 : -16);
		data.bottom = new FormAttachment(100, 0);
		_bottomSpacer.setLayoutData(data);

		data = new FormData();
		data.left = new FormAttachment(_topSpacer, 0, SWT.RIGHT);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(0, 12);
		data.bottom = new FormAttachment(0, 30);
		_timelineMeter.setLayoutData(data);

		data = new FormData();
		data.left = new FormAttachment(_bottomSpacer, 0, SWT.RIGHT);
		data.right = new FormAttachment(_bottomSpacer, 150, SWT.RIGHT);
		data.top = new FormAttachment(100, COCOA ? -15 : -27);
		data.bottom = new FormAttachment(100, 0);
		_timelineScale.setLayoutData(data);

		data = new FormData();
		data.left = new FormAttachment(_timelineScale, 0, SWT.RIGHT);
		data.right = new FormAttachment(100, COCOA ? -15 : -16);
		data.top = new FormAttachment(100, COCOA ? -15 : -16);
		data.bottom = new FormAttachment(100, 0);
		_timelineSlider.setLayoutData(data);

		data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(_topSpacer, 0, SWT.BOTTOM);
		data.bottom = new FormAttachment(_timelineSlider, 0, SWT.TOP);
		_treeViewer.getTree().setLayoutData(data);
	}

	private void treeColumnSelected(int columnIndex) {
		switch (columnIndex) {
			case NAME_COL:
				setLayerNameMode(!isLayerNameMode());
				break;
		}
	}

	private boolean isLayerNameMode() {
		return Boolean.TRUE.equals(_treeViewer.getData(LAYER_NAME_MODE));
	}

	private void setLayerNameMode(boolean layerNameMode) {
		if (isLayerNameMode() != layerNameMode) {
			_treeViewer.setData(LAYER_NAME_MODE, layerNameMode);
			_nameColumn.getColumn().setText(layerNameMode ? "レイヤー名" : "ソース名");
			_treeViewer.refresh();
			redrawLayerReferences();
		}
	}

	public void editElementName(Element element) {
		if (element instanceof LayerElement
				|| element instanceof EffectElement
				|| element instanceof TextAnimatorElement
				|| element instanceof TASelectorElement<?>) {
			if (element instanceof LayerElement) {
				setLayerNameMode(true);
			}
			_treeViewer.setData(EDIT_ELEMENT_NAME, Boolean.TRUE);
			_treeViewer.editElement(element, NAME_COL);
			_treeViewer.setData(EDIT_ELEMENT_NAME, null);
		} else {
			throw new IllegalArgumentException(
					"element must be LayerElement, EffectElement, TextAnimatorElement or TASelectorElement");
		}
	}

	private static final Transliterator H2K_TRANSLITERATOR = Transliterator.getInstance("Hiragana-Katakana");
	private static final Pattern SPACE_PATTERN = Pattern.compile("\\s");

	private void searchTreeItem() {
		String query = _searchText.getText();
		query = Normalizer.normalize(query, Normalizer.NFKC);
		query = H2K_TRANSLITERATOR.transliterate(query);
		query = SPACE_PATTERN.matcher(query).replaceAll("").toLowerCase();

		Tree tree = _treeViewer.getTree();
		TreeItem found = null;

		if (query.length() != 0) {
			TreeItem item = null;

			TreeSelection selection = (TreeSelection) _treeViewer.getSelection();
			Object element = selection.getFirstElement();
			if (element != null) {
				for (TreeItem ti : tree.getSelection()) {
					if (ti.getData() == element) {
						item = ti;
						break;
					}
				}
			}

			if (item != null) {
				found = searchTreeItem(query, item, false);

				while (found == null && item != null) {
					TreeItem parentItem = item.getParentItem();
					List<TreeItem> items = Arrays.asList(
							parentItem != null ? parentItem.getItems() : tree.getItems());

					int index = items.indexOf(item);
					for (int i = index + 1; found == null && i < items.size(); ++i) {
						found = searchTreeItem(query, items.get(i), true);
					}
					item = parentItem;
				}
			}

			if (found == null) {
				for (TreeItem ti : tree.getItems()) {
					found = searchTreeItem(query, ti, true);
					if (found != null) {
						break;
					}
				}
			}
		}

		selectAndReveal(found);
		if (found == null) {
			tree.getDisplay().beep();
		}
	}

	private TreeItem searchTreeItem(String query, TreeItem fromItem, boolean includeFromItem) {
		if (includeFromItem) {
			String text = fromItem.getText();
			text = Normalizer.normalize(text, Normalizer.NFKC);
			text = H2K_TRANSLITERATOR.transliterate(text);
			text = SPACE_PATTERN.matcher(text).replaceAll("").toLowerCase();
			if (text.contains(query)) {
				return fromItem;
			}
		}

		Object element = fromItem.getData();
		if (_treeViewer.isExpandable(element)) {
			boolean expanded = _treeViewer.getExpandedState(element);
			_treeViewer.getTree().setRedraw(false);
			try {
				_treeViewer.setExpandedState(element, true);

				for (TreeItem ti: fromItem.getItems()) {
					ti = searchTreeItem(query, ti, true);
					if (ti != null) {
						return ti;
					}
				}
			} finally {
				_treeViewer.setExpandedState(element, expanded);
				_treeViewer.getTree().setRedraw(true);
			}
		}

		return null;
	}

	private void adjustColumnWidth() {
		Tree tree = _treeViewer.getTree();

		int clientWidth = tree.getClientArea().width;
		if (clientWidth == 0) {
			return;
		}

		TreeColumn[] columns = tree.getColumns();
		int totalColumnWidth = 0;
		for (TreeColumn column : columns) {
			totalColumnWidth += column.getWidth();
		}

		TreeColumn lastColumn = columns[columns.length-1];
		int lastColumnWidth = lastColumn.getWidth() + clientWidth - totalColumnWidth;
		if (lastColumnWidth > 0) {
			lastColumn.setWidth(lastColumnWidth);
		}

		FormData data1 = (FormData) _topSpacer.getLayoutData();
		FormData data2 = (FormData) _bottomSpacer.getLayoutData();
		data1.right.numerator = data2.right.numerator = 0;
		data1.right.offset = data2.right.offset = clientWidth - lastColumnWidth;

		final Composite parent = _bottomSpacer.getParent();
		parent.getDisplay().asyncExec(new Runnable() {
			public void run() {
				if (!parent.isDisposed()) {
					parent.layout(new Control[] { _topSpacer, _bottomSpacer });
				}
			}
		});
	}

	private void updateCursor(MouseEvent e) {
		Point pt = new Point(e.x, e.y);
		TreeItem item = getItem(pt);
		if (item != null) {
			Object element = item.getData();
			if (element instanceof Element) {
				for (int i = 0, cc = _treeViewer.getTree().getColumnCount(); i < cc; ++i) {
					if (item.getBounds(i).contains(pt)) {
						((Element) element).updateCursor(e, i);
						return;
					}
				}
			}
		}
		_treeViewer.getTree().setCursor(null);
	}

	private void setupActionBars() {
		IActionBars actionBars = getViewSite().getActionBars();
		IToolBarManager tbm = actionBars.getToolBarManager();

		tbm.add(new LayerCompShyAction(this));
		tbm.add(new Separator());
		tbm.add(new LayerCompFrameBlendAction(this));
		tbm.add(new LayerCompMotionBlurAction(this));
		tbm.add(new LayerCompSuperSamplingAction(this));
		tbm.add(new Separator());
		tbm.add(new LayerCompSettingsAction(this));

		actionBars.updateActionBars();
	}

	private TreeItem _shiftSelectionBaseItem;

	private void emulateFullSelectionMouseDown(MouseEvent e, TreeItem item, int columnIndex) {
		final Tree tree = _treeViewer.getTree();
		Collection<TreeItem> treeItems = null;

		if ((e.stateMask & SWT.MOD1) != 0) {
			treeItems = Util.newSet(Arrays.asList(tree.getSelection()));
			if (treeItems.contains(item)) {
				treeItems.remove(item);
			} else {
				treeItems.add(item);
			}
			_shiftSelectionBaseItem = item;

		} else if ((e.stateMask & SWT.MOD2) != 0 && _shiftSelectionBaseItem != null) {
			List<TreeItem> list = new Object() {
				final List<TreeItem> items = Util.newList();
				List<TreeItem> getVisibleExpandedItems() {
					getVisibleExpandedItems(tree.getItems());
					return items;
				}
				void getVisibleExpandedItems(TreeItem[] treeItems) {
					for (TreeItem ti : treeItems) {
						items.add(ti);
						if (ti.getExpanded()) {
							getVisibleExpandedItems(ti.getItems());
						}
					}
				}
			}.getVisibleExpandedItems();

			int i1 = list.indexOf(_shiftSelectionBaseItem);
			int i2 = list.indexOf(item);
			if (i1 != -1 && i2 != -1) {
				treeItems = list.subList(Math.min(i1, i2), Math.max(i1, i2)+1);
			}
		} else if (columnIndex == TIMELINE_COL
				&& (e.stateMask & SWT.BUTTON_MASK) == 0
				&& Arrays.asList(tree.getSelection()).contains(item)) {

			// フルセレクションをエミュレートするだけならこれは不要だが、
			// デュレーションバー上でのドラッグなどを適切に処理するためには
			// ここで抜けたほうが都合が良い。
			return;
		}

		if (treeItems == null) {
			treeItems = Collections.singleton(item);
			_shiftSelectionBaseItem = item;
		}

		if (columnIndex != NAME_COL || !item.getBounds().contains(new Point(e.x, e.y))) {
			selectAndReveal(treeItems);
		}
	}

	private void handleMouseDown(MouseEvent e) {
		Point pt = new Point(e.x, e.y);
		TreeItem item = getItem(pt);
		if (item != null) {
			Object element = item.getData();
			if (element instanceof Element) {
				for (int i = 0, cc = _treeViewer.getTree().getColumnCount(); i < cc; ++i) {
					if (item.getBounds(i).contains(pt)) {
						if (WIN32) {
							emulateFullSelectionMouseDown(e, item, i);
						}
						((Element) element).mouseDown(e, i);
						return;
					}
				}
			}
		}
		_timelineManager.mouseDown(e);
	}

	private void handleKeyPressed(KeyEvent e) {
		// DELキーは削除のショートカットに割り当てられてるので、ここでは処理しない。
		if (e.stateMask == 0 && (/*e.keyCode == SWT.DEL ||*/ (COCOA && e.keyCode == SWT.BS))) {
			if (_timelineManager.hasKeyframeSelection()) {
				_timelineManager.removeSelectedKeyframes();
			} else {
				removeTreeSelection();
			}
		}
	}

	private boolean canRemoveTreeSelection() {
		return canRemoveOrDuplicateTreeSelection(
				Util.<LayerElement>newSet(),
				Util.<EffectElement>newSet(),
				Util.<TextAnimatorElement>newSet(),
				Util.<TASelectorElement<?>>newSet(),
				Util.<LayerAnimatableValueElement>newSet());
	}

	private boolean canDuplicateTreeSelection() {
		return canRemoveOrDuplicateTreeSelection(
				Util.<LayerElement>newSet(),
				Util.<EffectElement>newSet(),
				Util.<TextAnimatorElement>newSet(),
				Util.<TASelectorElement<?>>newSet(),
				null);
	}

	private boolean canRemoveOrDuplicateTreeSelection(
			Set<LayerElement> layerElements,
			Set<EffectElement> effectElements,
			Set<TextAnimatorElement> textAnimatorElements,
			Set<TASelectorElement<?>> taSelectorElements,
			Set<LayerAnimatableValueElement> taPropertyElements) {

		layerElements.clear();
		effectElements.clear();
		textAnimatorElements.clear();
		taSelectorElements.clear();

		if (taPropertyElements != null) {
			taPropertyElements.clear();
		}

		for (Object o : ((TreeSelection) _treeViewer.getSelection()).toList()) {
			if (o instanceof LayerElement) {
				layerElements.add((LayerElement) o);
			} else if (o instanceof EffectElement) {
				effectElements.add((EffectElement) o);
			} else if (o instanceof EffectsElement) {
				for (Element child : ((EffectsElement) o).getChildren()) {
					effectElements.add((EffectElement) child);
				}
			} else if (o instanceof TextAnimatorElement) {
				textAnimatorElements.add((TextAnimatorElement) o);
			} else if (o instanceof TextAnimatorsElement) {
				for (Element child : ((TextAnimatorsElement) o).getChildren()) {
					if (child instanceof TextAnimatorElement) {
						textAnimatorElements.add((TextAnimatorElement) child);
					}
				}
			} else if (o instanceof TASelectorElement<?>) {
				taSelectorElements.add((TASelectorElement<?>) o);

			} else if (taPropertyElements != null
					&& o instanceof LayerAnimatableValueElement
					&& ((LayerAnimatableValueElement) o).parent instanceof TextAnimatorElement) {

				taPropertyElements.add((LayerAnimatableValueElement) o);

			} else {
				return false;
			}
		}

		int notEmpty = (layerElements.isEmpty() ? 0 : 1)
					 + (effectElements.isEmpty() ? 0 : 1)
					 + (textAnimatorElements.isEmpty() ? 0 : 1)
					 + (taSelectorElements.isEmpty() ? 0 : 1)
					 + (taPropertyElements == null || taPropertyElements.isEmpty() ? 0 : 1);
		return (notEmpty == 1);
	}

	private void removeTreeSelection() {
		Set<LayerElement> layerElements = Util.newSet();
		Set<EffectElement> effectElements = Util.newSet();
		Set<TextAnimatorElement> textAnimatorElements = Util.newSet();
		Set<TASelectorElement<?>> taSelectorElements = Util.newSet();
		Set<LayerAnimatableValueElement> taPropertyElements = Util.newSet();

		if (canRemoveOrDuplicateTreeSelection(layerElements, effectElements,
				textAnimatorElements, taSelectorElements, taPropertyElements)) {

			if (!taPropertyElements.isEmpty()) {
				List<Object[]> properties = Util.newList();
				for (LayerAnimatableValueElement lave : taPropertyElements) {
					String property = lave.getProperty();
					properties.add(new Object[] {
							((LayerElement) lave.parent.parent.parent.parent).layer,
							((TextAnimatorElement) lave.parent).animator,
							property.substring(property.lastIndexOf('.') + 1)
					});
				}
				_projectManager.postOperation(new RemoveTAPropertiesOperation(_projectManager, properties));

			} else if (!taSelectorElements.isEmpty()) {
				List<Object[]> selectors = Util.newList();
				for (TASelectorElement<?> tase : taSelectorElements) {
					selectors.add(new Object[] {
							((LayerElement) tase.parent.parent.parent.parent).layer,
							((TextAnimatorElement)tase.parent).animator,
							tase.selector
					});
				}
				_projectManager.postOperation(new RemoveTASelectorsOperation(_projectManager, selectors));

			} else if (!textAnimatorElements.isEmpty()) {
				List<Object[]> animators = Util.newList();
				for (TextAnimatorElement tae : textAnimatorElements) {
					animators.add(new Object[] { ((LayerElement) tae.parent.parent.parent).layer, tae.animator });
				}
				_projectManager.postOperation(new RemoveTextAnimatorsOperation(_projectManager, animators));

			} else if (!effectElements.isEmpty()) {
				List<Object[]> effects = Util.newList();
				for (EffectElement ee : effectElements) {
					effects.add(new Object[] { ((LayerElement) ee.parent.parent).layer, ee.effect });
				}
				_projectManager.postOperation(new RemoveEffectsOperation(_projectManager, effects));
			} else if (!layerElements.isEmpty()) {
				List<Layer> layers = Util.newList();
				for (LayerElement le : layerElements) {
					layers.add(le.layer);
				}
				_projectManager.postOperation(new RemoveLayersOperation(_projectManager, layers));
			}
		}
	}

	private void duplicateTreeSelection() {
		Set<LayerElement> layerElements = Util.newSet();
		Set<EffectElement> effectElements = Util.newSet();
		Set<TextAnimatorElement> textAnimatorElements = Util.newSet();
		Set<TASelectorElement<?>> taSelectorElements = Util.newSet();

		if (canRemoveOrDuplicateTreeSelection(layerElements, effectElements,
				textAnimatorElements, taSelectorElements, null)) {

			if (!taSelectorElements.isEmpty()) {
				duplicateTASelectors(taSelectorElements);
			} else if (!textAnimatorElements.isEmpty()) {
				duplicateTextAnimators(textAnimatorElements);
			} else if (!effectElements.isEmpty()) {
				duplicateEffects(effectElements);
			} else {
				duplicateLayers(layerElements); 
			}
		}
	}

	private boolean canSplitSelectedLayers() {
		return canSplitSelectedLayers(Util.<LayerElement>newSet(), new Time[1]);
	}

	private boolean canSplitSelectedLayers(Set<LayerElement> layerElements, Time[] time) {
		layerElements.clear();

		Time t = _time;

		for (Object o : ((TreeSelection) _treeViewer.getSelection()).toList()) {
			if (o instanceof LayerElement) {
				Layer layer = ((LayerElement) o).layer;
				if (t.after(layer.getInPoint()) && t.before(layer.getOutPoint())) {
					layerElements.add((LayerElement) o);
				}
			}
		}

		if (layerElements.isEmpty()) {
			return false;
		} else {
			time[0] = t;
			return true;
		}
	}

	private void splitSelectedLayers() {
		Set<LayerElement> layerElements = Util.newSet();
		Time[] time = new Time[1];

		if (canSplitSelectedLayers(layerElements, time)) {
			Set<Layer> layers = Util.newSet();
			for (LayerElement e : layerElements) {
				layers.add(e.layer);
			}
			_projectManager.postOperation(new DuplicateLayersOperation(_projectManager, layers, time[0]));
		}
	}

	private boolean canDrag(DragSourceEvent e) {
		// CocoaではDragSourceEventのx,yの値がマウスをクリックした座標ではなく
		// ドラッグ検出時の座標になるようなので、mouseDownイベントで取得しておいた座標を使う。
		Point pt = (COCOA && _mouseDownPoint != null) ? _mouseDownPoint : new Point(e.x, e.y);
		TreeItem item = getItem(pt);
		if (item == null || !item.getBounds(NAME_COL).contains(pt)) {
			return false;
		}


		// 選択中の行が全てレイヤー、全てエフェクト、もしくは全てテキストアニメータの場合のみドラッグ可

		TreeSelection selection = (TreeSelection) _treeViewer.getSelection();
		if (selection.isEmpty()) {
			return false;
		}

		Object first = selection.getFirstElement();
		if (first instanceof LayerElement) {
			for (Object o : selection.toList()) {
				if (!(o instanceof LayerElement)) return false;
			}
			return true;

		} else if (first instanceof EffectElement) {
			for (Object o : selection.toList()) {
				if (!(o instanceof EffectElement)) return false;
			}
			return true;

		} else if (first instanceof TextAnimatorElement) {
			for (Object o : selection.toList()) {
				if (!(o instanceof TextAnimatorElement)) return false;
			}
			return true;

		} else if (first instanceof TASelectorElement<?>) {
			for (Object o : selection.toList()) {
				if (!(o instanceof TASelectorElement<?>)) return false;
			}
			return true;

		} else {
			return false;
		}
	}

	private TreeItem getItem(Point pt) {
		return getItem(pt, _treeViewer.getTree().getItems());
	}

	private TreeItem getItem(Point pt, TreeItem[] items) {
		// Tree#getItem(Point) はユーザーのクリックによって選択が生じるポイントでしかアイテムを返さないため、
		// TreeItemを順番にスキャンする必要がある。(Cocoaの場合は常にFULL_SELECTIONなので、Tree#getItem(Point)でもOK)
		// また、TreeItem#getBounds() も選択が生じる領域を返すようである(Windowsのみ？)。
		// 
		// なお、各階層において末尾のアイテムから先頭に向かって処理する方が無駄なスキャンをしなくて済む。

		for (int i = items.length-1; i >= 0; --i) {
			TreeItem item = items[i];
			Rectangle bounds = item.getBounds();
			if (bounds.y <= pt.y) {
				if (bounds.y + bounds.height > pt.y) {
					return item;
				} else if (item.getExpanded()) {
					return getItem(pt, item.getItems());
				}
			}
		}
		return null;
	}

	private List<LayerElement> getLayerElements() {
		List<LayerElement> list = Util.newList();
		LayerCompositionContentProvider cp = (LayerCompositionContentProvider) _treeViewer.getContentProvider();
		for (Object element : cp.getElements(_composition)) {
			list.add((LayerElement) element);
		}
		return list;
	}

	private void moveLayers(List<LayerElement> layerElements, LayerElement targetElement, boolean insertAfter) {
		List<LayerElement> current = getLayerElements();
		int moveToIndex;
		if (targetElement == null) {
			moveToIndex = current.size();
		} else {
			moveToIndex = current.indexOf(targetElement);
			if (moveToIndex == -1) {
				throw new IllegalArgumentException("no such layerElement found");
			}
			if (insertAfter) {
				++moveToIndex;
			}
		}

		// サブリストを並行して更新すると ConcurrentModificationException が発生するので複製して使う。
		List<LayerElement> frontSub = Util.newList(current.subList(0, moveToIndex));
		List<LayerElement> backSub = Util.newList(current.subList(moveToIndex, current.size()));

		frontSub.removeAll(layerElements);
		backSub.removeAll(layerElements);

		List<LayerElement> newOrderOfElements = Util.newList();
		newOrderOfElements.addAll(frontSub);
		newOrderOfElements.addAll(layerElements);
		newOrderOfElements.addAll(backSub);

		// Treeの表示順は内部データと逆順。
		Collections.reverse(newOrderOfElements);

		List<Layer> newOrder = Util.newList();
		for (LayerElement e : newOrderOfElements) {
			newOrder.add(e.layer);
		}

		List<Layer> layers = Util.newList();
		for (LayerElement e : layerElements) {
			layers.add(e.layer);
		}

		_projectManager.postOperation(new ReorderLayersOperation(_projectManager, newOrder, layers));
	}

	private void duplicateLayers(Collection<LayerElement> layerElements) {
		duplicateLayers(layerElements, -1);
	}

	private void duplicateLayers(Collection<LayerElement> layerElements, LayerElement targetElement, boolean insertAfter) {
		if (targetElement != null) {
			int insertIndex = _composition.getLayers().indexOf(targetElement.layer);
			// Treeの表示順は内部データと逆順なので insertAfter の場合はそのまま、そうでない場合は１加える必要がある。
			if (!insertAfter) {
				++insertIndex;
			}
			duplicateLayers(layerElements, insertIndex);
		} else {
			duplicateLayers(layerElements, 0);
		}
	}

	private void duplicateLayers(Collection<LayerElement> layerElements, int insertIndex) {
		Set<Layer> layers = Util.newSet();
		for (LayerElement e : layerElements) {
			layers.add(e.layer);
		}
		_projectManager.postOperation(new DuplicateLayersOperation(_projectManager, layers, insertIndex));
	}

	private void newLayers(List<Item> items, LayerElement targetElement, boolean insertAfter) {
		List<LayerElement> current = getLayerElements();
		int insertIndex;
		if (targetElement == null) {
			insertIndex = current.size();
		} else {
			insertIndex = current.indexOf(targetElement);
			if (insertIndex == -1) {
				throw new IllegalArgumentException("no such layerElement found");
			}
			if (insertAfter) {
				++insertIndex;
			}
		}

		// Treeの表示順は内部データと逆順。
		insertIndex = current.size() - insertIndex;

		items = Util.newList(items);
		Collections.reverse(items);

		_projectManager.postOperation(new NewLayerFromItemOperation(_projectManager, _composition, insertIndex, items));
	}

	private void moveEffects(List<EffectElement> effectElements, EffectableLayer dstLayer, int moveBefore) {
		List<EffectableLayer> srcLayers = Util.newList();
		List<Effect> effects = Util.newList();

		for (EffectElement ee : effectElements) {
			srcLayers.add((EffectableLayer) ((LayerElement) ee.parent.parent).layer);
			effects.add(ee.effect);
		}

		_projectManager.postOperation(new MoveEffectsOperation(_projectManager, srcLayers, effects, dstLayer, moveBefore));
	}

	private void duplicateEffects(Collection<EffectElement> effectElements) {
		duplicateEffects(effectElements, null, 0);
	}

	private void duplicateEffects(Collection<EffectElement> effectElements, EffectableLayer dstLayer, int insertBefore) {
		List<EffectableLayer> srcLayers = Util.newList();
		List<Effect> effects = Util.newList();

		for (EffectElement ee : effectElements) {
			srcLayers.add((EffectableLayer) ((LayerElement) ee.parent.parent).layer);
			effects.add(ee.effect);
		}

		_projectManager.postOperation(new DuplicateEffectsOperation(_projectManager, srcLayers, effects, dstLayer, insertBefore));
	}

	private void moveTextAnimators(List<TextAnimatorElement> textAnimatorElements, TextLayer dstLayer, int moveBefore) {
		List<Object[]> animators = Util.newList();

		for (TextAnimatorElement tae : textAnimatorElements) {
			TextLayer layer = (TextLayer) ((LayerElement) tae.parent.parent.parent).layer;
			animators.add(new Object[] { layer, tae.animator });
		}

		_projectManager.postOperation(new MoveTextAnimatorsOperation(_projectManager, animators, dstLayer, moveBefore));
	}

	private void duplicateTextAnimators(Collection<TextAnimatorElement> textAnimatorElements) {
		duplicateTextAnimators(textAnimatorElements, null, 0);
	}

	private void duplicateTextAnimators(Collection<TextAnimatorElement> textAnimatorElements, TextLayer dstLayer, int insertBefore) {
		List<Object[]> animators = Util.newList();

		for (TextAnimatorElement tae : textAnimatorElements) {
			TextLayer layer = (TextLayer) ((LayerElement) tae.parent.parent.parent).layer;
			animators.add(new Object[] { layer, tae.animator });
		}

		_projectManager.postOperation(new DuplicateTextAnimatorsOperation(_projectManager, animators, dstLayer, insertBefore));
	}

	private void moveTASelectors(List<TASelectorElement<?>> taSelectorElements,
			TextLayer dstLayer, TextAnimator dstAnimator, int moveBefore) {

		List<Object[]> selectors = Util.newList();

		for (TASelectorElement<?> tase : taSelectorElements) {
			TextLayer layer = (TextLayer) ((LayerElement) tase.parent.parent.parent.parent).layer;
			TextAnimator animator = ((TextAnimatorElement)tase.parent).animator;
			selectors.add(new Object[] { layer, animator, tase.selector });
		}

		_projectManager.postOperation(new MoveTASelectorsOperation(
				_projectManager, selectors, dstLayer, dstAnimator, moveBefore));
	}

	private void duplicateTASelectors(Collection<TASelectorElement<?>> taSelectorElements) {
		duplicateTASelectors(taSelectorElements, null, null, 0);
	}

	private void duplicateTASelectors(Collection<TASelectorElement<?>> taSelectorElements,
			TextLayer dstLayer, TextAnimator dstAnimator, int insertBefore) {

		List<Object[]> selectors = Util.newList();

		for (TASelectorElement<?> tase : taSelectorElements) {
			TextLayer layer = (TextLayer) ((LayerElement) tase.parent.parent.parent.parent).layer;
			TextAnimator animator = ((TextAnimatorElement)tase.parent).animator;
			selectors.add(new Object[] { layer, animator, tase.selector });
		}

		_projectManager.postOperation(new DuplicateTASelectorsOperation(
				_projectManager, selectors, dstLayer, dstAnimator, insertBefore));
	}

	private void collectAnimatableValues() {
		collectAnimatableValues(_timelineManager.getCurrentTime());
	}

	private void collectAnimatableValues(Time time) {
		_collector.reset();
		_collector.setTime(time);

		// TODO prepareExpressionはプロジェクトに構造的な変更があった場合のみ行えばよい。
		_composition.prepareExpression(_collector.createInitialExpressionScope(_composition));

		_treeViewer.setData(ANIMATABLE_VALUES, _collector.collect());
		updateProperties();
	}

	private void updateProperties() {
		updateProperties(_treeViewer.getTree().getItems());
	}

	private void updateProperties(TreeItem[] treeItems) {
		for (TreeItem item : treeItems) {
			Object element = item.getData();
			if (element instanceof AnimatableValueElement) {
				AnimatableValueElement avalueElement = (AnimatableValueElement) element;
				if (avalueElement.updateValue()) {
					Rectangle bounds = item.getBounds(VALUE_COL);
					_treeViewer.getTree().redraw(bounds.x, bounds.y, bounds.width, bounds.height, false);
				}
			}
			if (item.getExpanded()) {
				updateProperties(item.getItems());
			}
		}
	}

	private void redrawLayerReferences() {
		redrawLayerReferences(_treeViewer.getTree().getItems());
	}

	private void redrawLayerReferences(TreeItem[] treeItems) {
		for (TreeItem item : treeItems) {
			Object element = item.getData();
			if (element instanceof AnimatableValueElement
					&& ((AnimatableValueElement) element).getAnimatableValue() instanceof AnimatableLayerReference) {
				Rectangle bounds = item.getBounds(VALUE_COL);
				_treeViewer.getTree().redraw(bounds.x, bounds.y, bounds.width, bounds.height, false);
			}
			if (item.getExpanded()) {
				redrawLayerReferences(item.getItems());
			}
		}
	}

	private TreeItem findLayerPropertyItem(TreeItem layerItem, final String property) {
		return new Object() {
			TreeItem find(TreeItem treeItem) {
				Object element = treeItem.getData();
				if (element instanceof LayerPropertyElement<?>
						&& ((LayerPropertyElement<?>) element).getProperty().equals(property)) {
					return treeItem;

				} else if (treeItem.getItemCount() > 0) {
					boolean expanded = _treeViewer.getExpandedState(element);
					_treeViewer.getTree().setRedraw(false);
					try {
						_treeViewer.setExpandedState(element, true);

						for (TreeItem ti : treeItem.getItems()) {
							ti = find(ti);
							if (ti != null) {
								return ti;
							}
						}
						return null;

					} finally {
						_treeViewer.setExpandedState(element, expanded);
						_treeViewer.getTree().setRedraw(true);
					}
				} else {
					return null;
				}
			}
		}.find(layerItem);
	}

	private TreeItem findLayerAnimatableValueItem(TreeItem layerItem, final String property) {
		return new Object() {
			TreeItem find(TreeItem treeItem) {
				Object element = treeItem.getData();
				if (element instanceof LayerAnimatableValueElement
						&& ((LayerAnimatableValueElement) element).getProperty().equals(property)) {
					return treeItem;

				} else if (treeItem.getItemCount() > 0) {
					boolean expanded = _treeViewer.getExpandedState(element);
					_treeViewer.getTree().setRedraw(false);
					try {
						_treeViewer.setExpandedState(element, true);

						for (TreeItem ti : treeItem.getItems()) {
							ti = find(ti);
							if (ti != null) {
								return ti;
							}
						}
						return null;

					} finally {
						_treeViewer.setExpandedState(element, expanded);
						_treeViewer.getTree().setRedraw(true);
					}
				} else {
					return null;
				}
			}
		}.find(layerItem);
	}

	private TreeItem findEffectAnimatableValueItem(TreeItem layerItem, final Effect effect, final String property) {
		return new Object() {
			TreeItem find(TreeItem treeItem) {
				Object element = treeItem.getData();
				if (element instanceof EffectAnimatableValueElement
						&& ((EffectAnimatableValueElement) element).getProperty().equals(property)) {
					return treeItem;

				} else if ((element instanceof LayerElement
						|| element instanceof EffectsElement
						|| element instanceof EffectElement) && treeItem.getItemCount() > 0) {

					if (element instanceof EffectElement && ((EffectElement) element).effect != effect) {
						return null;
					}

					boolean expanded = _treeViewer.getExpandedState(element);
					_treeViewer.getTree().setRedraw(false);
					try {
						_treeViewer.setExpandedState(element, true);

						for (TreeItem ti : treeItem.getItems()) {
							ti = find(ti);
							if (ti != null) {
								return ti;
							}
						}
						return null;

					} finally {
						_treeViewer.setExpandedState(element, expanded);
						_treeViewer.getTree().setRedraw(true);
					}
				} else {
					return null;
				}
			}
		}.find(layerItem);
	}

	private TreeItem findEffectItem(TreeItem layerItem, final Effect effect) {
		return new Object() {
			TreeItem find(TreeItem treeItem) {
				Object element = treeItem.getData();
				if (element instanceof EffectElement) {
					return (((EffectElement) element).effect == effect) ? treeItem : null;

				} else if ((element instanceof LayerElement
						|| element instanceof EffectsElement) && treeItem.getItemCount() > 0) {

					boolean expanded = _treeViewer.getExpandedState(element);
					_treeViewer.getTree().setRedraw(false);
					try {
						_treeViewer.setExpandedState(element, true);

						for (TreeItem ti : treeItem.getItems()) {
							ti = find(ti);
							if (ti != null) {
								return ti;
							}
						}
						return null;

					} finally {
						_treeViewer.setExpandedState(element, expanded);
						_treeViewer.getTree().setRedraw(true);
					}
				} else {
					return null;
				}
			}
		}.find(layerItem);
	}

	private TreeItem findTextAnimatorsItem(TreeItem layerItem) {
		return new Object() {
			TreeItem find(TreeItem treeItem) {
				Object element = treeItem.getData();
				if (element instanceof TextAnimatorsElement) {
					return treeItem;

				} else if ((element instanceof LayerElement
						|| element instanceof TextElement) && treeItem.getItemCount() > 0) {

					boolean expanded = _treeViewer.getExpandedState(element);
					_treeViewer.getTree().setRedraw(false);
					try {
						_treeViewer.setExpandedState(element, true);

						for (TreeItem ti : treeItem.getItems()) {
							ti = find(ti);
							if (ti != null) {
								return ti;
							}
						}
						return null;

					} finally {
						_treeViewer.setExpandedState(element, expanded);
						_treeViewer.getTree().setRedraw(true);
					}
				} else {
					return null;
				}
			}
		}.find(layerItem);
	}

	private TreeItem findTextAnimatorItem(TreeItem layerItem, TextAnimator animator) {
		TreeItem treeItem = findTextAnimatorsItem(layerItem);
		if (treeItem != null) {
			Object element = treeItem.getData();

			boolean expanded = _treeViewer.getExpandedState(element);
			_treeViewer.getTree().setRedraw(false);
			try {
				_treeViewer.setExpandedState(element, true);

				for (TreeItem ti : treeItem.getItems()) {
					Object data = ti.getData();
					if (data instanceof TextAnimatorElement
							&& ((TextAnimatorElement) data).animator == animator) {
						return ti;
					}
				}
			} finally {
				_treeViewer.setExpandedState(element, expanded);
				_treeViewer.getTree().setRedraw(true);
			}
		}
		return null;
	}

	private TreeItem findTASelectorItem(TreeItem layerItem, final TASelector selector) {
		return new Object() {
			TreeItem find(TreeItem treeItem) {
				Object element = treeItem.getData();
				if (element instanceof TASelectorElement<?>
						&& ((TASelectorElement<?>) element).selector == selector) {
					return treeItem;

				} else if ((element instanceof LayerElement
						|| element instanceof TextElement
						|| element instanceof TextAnimatorsElement
						|| element instanceof TextAnimatorElement) && treeItem.getItemCount() > 0) {

					boolean expanded = _treeViewer.getExpandedState(element);
					_treeViewer.getTree().setRedraw(false);
					try {
						_treeViewer.setExpandedState(element, true);

						for (TreeItem ti : treeItem.getItems()) {
							ti = find(ti);
							if (ti != null) {
								return ti;
							}
						}
						return null;

					} finally {
						_treeViewer.setExpandedState(element, expanded);
						_treeViewer.getTree().setRedraw(true);
					}
				} else {
					return null;
				}
			}
		}.find(layerItem);
	}

	private void selectAndReveal(TreeItem treeItem) {
		if (treeItem == null) {
			_treeViewer.setSelection(null);
			return;
		}

		selectAndReveal(Collections.singleton(treeItem));
	}

	private void selectAndReveal(Collection<TreeItem> treeItems) {
		if (treeItems == null || treeItems.isEmpty()) {
			_treeViewer.setSelection(null);
			return;
		}

		List<TreePath> list = Util.newList();
		for (TreeItem treeItem : treeItems) {
			LinkedList<Object> segmentList = Util.newLinkedList();
			do {
				segmentList.addFirst(treeItem.getData());
			} while ((treeItem = treeItem.getParentItem()) != null);
			list.add(new TreePath(segmentList.toArray()));
		}
		_treeViewer.setSelection(new TreeSelection(list.toArray(new TreePath[list.size()])), true);
	}

	private void handleCompositionSettingsChange(ProjectEvent event) {
		if (event.composition != _composition) {
			return;
		}

		// 今のところ、ここでやることは無い。
	}

	private void handleCompositionPropertyChange(ProjectEvent event) {
		if (event.composition != _composition) {
			return;
		}

		if (event.property.equals("shyEnabled")) {
			if (_composition.isShyEnabled()) {
				// TODO ここでキーフレームの選択状態を解除するのは暫定的な措置。
				//      そうしないと handleKeyframesChange と handleLayerSlipEdit が正常に動作しない。
				//      選択状態を維持したままシャイを有効にできるようにするか、
				//      シャイで非表示になったレイヤー内のキーフレームだけ選択解除するようにするべき。
				_timelineManager.clearKeyframeSelection();

				_treeViewer.addFilter(ShyFilter.INSTANCE);
			} else {
				_treeViewer.removeFilter(ShyFilter.INSTANCE);
			}
		}
	}

	private void handleLayerPropertyChange(ProjectEvent event) {
		if (event.composition != _composition) {
			return;
		}

		if (event.property.equals("shy") && _composition.isShyEnabled()) {
			_treeViewer.refresh();
			return;
		}

		Tree tree = _treeViewer.getTree();
		TreeItem[] treeItems = tree.getItems();

		if (event.property.equals("name")) {
			Set<Element> elements = Util.newSet();
			for (TreeItem treeItem : treeItems) {
				LayerElement layerElement = (LayerElement) treeItem.getData();
				if (_composition.getParentLayer(layerElement.layer) == event.layer) {
					elements.add(layerElement);
				}
			}
			if (!elements.isEmpty()) {
				_treeViewer.update(elements.toArray(), null);
			}

			redrawLayerReferences();
		}

		for (TreeItem treeItem : treeItems) {
			LayerElement layerElement = (LayerElement) treeItem.getData();
			if (layerElement.layer == event.layer) {
				if (ProjectEvent.propertyIsAnimatable(event)) {
					// ソース名表示の場合、テキストレイヤーはソーステキストをソース名に表示する。
					if (layerElement.layer instanceof TextLayer && event.property.equals("sourceText") && !isLayerNameMode()) {
						_treeViewer.update(treeItem.getData(), null);
					}

					treeItem = findLayerAnimatableValueItem(treeItem, event.property);

				} else {
					if (layerElement.layer instanceof LightLayer && event.property.equals("lightType")) {
						// アンビエントの時はトランスフォームを非表示にするため、
						// トランスフォームの親のlayerElementを更新する必要がある。
						_treeViewer.refresh(layerElement, true);

						treeItem = findLayerPropertyItem(treeItem, "lightType");
						//_treeViewer.refresh(treeItem.getData(), true); // すぐ上でlayerElementを更新しているので不要

					} else if (layerElement.layer instanceof LightLayer && event.property.equals("castsShadows")) {
						treeItem = findLayerPropertyItem(treeItem, "castsShadows");

					} else if (layerElement.layer instanceof MediaLayer
							&& (event.property.equals("castsShadows")
									|| event.property.equals("acceptsShadows")
									|| event.property.equals("acceptsLights"))) {

						treeItem = findLayerPropertyItem(treeItem, event.property);

					} else if (layerElement.layer instanceof TextLayer && event.property.equals("textType")) {
						treeItem = findLayerPropertyItem(treeItem, "textType");
						_treeViewer.refresh(treeItem.getData(), true);

					} else if (layerElement.layer instanceof TextLayer && event.property.equals("perCharacter3D")) {
						_treeViewer.refresh(findTextAnimatorsItem(treeItem).getData(), false);

					} else if (layerElement.layer instanceof TextLayer && event.property.startsWith("textAnimators[")) {
						boolean unknown = true;

						if (event.property.endsWith(".enabled") || event.property.endsWith(".name")) {
							Object obj = PropertyUtil.getProperty(layerElement.layer,
									event.property.substring(0, event.property.lastIndexOf('.')));
							if (obj instanceof TextAnimator) {
								treeItem = findTextAnimatorItem(treeItem, (TextAnimator) obj);
								_treeViewer.update(treeItem.getData(), null);
								unknown = false;
							} else if (obj instanceof TASelector) {
								treeItem = findTASelectorItem(treeItem, (TASelector) obj);
								_treeViewer.update(treeItem.getData(), null);
								unknown = false;
							}
						} else if ((event.property.endsWith(".base") || event.property.endsWith(".shape")) // <=無駄に正規表現にかけなくて済むよう、先に末尾だけチェック
								&& event.property.matches("^textAnimators\\[\\d+\\]\\.selectors\\[\\d+\\]\\.(base|shape)$")) {
							treeItem = findLayerPropertyItem(treeItem, event.property);
							unknown = false;
						}

						if (unknown) {
							_logger.warn("unknown property of TextLayer: " + event.property);
						}

					} else if (layerElement.layer instanceof TransformableLayer && event.property.equals("threeD")) {
						_treeViewer.refresh(layerElement, true);
					} else {
						_treeViewer.update(layerElement, null);
					}

					// 表示／非表示のスイッチなどは独自に描画しているため、redraw もする必要がある。
					// また、win32 だと TreeItem#getBounds() が返す Rectangle ではダメ。

					Rectangle bounds = treeItem.getBounds();
					if (WIN32) {
						Rectangle clientArea = tree.getClientArea();
						bounds.x = clientArea.x;
						bounds.width = clientArea.width;
					}
					tree.redraw(bounds.x, bounds.y, bounds.width, bounds.height, false);
				}

				selectAndReveal(treeItem);
				break;
			}
		}

		collectAnimatableValues();
	}

	private void handleLayerExpressionChange(ProjectEvent event) {
		if (event.composition != _composition) {
			return;
		}

		for (TreeItem treeItem : _treeViewer.getTree().getItems()) {
			LayerElement layerElement = (LayerElement) treeItem.getData();
			if (layerElement.layer == event.layer) {
				treeItem = findLayerAnimatableValueItem(treeItem, event.property);
				if (treeItem != null) {
					Element element = (Element) treeItem.getData();

					// これをしなくても見た目はかわならいが、しないとundo/redo時にswtがクラッシュすることがある？
					_treeViewer.setExpandedState(element, false);

					_treeViewer.refresh(element, false);
					if (treeItem.getItemCount() > 0) {
						_treeViewer.setExpandedState(element, true);
						treeItem = treeItem.getItem(0);
					}
					selectAndReveal(treeItem);
				}
				break;
			}
		}

		collectAnimatableValues();
	}

	private void handleLayerItemChange(ProjectEvent event) {
		if (event.composition != _composition) {
			return;
		}

		_treeViewer.refresh();

		for (TreeItem treeItem : _treeViewer.getTree().getItems()) {
			LayerElement layerElement = (LayerElement) treeItem.getData();
			if (layerElement.layer == event.layer) {
				selectAndReveal(treeItem);
				break;
			}
		}

		collectAnimatableValues();
	}

	private void handleLayersAddRemoveReorder(ProjectEvent event) {
		if (event.composition != _composition) {
			return;
		}

		_treeViewer.getTree().setRedraw(false);
		try {
			// NOTE この時点ではモデルとTreeの内容が一致していないので、
			//      refreshの前にgetExpandedElementsを呼ぶのは安全ではないかもしれない。
			Object[] expandedElements = _treeViewer.getExpandedElements();
			_treeViewer.refresh();
			_treeViewer.setExpandedElements(expandedElements);

			if (event.type == Type.LAYERS_ADD || event.type == Type.LAYERS_REORDER) {
				Set<Layer> layers = ProjectEvent.getLayers(event);
				Set<TreeItem> treeItems = Util.newSet();
				for (TreeItem treeItem : _treeViewer.getTree().getItems()) {
					LayerElement layerElement = (LayerElement) treeItem.getData();
					if (layers.contains(layerElement.layer)) {
						treeItems.add(treeItem);
					}
				}
				selectAndReveal(treeItems);
			}

		} finally {
			_treeViewer.getTree().setRedraw(true);
		}

		// 追加削除により選択状態が変化するため、_splitLayersAction の状態も更新されるはずだが、念のため。
		_splitLayersAction.setEnabled(canSplitSelectedLayers());

		collectAnimatableValues();
	}

	private void handleItemUpdate(ProjectEvent event, boolean checkParent) {
		Item item = ProjectEvent.getItem(event);
		if (item == _compItem) {
			setPartName(_compItem.getName());
			return;	// コンポ内に自分自身を含んでいることはないので、これ以降の処理は不要。
		}

		Set<Element> elements = Util.newSet();

		for (TreeItem treeItem : _treeViewer.getTree().getItems()) {
			LayerElement element = (LayerElement) treeItem.getData();

			if (element.layer instanceof ItemLayer<?>) {
				ItemLayer<?> itemLayer = (ItemLayer<?>) element.layer;
				if (itemLayer.getItem() == item) {
					elements.add(element);
				}
			}

			if (checkParent) {
				Layer parentLayer = _composition.getParentLayer(element.layer);
				if (parentLayer instanceof ItemLayer<?>) {
					if (((ItemLayer<?>) parentLayer).getItem() == item) {
						elements.add(element);
					}
				}
			}
		}

		if (!elements.isEmpty()) {
			_treeViewer.update(elements.toArray(), null);
		}
	}

	private void handleExpressionsAddRemove(ProjectEvent event) {
		if (event.composition != _composition) {
			return;
		}

		boolean add = (event.type == Type.EXPRESSIONS_ADD);

		Map<Layer, TreeItem> layerToTreeItem = Util.newMap();
		for (TreeItem treeItem : _treeViewer.getTree().getItems()) {
			LayerElement element = (LayerElement) treeItem.getData();
			layerToTreeItem.put(element.layer, treeItem);
		}

		Set<TreeItem> treeItems = Util.newSet();

		for (Object[] o : (Object[][]) event.data) {
			TreeItem treeItem = layerToTreeItem.get(o[0]);

			if (o[1] instanceof String) {
				String property = (String) o[1];
				treeItem = findLayerAnimatableValueItem(treeItem, property);
			} else if (o[1] instanceof Integer) {
				EffectableLayer layer = (EffectableLayer) o[0];
				Effect effect = layer.getEffects().get((Integer) o[1]);
				String property = (String) o[2];
				treeItem = findEffectAnimatableValueItem(treeItem, effect, property);
			} else {
				// never reach here
				throw new Error();
			}

			if (treeItem == null) {
				// never reach here ???
				continue;
			}

			// handleLayerExpressionChange, handleEffectExpressionChange で
			// Elementの初期化は行われているので、ここでは TreeView#refresh など何もせずに
			// treeItem.getItem(0) を呼び出して大丈夫なはず。
			if (add) {
				treeItem = treeItem.getItem(0);
			}
			treeItems.add(treeItem);
		}

		selectAndReveal(treeItems);
	}

	private void handleTextAnimatorsAddRemove(ProjectEvent event) {
		if (event.composition != _composition) {
			return;
		}

		// 現在の選択状態を先に解除した方が見栄えがいい。解除しなくても機能的には問題ない。
		selectAndReveal((TreeItem) null);


		boolean add = (event.type == Type.TEXT_ANIMATORS_ADD);

		Map<Layer, TreeItem> layerToTreeItem = Util.newMap();
		for (TreeItem treeItem : _treeViewer.getTree().getItems()) {
			LayerElement element = (LayerElement) treeItem.getData();
			layerToTreeItem.put(element.layer, treeItem);
		}

		Set<Object> refreshElements = Util.newSet();
		for (Object[] o : (Object[][]) event.data) {
			TreeItem layerItem = layerToTreeItem.get(o[0]);
			LayerElement layerElement = (LayerElement) layerItem.getData();

			TreeItem textAnimatorsItem = findTextAnimatorsItem(layerItem);
			TextAnimatorsElement textAnimatorsElement = (TextAnimatorsElement) textAnimatorsItem.getData();

			if (((TextLayer) layerElement.layer).getTextAnimators().isEmpty()) {
				_treeViewer.setExpandedState(textAnimatorsElement, false);
			}
			refreshElements.add(textAnimatorsElement);
		}
		for (Object o : refreshElements) {
			_treeViewer.refresh(o);
		}


		Set<TreeItem> treeItems = Util.newSet();

		for (Object[] o : (Object[][]) event.data) {
			TreeItem layerItem = layerToTreeItem.get(o[0]);

			if (add) {
				LayerElement layerElement = (LayerElement) layerItem.getData();
				Integer animatorIndex = (Integer) o[1];
				TextAnimator animator = ((TextLayer) layerElement.layer).getTextAnimators().get(animatorIndex);

				// 追加後は追加したテキストアニメータを選択する。
				TreeItem textAnimatorItem = findTextAnimatorItem(layerItem, animator);
				treeItems.add(textAnimatorItem);

				if ((o.length == 3) && "new".equals(o[2])) {
					_treeViewer.setExpandedState(textAnimatorItem.getData(), true);
					for (TreeItem item : textAnimatorItem.getItems()) {
						treeItems.add(item);
					}
				}
			} else {
				// 削除後は TextAnimatorsElement を選択する。
				treeItems.add(findTextAnimatorsItem(layerItem));
			}
		}

		selectAndReveal(treeItems);
		collectAnimatableValues();
	}

	private void handleTASelectorsAddRemove(ProjectEvent event) {
		if (event.composition != _composition) {
			return;
		}

		// 現在の選択状態を先に解除した方が見栄えがいい。解除しなくても機能的には問題ない。
		selectAndReveal((TreeItem) null);


		boolean add = (event.type == Type.TA_SELECTORS_ADD);

		Map<Layer, TreeItem> layerToTreeItem = Util.newMap();
		for (TreeItem treeItem : _treeViewer.getTree().getItems()) {
			LayerElement element = (LayerElement) treeItem.getData();
			layerToTreeItem.put(element.layer, treeItem);
		}

		Set<Object> refreshElements = Util.newSet();
		for (Object[] o : (Object[][]) event.data) {
			TextLayer layer = (TextLayer) o[0];
			TreeItem layerItem = layerToTreeItem.get(layer);
			int animatorIndex = (Integer) o[1];
			TextAnimator animator = layer.getTextAnimators().get(animatorIndex);
			
			TreeItem animatorItem = findTextAnimatorItem(layerItem, animator);
			refreshElements.add(animatorItem.getData());
		}
		for (Object o : refreshElements) {
			_treeViewer.refresh(o);
		}


		Set<TreeItem> treeItems = Util.newSet();

		for (Object[] o : (Object[][]) event.data) {
			TextLayer layer = (TextLayer) o[0];
			TreeItem layerItem = layerToTreeItem.get(layer);
			int animatorIndex = (Integer) o[1];
			TextAnimator animator = layer.getTextAnimators().get(animatorIndex);

			if (add) {
				TASelector selector = animator.getSelectors().get((Integer) o[2]);
				treeItems.add(findTASelectorItem(layerItem, selector));
			} else {
				treeItems.add(findTextAnimatorItem(layerItem, animator));
			}
		}

		selectAndReveal(treeItems);
		collectAnimatableValues();
	}

	private void handleTAPropertiesAddRemove(ProjectEvent event) {
		if (event.composition != _composition) {
			return;
		}

		// 現在の選択状態を先に解除した方が見栄えがいい。解除しなくても機能的には問題ない。
		selectAndReveal((TreeItem) null);


		boolean add = (event.type == Type.TA_PROPERTIES_ADD);

		Map<Layer, TreeItem> layerToTreeItem = Util.newMap();
		for (TreeItem treeItem : _treeViewer.getTree().getItems()) {
			LayerElement element = (LayerElement) treeItem.getData();
			layerToTreeItem.put(element.layer, treeItem);
		}

		Set<Object> refreshElements = Util.newSet();
		for (Object[] o : (Object[][]) event.data) {
			TextLayer layer = (TextLayer) o[0];
			TreeItem layerItem = layerToTreeItem.get(layer);
			int animatorIndex = (Integer) o[1];
			TextAnimator animator = layer.getTextAnimators().get(animatorIndex);
			
			TreeItem animatorItem = findTextAnimatorItem(layerItem, animator);
			refreshElements.add(animatorItem.getData());
		}
		for (Object o : refreshElements) {
			_treeViewer.refresh(o);
		}


		Set<TreeItem> treeItems = Util.newSet();

		for (Object[] o : (Object[][]) event.data) {
			TextLayer layer = (TextLayer) o[0];
			TreeItem layerItem = layerToTreeItem.get(layer);
			int animatorIndex = (Integer) o[1];
			TextAnimator animator = layer.getTextAnimators().get(animatorIndex);

			if (add) {
				String property = String.format("textAnimators[%d].%s", animatorIndex, o[2]);
				treeItems.add(findLayerAnimatableValueItem(layerItem, property));
			} else {
				treeItems.add(findTextAnimatorItem(layerItem, animator));
			}
		}

		selectAndReveal(treeItems);
		collectAnimatableValues();
	}

	private void handleEffectsAddRemove(ProjectEvent event) {
		if (event.composition != _composition) {
			return;
		}

		// 現在の選択状態を先に解除した方が見栄えがいい。解除しなくても機能的には問題ない。
		selectAndReveal((TreeItem) null);


		boolean add = (event.type == Type.EFFECTS_ADD);

		Map<Layer, TreeItem> layerToTreeItem = Util.newMap();
		for (TreeItem treeItem : _treeViewer.getTree().getItems()) {
			LayerElement element = (LayerElement) treeItem.getData();
			layerToTreeItem.put(element.layer, treeItem);
		}

		Set<TreeItem> treeItems = Util.newSet();

		for (Object[] o : (Object[][]) event.data) {
			TreeItem treeItem = layerToTreeItem.get(o[0]);
			LayerElement element = (LayerElement) treeItem.getData();
			Integer effectIndex = (Integer) o[1];

			_treeViewer.refresh(element);
			_treeViewer.setExpandedState(element, true);

			for (TreeItem child : treeItem.getItems()) {
				if (!(child.getData() instanceof EffectsElement)) {
					continue;
				}
				TreeItem itemToSelect;
				if (add) {
					// 追加後は追加したエフェクトを選択する。
					_treeViewer.setExpandedState(child.getData(), true);
					itemToSelect = child.getItem(effectIndex);
				} else {
					// 削除後は EffectsElement を選択する。
					itemToSelect = child;
				}
				treeItems.add(itemToSelect);
				break;
			}
		}

		selectAndReveal(treeItems);
		collectAnimatableValues();
	}

	private void handleEffectPropertyChange(ProjectEvent event) {
		if (event.composition != _composition) {
			return;
		}

		EffectableLayer layer = (EffectableLayer) event.layer;
		Effect effect = layer.getEffects().get(event.effectIndex);
		String property = event.property;

		for (TreeItem treeItem : _treeViewer.getTree().getItems()) {
			LayerElement layerElement = (LayerElement) treeItem.getData();
			if (layerElement.layer == layer) {
				if (ProjectEvent.propertyIsAnimatable(event)) {
					treeItem = findEffectAnimatableValueItem(treeItem, effect, property);

				} else if (property.equals("name") || property.equals("enabled")) {
					treeItem = findEffectItem(treeItem, effect);
					_treeViewer.update(treeItem.getData(), null);

				} else {
					//treeItem = findEffectPropertyItem(treeItem, effect, property);
				}

				selectAndReveal(treeItem);
				break;
			}
		}

		collectAnimatableValues();
	}

	private void handleEffectExpressionChange(ProjectEvent event) {
		if (event.composition != _composition) {
			return;
		}

		EffectableLayer layer = (EffectableLayer) event.layer;
		Effect effect = layer.getEffects().get(event.effectIndex);

		for (TreeItem treeItem : _treeViewer.getTree().getItems()) {
			LayerElement layerElement = (LayerElement) treeItem.getData();
			if (layerElement.layer == layer) {
				treeItem = findEffectAnimatableValueItem(treeItem, effect, event.property);
				Element element = (Element) treeItem.getData();

				// これをしなくても見た目はかわならいが、しないとundo/redo時にswtがクラッシュすることがある？
				_treeViewer.setExpandedState(element, false);

				_treeViewer.refresh(element, false);
				if (treeItem.getItemCount() > 0) {
					_treeViewer.setExpandedState(element, true);
					treeItem = treeItem.getItem(0);
				}
				selectAndReveal(treeItem);
				break;
			}
		}

		collectAnimatableValues();
	}

	private void handleLayerTimesChange(ProjectEvent event) {
		if (event.composition != _composition) {
			return;
		}

		Tree tree = _treeViewer.getTree();

		Map<Layer, TreeItem> layerToTreeItem = Util.newMap();
		for (TreeItem treeItem : tree.getItems()) {
			LayerElement element = (LayerElement) treeItem.getData();
			layerToTreeItem.put(element.layer, treeItem);
		}

		Set<TreeItem> treeItems = Util.newSet();

		@SuppressWarnings("unchecked")
		Collection<? extends Layer> layers = (Collection<? extends Layer>) event.data;
		for (Layer layer : layers) {
			TreeItem treeItem = layerToTreeItem.get(layer);
			treeItems.add(treeItem);
			//_treeViewer.update(treeItem.getData(), null);
		}

		_timelineManager.redraw();
		_splitLayersAction.setEnabled(canSplitSelectedLayers());
		selectAndReveal(treeItems);
		collectAnimatableValues();
	}

	private void handleKeyframesChange(ProjectEvent event) {
		if (event.composition != _composition) {
			return;
		}

		Map<Layer, TreeItem> layerToTreeItem = Util.newMap();
		for (TreeItem treeItem : _treeViewer.getTree().getItems()) {
			LayerElement element = (LayerElement) treeItem.getData();
			layerToTreeItem.put(element.layer, treeItem);
		}

		Set<TreeItem> treeItems = Util.newSet();
		_timelineManager.clearKeyframeSelection();

		Object[][] data = (Object[][]) event.data;

		for (Object[] o : data) {
			Layer layer = (Layer) o[0];
			int effectIndex = (Integer) o[1];
			String property = (String) o[2];

			TreeItem layerItem = layerToTreeItem.get(layer);
			TreeItem treeItem;
			AnimatableValue<?> avalue;

			if (effectIndex == -1) {
				Object obj = PropertyUtil.getProperty(layer, property);

				if (obj instanceof AnimatableValue<?>) {
					avalue = (AnimatableValue<?>) obj;
					treeItem = findLayerAnimatableValueItem(layerItem, property);

					if (property.equals("timeRemap")) {
						if (treeItem == null && avalue.hasKeyframe()) {
							_treeViewer.refresh(layerItem.getData(), false);
							treeItem = findLayerAnimatableValueItem(layerItem, property);
						} else if (treeItem != null && !avalue.hasKeyframe()) {
							_treeViewer.refresh(layerItem.getData(), false);
							treeItem = null;
							treeItems.add(layerItem);
						}
					}
				} else {
					if (layer instanceof LightLayer && property.equals("lightType")) {
						_treeViewer.refresh(layerItem.getData(), true);
						treeItem = findLayerPropertyItem(layerItem, "lightType");

					} else if (layer instanceof LightLayer && property.equals("castsShadows")) {
						treeItem = findLayerPropertyItem(layerItem, "castsShadows");

					} else if (layer instanceof MediaLayer
							&& (property.equals("castsShadows")
									|| property.equals("acceptsShadows")
									|| property.equals("acceptsLights"))) {

						treeItem = findLayerPropertyItem(layerItem, property);

					} else if (layer instanceof TextLayer && property.equals("textType")) {
						_treeViewer.refresh(layerItem.getData(), true);
						treeItem = findLayerPropertyItem(layerItem, "textType");

					} else {
						treeItem = null;
					}

					if (treeItem != null) {
						treeItems.add(treeItem);
					}
					continue;
				}

			} else {
				Effect effect = ((EffectableLayer) layer).getEffects().get(effectIndex);
				treeItem = findEffectAnimatableValueItem(layerItem, effect, property);
				avalue = PropertyUtil.getProperty(effect, property);
			}

			if (treeItem != null) {
				treeItems.add(treeItem);

				AnimatableValueElement element = (AnimatableValueElement) treeItem.getData();
				_treeViewer.update(element, null);

				if (o[3] instanceof Time) {
					Keyframe<?> kf = avalue.getKeyframe((Time) o[3]);
					_timelineManager.addKeyframeSelection(kf, element);
				} else if (o[3] instanceof List<?>) {
					@SuppressWarnings("unchecked")
					List<Time> times = (List<Time>) o[3];
					for (Time time : times) {
						Keyframe<?> kf = avalue.getKeyframe(time);
						_timelineManager.addKeyframeSelection(kf, element);
					}
				}
			}
		}

		selectAndReveal(treeItems);
		collectAnimatableValues();
	}

	private void handleLayerSlipEdit(ProjectEvent event) {
		if (event.composition != _composition) {
			return;
		}

		Tree tree = _treeViewer.getTree();

		Map<Layer, TreeItem> layerToTreeItem = Util.newMap();
		for (TreeItem treeItem : tree.getItems()) {
			LayerElement element = (LayerElement) treeItem.getData();
			layerToTreeItem.put(element.layer, treeItem);
		}

		Set<TreeItem> treeItems = Util.newSet();

		@SuppressWarnings("unchecked")
		Collection<? extends Layer> layers = (Collection<? extends Layer>) ((Object[]) event.data)[0];
		for (Layer layer : layers) {
			TreeItem treeItem = layerToTreeItem.get(layer);
			treeItems.add(treeItem);

			_treeViewer.update(treeItem.getData(), null);

			// タイムラインは独自に描画しているため、redraw もする必要がある？
			// また、win32 だと TreeItem#getBounds() が返す Rectangle ではダメ。

			Rectangle bounds = treeItem.getBounds();
			if (WIN32) {
				Rectangle clientArea = tree.getClientArea();
				bounds.x = clientArea.x;
				bounds.width = clientArea.width;
			}
			tree.redraw(bounds.x, bounds.y, bounds.width, bounds.height, false);
		}

	
		_timelineManager.clearKeyframeSelection();

		Object[][] data = (Object[][]) ((Object[]) event.data)[1];

		for (Object[] o : data) {
			Layer layer = (Layer) o[0];
			int effectIndex = (Integer) o[1];
			String property = (String) o[2];

			TreeItem treeItem;
			AnimatableValue<?> avalue;

			if (effectIndex == -1) {
				treeItem = findLayerAnimatableValueItem(layerToTreeItem.get(layer), property);
				avalue = PropertyUtil.getProperty(layer, property);
			} else {
				Effect effect = ((EffectableLayer) layer).getEffects().get(effectIndex);
				treeItem = findEffectAnimatableValueItem(layerToTreeItem.get(layer), effect, property);
				avalue = PropertyUtil.getProperty(effect, property);
			}

			treeItems.add(treeItem);

			AnimatableValueElement element = (AnimatableValueElement) treeItem.getData();
			_treeViewer.update(element, null);

			@SuppressWarnings("unchecked")
			List<Time> times = (List<Time>) o[3];
			for (Time time : times) {
				Keyframe<?> kf = avalue.getKeyframe(time);
				_timelineManager.addKeyframeSelection(kf, element);
			}
		}


		selectAndReveal(treeItems);
		collectAnimatableValues();
	}

	private void handleNewLightLayerAction() {
		if (_projectManager != null && _composition != null) {
			// TODO ライトの設定Wizardを作る
			_projectManager.postOperation(new NewLightLayerOperation(_projectManager, _composition));
		}
	}

	private void handleNewCameraLayerAction() {
		if (_projectManager != null && _composition != null) {
			// TODO カメラオプションの設定Wizardを作る
			_projectManager.postOperation(new NewCameraLayerOperation(_projectManager, _composition));
		}
	}

	private void handleNewNullLayerAction() {
		if (_projectManager != null && _composition != null) {
			_projectManager.postOperation(new NewNullLayerOperation(_projectManager, _composition));
		}
	}

	private void handleNewTextLayerAction() {
		if (_projectManager != null && _composition != null) {
			_projectManager.postOperation(new NewTextLayerOperation(_projectManager, _composition));
		}
	}

	public CompositionItem getCompositionItem() {
		return _compItem;
	}

	public LayerComposition getLayerComposition() {
		return _composition;
	}

	public void editExpression(AnimatableValueElement avalueElement) {
		_treeViewer.setExpandedState(avalueElement, true);

		AnimatableValueElementDelegate<?>.ExpressionElement exprElement
				= (AnimatableValueElementDelegate<?>.ExpressionElement) avalueElement.getChildren()[0];

		exprElement.openInPlaceEditor();
	}

	public void update(Time time, Boolean scrubbing) {
		if (_playerLinkListeners.isEmpty()) {
			_time = time;
			collectAnimatableValues(time);
			_timelineManager.update(time, true);
			_timeCodeComposite.redraw();
			_splitLayersAction.setEnabled(canSplitSelectedLayers());

		} else {
			collectAnimatableValues(time);
			refreshPlayer(time, scrubbing);
		}
	}


	private final ListenerList _playerLinkListeners = new ListenerList();

	public void addPlayerLinkListener(PlayerLinkListener listener) {
		_playerLinkListeners.add(listener);
	}

	public void removePlayerLinkListener(PlayerLinkListener listener) {
		_playerLinkListeners.remove(listener);
	}

	private void refreshPlayer(Time time, Boolean scrubbing) {
		PlayerLinkEvent event = new PlayerLinkEvent(this, time, scrubbing);
		for (Object l : _playerLinkListeners.getListeners()) {
			((PlayerLinkListener) l).handlePlayerLinkEvent(event);
		}
	}

	public void handlePlayerLinkEvent(PlayerLinkEvent event) {
		_time = event.time;
		_timelineUpdateTask.schedule();
		_collectTask.schedule();
	}


	private static final Set<String> _connectingIds = Util.newSet();

	private void connectToPlayer() {
		String itemId = getItemId();
		if (!_connectingIds.contains(itemId)) {
			_connectingIds.add(itemId);
			try {

				MediaPlayerView playerView = UIUtil.findView(getSite().getPage(), MediaPlayerView.ID, itemId);
				connectToPlayer(this, playerView);

			} finally {
				_connectingIds.remove(itemId);
			}
		}
	}

	static void connectToPlayer(MediaPlayerView playerView) {
		String itemId = playerView.getItemId();
		if (!_connectingIds.contains(itemId)) {
			_connectingIds.add(itemId);
			try {

				LayerCompositionView compView = UIUtil.findView(
						playerView.getSite().getPage(), LayerCompositionView.ID, itemId);
				connectToPlayer(compView, playerView);

			} finally {
				_connectingIds.remove(itemId);
			}
		}
	}

	private static void connectToPlayer(final LayerCompositionView compView, final MediaPlayerView playerView) {
		if (compView == null || playerView == null) {
			return;
		}

		final MediaPlayer player = playerView.getMediaPlayer();

		compView.addPlayerLinkListener(player);
		player.addPlayerLinkListener(compView);

		final IWorkbenchPage page = compView.getSite().getPage();
		IPartListener partListener = new IPartListener() {
			public void partClosed(IWorkbenchPart part) {
				if (part == playerView || part == compView) {
					compView.removePlayerLinkListener(player);
					player.removePlayerLinkListener(compView);
					page.removePartListener(this);
				}
			}

			public void partActivated(IWorkbenchPart part) {
				if (part == compView) {
					showView(playerView, compView);
				} else if (part == playerView) {
					showView(compView, playerView);
				}
			}

			private void showView(IViewPart targetView, IViewPart companionView) {
				for (IViewPart vp : page.getViewStack(companionView)) {
					if (vp == targetView) {
						return;
					}
				}
				IViewPart[] stack = page.getViewStack(targetView);
				if (stack == null) {
					return;
				}
				for (IViewPart vp : stack) {
					if (vp != targetView && page.isPartVisible(vp) &&
							vp instanceof MediaPlayerView && ((MediaPlayerView) vp).isPinned()) {
						return;
					}
				}
				try {
					IViewSite viewSite = targetView.getViewSite();
					page.showView(viewSite.getId(), viewSite.getSecondaryId(), IWorkbenchPage.VIEW_VISIBLE);
				} catch (PartInitException e) {
					_logger.error("error activating ViewPart", e);
				}
			}

			public void partOpened(IWorkbenchPart part) { }
			public void partDeactivated(IWorkbenchPart part) { }
			public void partBroughtToTop(IWorkbenchPart part) { }
		};
		page.addPartListener(partListener);
	}

}

class ShyFilter extends ViewerFilter {

	static final ShyFilter INSTANCE = new ShyFilter();


	private ShyFilter() {
		super();
	}

	@Override
	public boolean select(Viewer viewer, Object parentElement, Object element) {
		if (element instanceof LayerElement) {
			return !((LayerElement) element).layer.isShy();
		}
		return true;
	}

}
