/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.player;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.concurrent.atomic.AtomicReference;

import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.DataLine;
import javax.sound.sampled.LineUnavailableException;
import javax.sound.sampled.Mixer;
import javax.sound.sampled.SourceDataLine;

import org.eclipse.core.runtime.ListenerList;
import org.eclipse.swt.SWT;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaItem;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;

public class AudioThread extends Thread {

	private static final Logger _logger = LoggerFactory.getLogger(AudioThread.class);


	private static final Mixer.Info _jsaeMixerInfo;	// jsae = Java Sound Audio Engine

	static {
		Mixer.Info mixerInfo = null;
		String platform = SWT.getPlatform();
		if (platform.equals("win32") || platform.equals("wpf")) {
			for (Mixer.Info mi : AudioSystem.getMixerInfo()) {
				if (mi.getName().equals("Java Sound Audio Engine")) {
					mixerInfo = mi;
					break;
				}
			}
		}
		_jsaeMixerInfo = mixerInfo;
	}


	private final SourceDataLine _playLine;

	private final SourceDataLine _scrubLine;

	private final TimeKeeper _timeKeeper;

	private final MediaItem _mediaItem;

	private final AtomicReference<Time> _newTime = new AtomicReference<Time>();

	private final ListenerList _playerThreadListeners = new ListenerList();

	private final Object _monitor = new Object();

	private volatile boolean _playing;

	private volatile boolean _scrubbing;

	private volatile boolean _finished;

	private volatile boolean _muted;

	@Inject
	private RenderContext _context;

	@Inject
	private IArrayPools _arrayPools;


	public AudioThread(MediaItem mediaItem) throws LineUnavailableException {
		super("AudioThread: " + mediaItem.getName());
		setPriority(getThreadGroup().getMaxPriority());

		MediaInput input = mediaItem.getMediaInput();
		if (input == null) {
			throw new IllegalArgumentException("no MediaInput is available");
		}
		if (!input.isAudioAvailable()) {
			throw new IllegalArgumentException("no audio is available");
		}

		AudioFormat lineFormat = new AudioFormat(48000, 16, 2, true, ByteOrder.nativeOrder().equals(ByteOrder.BIG_ENDIAN));
		DataLine.Info lineInfo = new DataLine.Info(SourceDataLine.class, lineFormat);
		_playLine = (SourceDataLine) AudioSystem.getLine(lineInfo);
		_playLine.open(lineFormat, lineFormat.getFrameSize() * (int) (lineFormat.getFrameRate() / 10));	// 0.1秒分のバッファサイズ
//		_playLine.open(lineFormat, lineFormat.getFrameSize() * (int) (lineFormat.getFrameRate() / 4));	// 0.25秒分のバッファサイズ

		// Windowsの場合、デフォルトのミキサーでスクラブ再生すると、スクラブ終了後も音の断片がなり続けることがある。
		// そのため、Java Sound Audio Engine のミキサーで代用する。
		if (_jsaeMixerInfo != null) {
			_scrubLine = AudioSystem.getSourceDataLine(lineFormat, _jsaeMixerInfo);
		} else {
			_scrubLine = (SourceDataLine) AudioSystem.getLine(lineInfo);
		}
		_scrubLine.open(lineFormat, lineFormat.getFrameSize() * (int) (lineFormat.getFrameRate() / 10));	// 0.1秒分のバッファサイズ
//		_scrubLine.open(lineFormat, lineFormat.getFrameSize() * (int) (lineFormat.getFrameRate() / 4));		// 0.25秒分のバッファサイズ

		_timeKeeper = TimeKeeper.fromAudioLine(_playLine);

		InjectorHolder.getInjector().injectMembers(this);
		_mediaItem = mediaItem;
	}

	public TimeKeeper getTimeKeeper() {
		return _timeKeeper;
	}

	public boolean isMuted() {
		return _muted;
	}

	public void setMuted(boolean muted) {
		_muted = muted;
	}

	public void run() {
		try {
			MediaInput input = _mediaItem.getMediaInput();
			Time duration = input.getDuration();

			// TODO プロジェクト設定などから取得
			AudioMode audioMode = AudioMode.STEREO_48KHZ_INT16;

			int chunkFrames = audioMode.sampleRate / 10;	// 1回あたり0.1秒分のフレームを処理する
//			int chunkFrames = audioMode.sampleRate / 4;		// 1回あたり0.25秒分のフレームを処理する

			Time chunkDuration = Time.fromFrameNumber(chunkFrames, audioMode.sampleDuration); 
			Time halfChunkDur = Time.fromFrameNumber(chunkFrames/2, audioMode.sampleDuration); 
			int chunkBytes = audioMode.frameSize*chunkFrames;
			int one3rdChunkBytes = chunkBytes/3;

			_playLine.start();
			_scrubLine.start();
			_context.activate();

			SourceDataLine line;
			Time nextTime = Time.fromFrameNumber(0, audioMode.sampleDuration);

			while (waitForPlaying()) {

				Time time = _newTime.getAndSet(null);

				if (_scrubbing) {
					synchronized (_monitor) {
						_playing = false;
						_scrubbing = false;
					}

					if (time == null) {
						continue;
					}

					line = _scrubLine;

					if (nextTime.timeValue != 0 && line.available() < chunkBytes) {
						Time t1 = nextTime.subtract(halfChunkDur);
						Time t2 = t1.add(chunkDuration);

						if (!time.before(t1) && time.before(t2)) {
							time = nextTime;

						} else if (!time.before(t1 = t1.subtract(chunkDuration))
								&& time.before(t2 = t2.subtract(chunkDuration))) {

							if (line.available() < one3rdChunkBytes) {
								continue;
							}

						} else if (!time.before(t1.subtract(chunkDuration))
								&& time.before(t2.subtract(chunkDuration))) {

							time = nextTime.subtract(chunkDuration).subtract(chunkDuration);
							if (time.timeValue < 0) {
								time = new Time(0, time.timeScale);
							}
						}
					}

					if (!time.before(duration)) {
						continue;
					}

				} else {
					line = _playLine;

					if (time != null) {
						line.flush();
						_timeKeeper.setTime(time);
					} else {
						time = nextTime;
					}

					if (!time.before(duration)) {
						line.drain();
						synchronized (_monitor) {
							_playing = false;
						}
						fireThreadEndOfDuration(time);
						continue;
					}
				}

				IAudioBuffer ab = null;

				if (!_muted) {
					_context.reset();
					_context.setAudioMode(audioMode);
					_context.setAudioAnimationRate(audioMode.sampleRate/100);
					_context.setAudioFrameCount(chunkFrames);
					_context.setVideoResolution(Resolution.FULL);

					if (_mediaItem instanceof CompositionItem) {
						Composition comp = ((CompositionItem) _mediaItem).getComposition();
						_context.setVideoFrameDuration(comp.getFrameDuration());

						PlayerLock.readLock().lock();
						try {
							// TODO prepareExpressionはプロジェクトに構造的な変更があった場合のみ行えばよい。
							comp.prepareExpression(_context.createInitialExpressionScope(comp));

							ab = input.getAudioChunk(time);
						} catch (Exception e) {
							_logger.error("error getting audio chunk", e);
						} finally {
							PlayerLock.readLock().unlock();
						}
					} else {
						try {
							ab = input.getAudioChunk(time);
						} catch (Exception e) {
							_logger.error("error getting audio chunk", e);
						}
					}
				}

				if (ab != null) {
					write(ab, line);
					ab.dispose();
				} else {
					writeZero(chunkFrames, line);
				}

				// 現在のところ、要求したフレーム数分だけ必ず返す事になっているので、
				// chunkFramesの値の分だけ時間を進めればよいが、一応 IAudioBuffer のフレーム数を見て時間を進める。
				nextTime = time.add(Time.fromFrameNumber(
						(ab != null) ? ab.getFrameCount() : chunkFrames, audioMode.sampleDuration));

				fireThreadRender(time);
			}
		} finally {
			_context.deactivate();
			_playLine.close();
			_scrubLine.close();
		}
	}

	public void setTime(Time time) {
		_newTime.set(time);
	}

	public void play(boolean play) {
		if (_playing != play || _scrubbing) {
			synchronized (_monitor) {
				_playing = play;
				_scrubbing = false;
				_monitor.notify();
			}
		}
	}

	public void scrub(Time time) {
		_newTime.set(time);

		if (!_playing) {
			synchronized (_monitor) {
				_playing = true;
				_scrubbing = true;
				_monitor.notify();
			}
		}
	}

	public void finish() {
		if (_finished) {
			return;
		}

		synchronized (_monitor) {
			_finished = true;
			_monitor.notify();
		}

		try {
			join();
		} catch (InterruptedException e) {
			// ignore
		}
	}

	private boolean waitForPlaying() {
		if (!_playing) {
			_playLine.flush();

			synchronized (_monitor) {
				while (!_finished && !_playing) {
					try {
						_monitor.wait();
					} catch (InterruptedException e) {
						// ignore
					}
				}
			}
		}
		return !_finished;
	}

	private void write(IAudioBuffer ab, SourceDataLine line) {
		// Java Sound Audio Engine のミキサーは、デフォルトのミキサーと比べて出力レベルが半分程度のようなので、その補正をする。
		double gain = (line == _scrubLine && _jsaeMixerInfo != null) ? 2 : 1;

		AudioMode audioMode = ab.getAudioMode();
		int dataLen = ab.getDataLength();
		int lineDataLen = line.getFormat().getFrameSize() * ab.getFrameCount();
		IArray<byte[]> pa = _arrayPools.getByteArray(lineDataLen);
		byte[] lineBuf = pa.getArray();

		switch (audioMode.dataType) {
			case SHORT:
				if (gain == 1) {
					ByteBuffer.wrap(lineBuf, 0, lineDataLen).order(ByteOrder.nativeOrder())
							.asShortBuffer().put((short[]) ab.getData(), 0, dataLen);
				} else {
					short[] data = (short[]) ab.getData();
					for (int i = 0; i < dataLen; ++i) {
						short shortVal = (short) (Math.min(Math.max(data[i]*gain/Short.MAX_VALUE, -1.0), 1.0) * Short.MAX_VALUE);
						lineBuf[i*2  ] = (byte) ((shortVal      ) & 0xff);
						lineBuf[i*2+1] = (byte) ((shortVal >>> 8) & 0xff);
					}
				}
				break;
			case INT: {
				int[] data = (int[]) ab.getData();
				for (int i = 0; i < dataLen; ++i) {
					short shortVal = (short) (Math.min(Math.max(data[i]*gain/Integer.MAX_VALUE, -1.0), 1.0) * Short.MAX_VALUE);
					lineBuf[i*2  ] = (byte) ((shortVal      ) & 0xff);
					lineBuf[i*2+1] = (byte) ((shortVal >>> 8) & 0xff);
				}
				break;
			}
			case FLOAT: {
				float[] data = (float[]) ab.getData();
				for (int i = 0; i < dataLen; ++i) {
					short shortVal = (short) (Math.min(Math.max(data[i]*gain, -1.0), 1.0) * Short.MAX_VALUE);
					lineBuf[i*2  ] = (byte) ((shortVal      ) & 0xff);
					lineBuf[i*2+1] = (byte) ((shortVal >>> 8) & 0xff);
				}
				break;
			}
			default:
				throw new UnsupportedOperationException(
						"unsupported AudioMode.DataType: " + audioMode.dataType);
		}

		if (audioMode.sampleRate == 96000) {
			for (int i = 0, n = dataLen/2; i < n; ++i) {
				short s1 = (short) ((lineBuf[i*2*2  ] & 0xff) | ((lineBuf[i*2*2+1] & 0xff) << 8));
				short s2 = (short) ((lineBuf[i*2*2+2] & 0xff) | ((lineBuf[i*2*2+3] & 0xff) << 8));
				short s = (short) ((s1 + s2) / 2);
				lineBuf[i*2  ] = (byte) ((s      ) & 0xff);
				lineBuf[i*2+1] = (byte) ((s >>> 8) & 0xff);
			}
			lineDataLen /= 2;
		}

		line.write(lineBuf, 0, lineDataLen);
		pa.release();
	}

	private void writeZero(int frameCount, SourceDataLine line) {
		IArray<byte[]> pa = _arrayPools.getByteArray(line.getFormat().getFrameSize() * frameCount);
		pa.clear();
		line.write(pa.getArray(), 0, pa.getLength());
		pa.release();
	}

	public void addPlayerThreadListener(PlayerThreadListener listener) {
		_playerThreadListeners.add(listener);
	}

	public void removePlayerThreadListener(PlayerThreadListener listener) {
		_playerThreadListeners.remove(listener);
	}

	private void fireThreadRender(Time time) {
		PlayerThreadEvent event = new PlayerThreadEvent(this, time);
		for (Object l : _playerThreadListeners.getListeners()) {
			((PlayerThreadListener) l).threadRender(event);
		}
	}

	private void fireThreadEndOfDuration(Time time) {
		PlayerThreadEvent event = new PlayerThreadEvent(this, time);
		for (Object l : _playerThreadListeners.getListeners()) {
			((PlayerThreadListener) l).threadEndOfDuration(event);
		}
	}

}
