/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.expression;

import org.mozilla.javascript.Context;
import org.mozilla.javascript.Function;
import org.mozilla.javascript.Scriptable;

public abstract class Interpolation {

	public static final String[] METHOD_NAMES = { "linear", "ease", "easeIn", "easeOut" };


	public static Object linear(Context cx, Scriptable thisObj, Object[] args, Function funOb) {
		return LINEAR.interpolate(cx, thisObj, args, funOb);
	}

	public static Object ease(Context cx, Scriptable thisObj, Object[] args, Function funOb) {
		return EASE.interpolate(cx, thisObj, args, funOb);
	}

	public static Object easeIn(Context cx, Scriptable thisObj, Object[] args, Function funOb) {
		return EASE_IN.interpolate(cx, thisObj, args, funOb);
	}

	public static Object easeOut(Context cx, Scriptable thisObj, Object[] args, Function funOb) {
		return EASE_OUT.interpolate(cx, thisObj, args, funOb);
	}


	private static final Interpolation LINEAR = new Interpolation() {
		@Override
		public double interpolate(double t) {
			return t;
		}
	};

	private static final Interpolation EASE = new Interpolation() {
		@Override
		public double interpolate(double t) {
			return t*t*(3-2*t);
		}
	};

	private static final Interpolation EASE_IN = new Interpolation() {
		@Override
		public double interpolate(double t) {
			t *= 0.75;
			return t*t*(3-2*t) / 0.84375;
		}
	};

	private static final Interpolation EASE_OUT = new Interpolation() {
		@Override
		public double interpolate(double t) {
			t = 0.75*(t+1.0/3);
			return (t*t*(3-2*t)-0.15625) / (1-0.15625);
		}
	};


	private Interpolation() { }

	protected abstract double interpolate(double t);

	private Object interpolate(Context cx, Scriptable thisObj, Object[] args, Function funOb) {
		double t;
		Object o1, o2;
		if (args.length == 5) {
			t = Context.toNumber(args[0]);
			double tMin = Context.toNumber(args[1]);
			double tMax = Context.toNumber(args[2]);
			o1 = ExpressionUtils.toDoubleOrDoubleArray(args[3]);
			o2 = ExpressionUtils.toDoubleOrDoubleArray(args[4]);

			t = (t - tMin) / (tMax - tMin);

		} else if (args.length == 3) {
			t = Context.toNumber(args[0]);
			o1 = ExpressionUtils.toDoubleOrDoubleArray(args[1]);
			o2 = ExpressionUtils.toDoubleOrDoubleArray(args[2]);

		} else {
			throw new IllegalArgumentException();
		}

		t = Math.min(Math.max(0, t), 1);
		t = interpolate(t);

		if (o1 instanceof Double && o2 instanceof Double) {
			double d1 = (Double) o1;
			double d2 = (Double) o2;
			return d1 + (d2 - d1) * t;
		}

		double[] a1 = (o1 instanceof Double) ? new double[] { (Double)o1 } : (double[])o1;
		double[] a2 = (o2 instanceof Double) ? new double[] { (Double)o2 } : (double[])o2;
		double[] a = new double[Math.max(a1.length, a2.length)];

		for (int i = 0; i < a.length; ++i) {
			double d1 = (i < a1.length) ? a1[i] : 0;
			double d2 = (i < a2.length) ? a2[i] : 1;
			a[i] = d1 + (d2 - d1) * t;
		}

		return a;
	}

}
